! gtool_history.f90 - traditional interface for `history' output
! vi: set sw=4 ts=8: 
! Copyright (C) TOYODA Eizi, 2000.  All rights reserved.

module gtool_history

    use gtdata_types
    implicit none

    private
    public:: GT_HISTORY
    public:: HistoryCreate, HistoryAddVariable, HistoryClose
    public:: HistoryPutEx, HistoryPut

    interface HistoryPut
	module procedure HistoryPut1, HistoryPut2, HistoryPut3
    end interface

    type GT_HISTORY
	type(GT_VARIABLE), pointer:: dimvars(:)
	logical, pointer:: dim_value_written(:)
	integer:: unlimited_index
	type(GT_VARIABLE), pointer:: vars(:)
	real:: origin, interval
	integer, pointer:: count(:)
    end type

    type(GT_HISTORY), save, target:: default

contains

    subroutine HistoryCreate(file, title, source, institution, &
	& dims, dimsizes, longnames, units, origin, interval, history)
        use gtdata_generic
        use gtool_url
        implicit none
    	character(len=*), intent(in):: file
	character(len=*), intent(in):: title, source, institution
	character(len=*), intent(in):: dims(:)
	integer, intent(in):: dimsizes(:)
	character(len=*), intent(in):: longnames(:)
	character(len=*), intent(in):: units(:)
	real, intent(in):: origin, interval
	type(GT_HISTORY), intent(out), optional, target:: history
	integer:: numdims, i
	type(GT_HISTORY), pointer:: hst
        logical:: first
    continue
	if (present(history)) then
	    hst => history
	else
	    hst => default
	endif
	numdims = size(dims)
	if (size(dimsizes) /= numdims .or. size(longnames) /= numdims .or. &
	    & size(units) /= numdims) then
	    stop 'HistoryCreate: bad argument size'
	endif

	! ϐ\쐬
	allocate(hst%dimvars(numdims), hst%dim_value_written(numdims))
	hst%dim_value_written(:) = .FALSE.
	hst%unlimited_index = 0

        first = .TRUE.
	do, i = 1, numdims
	    call Create(hst%dimvars(i), UrlMerge(file, trim(dims(i))), &
                 dimsizes(i))
            if (first) then
                call put_attr(hst%dimvars(i), 'title', title)
                call put_attr(hst%dimvars(i), 'source', source)
                call put_attr(hst%dimvars(i), 'institution', institution)
                first = .FALSE.
            endif
	    call PutAttribute(hst%dimvars(i), 'long_name', trim(longnames(i)))
	    call PutAttribute(hst%dimvars(i), 'units', trim(units(i)))
	    if (dimsizes(i) == 0) hst%unlimited_index = i
	enddo

	! ϐ\
	nullify(hst%vars)

	! ԃJE^
	hst%origin = origin
	hst%interval = interval
	hst%count = 0
    end subroutine

    subroutine HistoryAddVariable(varname, dims, longname, units, history)
        use iso_varying_string
        use gtdata_generic
        implicit none
	character(len = *), intent(in):: varname
	character(len = *), intent(in):: dims(:)
	character(len = *), intent(in):: longname, units
	type(GT_HISTORY), intent(inout), optional, target:: history
	type(GT_HISTORY), pointer:: hst
	type(GT_VARIABLE), pointer:: vwork(:), dimvars(:)
	type(VARYING_STRING):: fullname, time_name
	integer, pointer:: count_work(:)
	integer:: nvars, numdims, i
	logical:: ok
    continue
	! Ώی
	if (present(history)) then
	    hst => history
	else
	    hst => default
	endif

	! ϐ\g
	if (associated(hst%vars)) then
	    nvars = size(hst%vars(:))
	    vwork => hst%vars
	    count_work => hst%count
	    nullify(hst%vars, hst%count)
	    allocate(hst%vars(nvars + 1), hst%count(nvars + 1))
	    hst%vars(1:nvars) = vwork(1:nvars)
	    hst%count(1:nvars) = count_work(1:nvars)
	    deallocate(vwork, count_work)
	else
	    allocate(hst%vars(1), hst%count(1))
	endif
	nvars = size(hst%vars(:))

	! ϐY
	numdims = size(dims(:))
	allocate(dimvars(numdims))
	do, i = 1, numdims
	    dimvars(i) = lookup_dimension(hst, dims(i))
	enddo

	! ϐ쐬
	fullname = Name(hst%dimvars(1)%gtfile) // "#" // varname
	call Create(hst%vars(nvars), fullname, dimvars)
	if (hst%unlimited_index /= 0) then
	    time_name = Name(hst%dimvars(hst%unlimited_index))
	    ok = Slice(hst%vars(nvars), char(time_name), start=1, count=1, stride=1)
	    if (.not. ok) call GtoolAbort()
	endif
	call PutAttribute(hst%vars(nvars), 'long_name', longname)
	call PutAttribute(hst%vars(nvars), 'units', units)
	hst%count(nvars) = 0
	deallocate(dimvars)
    end subroutine

    subroutine HistoryPutEx(varname, array, arraysize, history)
        use iso_varying_string
        use gtdata_generic
	character(len = *), intent(in):: varname
	integer, intent(in):: arraysize
	real, intent(in):: array(arraysize)
	type(GT_HISTORY), intent(inout), optional, target:: history
	type(GT_HISTORY), pointer:: hst
	type(GT_VARIABLE):: var
	logical:: ok
	integer:: v_ord, d_ord
	integer, pointer:: time
	type(VARYING_STRING):: time_name
    continue
	if (present(history)) then
	    hst => history
	else
	    hst => default
	endif
	var = lookup_variable(hst, varname, ord=v_ord)
	if (v_ord == 0) then
  	    var = lookup_dimension(hst, varname, ord=d_ord)
	    hst%dim_value_written(d_ord) = .TRUE.
	else if (hst%unlimited_index /= 0) then
  	    time => hst%count(v_ord)
	    time = time + 1
      	    time_name = Name(hst%dimvars(hst%unlimited_index))
	    ok = Slice(var, char(time_name), start=time)
	    if (.not. ok) call GtoolAbort()
	endif
	ok = GtoolPutReal(var, array, arraysize)
	if (.not. ok) call GtoolAbort()
    end subroutine

    subroutine HistoryPut1(varname, array, history)
	character(len = *), intent(in):: varname
	real, intent(in):: array(:)
	type(GT_HISTORY), intent(inout), optional, target:: history
    continue
	call HistoryPutEx(varname, array, size(array), history)
    end subroutine

    subroutine HistoryPut2(varname, array, history)
	character(len = *), intent(in):: varname
	real, intent(in):: array(:, :)
	type(GT_HISTORY), intent(inout), optional, target:: history
    continue
	call HistoryPutEx(varname, array, size(array), history)
    end subroutine

    subroutine HistoryPut3(varname, array, history)
	character(len = *), intent(in):: varname
	real, intent(in):: array(:, :, :)
	type(GT_HISTORY), intent(inout), optional, target:: history
    continue
	call HistoryPutEx(varname, array, size(array), history)
    end subroutine

    subroutine HistoryClose(history)
        use gtdata_generic
	type(GT_HISTORY), intent(inout), optional, target:: history
	type(GT_HISTORY), pointer:: hst
	logical:: fail
	integer:: i
    continue
	if (present(history)) then
	    hst => history
	else
	    hst => default
	endif
	do, i = 1, size(hst%dimvars)
	    if (.not. hst%dim_value_written(i)) &
		call set_fake_dim_value(hst, i)
	    call Close(hst%dimvars(i), fail=fail)
	    if (fail) call put(itos(i) // ":" // Message(NetcdfLastError()))
	enddo
	deallocate(hst%dimvars)
	do, i = 1, size(hst%vars)
	    call Close(hst%vars(i), fail=fail)
	    if (fail) call put(itos(i) // ":" // Message(NetcdfLastError()))
	enddo
	deallocate(hst%vars, hst%count)
    end subroutine

    subroutine set_fake_dim_value(history, dimord)
	type(GT_HISTORY), intent(inout):: history
	integer, intent(in):: dimord
	integer:: length, i
	real, allocatable:: value(:)
    continue
	if (dimord == history%unlimited_index) then
	    length = maxval(history%count(:))
	else
	    length = Len(history%dimvars(dimord)%ncdim)
	endif
	allocate(value(length))
        value(:) = (/(real(i), i = 1, length)/)
	if (dimord == history%unlimited_index) then
	    value(:) = history%origin + (value(:) - 1.0) * history%interval
	    if (.not. Slice(history%dimvars(dimord), 1, start=1, count=length)) &
		call GtoolAbort()
	endif

  	call Put(history%dimvars(dimord), value, size(value))
	deallocate(value)
    end subroutine

    type(GT_VARIABLE) &
    function lookup_variable(history, varname, ord) result(result)
        use iso_varying_string
	type(GT_HISTORY), intent(in):: history
	character(len = *):: varname
	integer, intent(out), optional:: ord
	integer:: i
    continue
	if (associated(history%vars)) then
	    do, i = 1, size(history%vars)
		if (Name(history%vars(i)) == varname) then
		    result = history%vars(i)
		    if (present(ord)) ord = i
		    return
		endif
	    enddo
	endif
	if (present(ord)) then
	    ord = 0
	else
	    print *, 'gtool_history: var lookup for ', varname, ' failed'
	    stop
	endif
    end function

    type(GT_VARIABLE) &
    function lookup_dimension(history, dimname, ord) result(result)
        use gtdata_generic
        use iso_varying_string
	type(GT_HISTORY), intent(in):: history
	character(len = *):: dimname
	integer, intent(out), optional:: ord
	integer:: i
    continue
	if (associated(history%dimvars)) then
	    do, i = 1, size(history%dimvars)
		if (Name(history%dimvars(i)) == dimname) then
		    result = history%dimvars(i)
		    if (present(ord)) ord = i
		    return
		endif
	    enddo
	endif
	if (present(ord)) then
	    ord = 0
	else
	    print *, 'gtool_history: dim lookup for ', dimname, ' failed'
	    stop
	endif
    end function

end module
