// -*- C++ -*-

// (c) COPYRIGHT OPeNDAP 2004
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      James Gallagher <jgallagher@opendap.org>

// Base class for methods common to all the netCDF DAP dat type classes.
// jhrg 08/31/04

#include "config_nc.h"

static char rcsid[] not_used ={"$Id: NCAccess.cc,v 1.14 2005/06/03 16:33:21 jimg Exp $"};

#include "BaseType.h"
#include "InternalErr.h"

#include "NCAccess.h"
#include "NCSequence.h"
#include "NCStr.h"
#include "nc_util.h"

#include "debug.h"

const string spr = "."; // structure rename

/** Clone this object. This method is used by the NC&lt;type&gt; classes to
    copy fields that are defined by this object. Code should never create
    an object that is only an NCAccess; the class should always be used
    in combination with one of the NetCDF type classes. */
void
NCAccess::clone(const NCAccess &nca)
{
    d_impl_version = nca.d_impl_version;
    d_translated = nca.d_translated;

    if (nca.d_source)
        d_source = nca.d_source->ptr_duplicate();
    else
        d_source = 0;
}

/** This is the default version of build_constraint(). Unless specialized,
    each type's CE is just the variable's name. Note that Array and Grid
    must be specialized to use the \e start, \e edges and \e stride info.
    
    @param outtype
    @param start
    @param edges
    @param stride
    @return The number of elements to be returned */
    
string
NCAccess::build_constraint(int outtype, const size_t *start,
            const size_t *edges, const ptrdiff_t *stride) throw(Error)
{
    if (!is_convertable(outtype))
        throw Error(NC_ECHAR, "Character conversion not supported.");
  
    switch(dynamic_cast<BaseType*>(this)->type()) {
      case dods_str_c:
      case dods_url_c:  
      case dods_byte_c:
      case dods_int16_c:
      case dods_uint16_c:
      case dods_int32_c:
      case dods_uint32_c: 
      case dods_float32_c:
      case dods_float64_c:
        return dynamic_cast<BaseType*>(this)->name();
        
      default:
        throw Error(NC_EBADTYPE,
            string("The netCDF Client Library cannot request variables of type: ")
            + dynamic_cast<BaseType*>(this)->type_name()
            + " [from NCAccess::build_constraint()]");
    }
}

/** Scan the projection clauses passed to nc_open() and extract any info 
    relevant to this variable. This method should be specialized for any
    class that will appear to be an Array after translation. Store the start,
    stop and stride information of the projection. There's no need to store
    the fact that a variable was projected if there was no hyper-slab operation
    since the netCDF CL can only ask for one variable at a time, it _must_ be
    the only thing in the current projection. But, a user may pass a CE with
    projections to nc_open() and will expect those hyper-slabbing operations
    to be reflected in any data that is acccessed. 
    
    @note This implementation does nothing; for most of the NCAccess methods,
    the default version throws InternalErr. This version does _not_ do that 
    so that it can be called without checking that the type of the variable
    can store the start, stop and stride information.
    
    @param proj A string that contains the projection clauses. */
void
NCAccess::store_projection(const string &proj)
{
}

/** Can the variable be converted to outtype? A caller should be sure to
    return NC_ECHAR if this methods returns false.
    
    @param outtype The output type.
    @return True if the variable can be converted, false otherwise. */
bool
NCAccess::is_convertable(int outtype)
{
    Type intype = dynamic_cast<BaseType*>(this)->type();
    DBG(cerr << "intype: " << intype << ", outtype: " << outtype << endl);

    if (((outtype == Ttext) && (intype != dods_str_c) &&(intype != dods_url_c))
        || ((outtype != Ttext) && (intype == dods_str_c) &&(outtype != Tvoid))
        || ((outtype != Ttext)&&(intype == dods_url_c)&&(outtype != Tvoid)))
        return false;
    else
        return true;
}

/** Return the netCDF type of this variable. 
    @note This \b must be specialized for all of the atomic types. This
    version throws an InternalErr exception. */
nc_type
NCAccess::get_nc_type() throw(InternalErr)
{
    throw InternalErr(__FILE__, __LINE__,
        string("NetCDF type information requested for a ")
        + dynamic_cast<BaseType*>(this)->type_name() 
        + string("; operation not supported."));
}

/** Given that the variable has values, extract those, convert to outtype
    and return them in \e values. 
    
    @note This version \b must be specialized.
    
    @param values Buffer for the extracted values.
    @param outtype The desired type for the values.
    @exception Error Thrown if no data are present, if there's an error 
    copying/converting values/types or if the type of the variable is not
    supported (the latter should never happen). 
    @see NCArray::extract_values() */
void
NCAccess::extract_values(void *values, int elements, int outtype) throw(Error)
{    
    BaseType *btp = dynamic_cast<BaseType*>(this);
    switch (btp->type()) {
      case dods_byte_c:
      case dods_int16_c:
      case dods_uint16_c:
      case dods_int32_c:
      case dods_uint32_c: 
      case dods_float32_c:
      case dods_float64_c: {
        // If this (apparently) scalar variable is part of a Sequence, then
        // The netCDF library must think it's an Array. Get the number of 
        // elements (which is the number of rows in the returned Sequence).
        // By calling find_ancestral_sequence() the case where this variable
        // is a member of a structure that's inside a sequence is covered.
        BaseType &bt = dynamic_cast<BaseType&>(*this);   
        NCSequence *ncq = dynamic_cast<NCSequence*>(find_ancestral_sequence(&bt));
        int nels = (ncq) ? ncq->number_of_rows() : 1;
    
        // Allocate storage for the values
        int data_width = bt.width();
        // Why allocate 'elements' instead of 'nels'? nels is the number of 
        // elements actually returned; elements is the number the client 
        // thinks it's getting. We allocate enough to padd with zeros. Then
        // convert_nc_type is called to transfer the information to the values 
        // buffer.
#if 0
        void *tmpbufin = new char[nels * data_width];
#endif
        char *tmpbufin = new char[elements * data_width];
        int bytes = 0;
    
        if (ncq) {
            char *tptr = tmpbufin;
            int i = 0;
            while (i < nels) {
                // Using an index in place of bt.name() would be faster
                bytes += ncq->var_value(i, bt.name())->buf2val((void**)&tptr);
                tptr += data_width;
                ++i;
            }
            // Pad tptr so any values not read are zeros
            while (i < elements) {
                for (int j = 0; j < data_width; ++j)
                    *tptr++ = 0;
                ++i;                
            }
        }
        else {
            bytes = bt.buf2val((void**)&tmpbufin);
        }
        
        if (bytes == 0)
            throw Error(-1, "Could not read any data from remote server.");
    
        // Get the netCDF type code for this variable.
        nc_type typep = dynamic_cast<NCAccess*>(this)->get_nc_type();
    
        int rcode = convert_nc_type(typep, outtype, elements, tmpbufin, values);
        delete[] tmpbufin; tmpbufin = 0;
        if (rcode != NC_NOERR)
            throw Error(rcode,
                "Error copying values between internal buffers [NCAccess::extract_values()]");
#if 0
        if (tmpbufin != values)
            delete tmpbufin;
#endif
        break;
      }
      
      case dods_str_c:
      case dods_url_c:
        throw InternalErr(__FILE__, __LINE__, "Should never get here!!");
   
      default:
        throw Error(NC_EBADTYPE,
            string("The netCDF Client Library cannot access variables of type: ")
            + dynamic_cast<BaseType*>(this)->type_name()
            + " [NCAccess::extract_values()]");
    } // End of the switch
}

/** If this variable was created during a translation operation, return the
    original variable (the source used to produce this 'translated' 
    variable). If this variable was not created during translation, return
    null.
    
    @return The source variable if this was translated, or null if not
    translated. */
BaseType *
NCAccess::get_source() const
{
    return d_source;
}

/** Set the source-variable pointer.
    @param s A pointer to this variable's source variable.
    @exception InternalErr Thrown by default; overload this in NCArray, 
    et cetera. */
void
NCAccess::set_source(BaseType *s) throw(InternalErr)
{
    throw InternalErr(__FILE__, __LINE__, "Unimplemented method");
}

/** Flatten this variable. For atomic variables, this does nothing except 
    to return the variable as an element of the returned list. For Structures
    and Sequences, this returns a list of the variables held in that 
    constructor. For both Sequence and Structure, the variables are renamed
    &lt;ctor-name&gt;.&lt;var-name&gt;. For a Sequence, each field is turned
    into an array whose size if given by the \t limit client-side URL 
    parameter. Since it's possible to have Arrays of Structures and Sequences,
    Array must also provide a specialization. 
    
    @note Grid variables are handled specially; don't flatten them.
    
    @note If parent_name is not "", then this method will assume that the
    variable is being translated. If it is "" then this will simply copy
    this variable.
    
    @param cp A const reference to the ClientParams object.
    @return A VarList that holds all of the 'flattened' variables. */
VarList
NCAccess::flatten(const ClientParams &cp, const string &parent_name)
{
    VarList new_vars;

    // Note that since BaseType includes an attribute table (AttrTable)
    // instance and ptr_duplicate() arranges to copy it, using ptr_duplicate()
    // also copies the attributes of the variable referenced by *ctor to
    // the new variable. jhrg 10/26/04
    BaseType *source = dynamic_cast<BaseType*>(this);
    if (!source)
        throw InternalErr(__FILE__, __LINE__, "Instance of NCAccess not a BaseType!");
    BaseType *btp = source->ptr_duplicate();
#if 0
    BaseType *btp = dynamic_cast<BaseType&>(*this).ptr_duplicate();
#endif    
    if (!parent_name.empty()) {
        btp->set_name(parent_name + spr + dynamic_cast<BaseType&>(*this).name());
        btp->get_attr_table().append_attr("translation", "String",
                                          "\"flatten\"");
        dynamic_cast<NCAccess&>(*btp).set_translated(true);
        DBG2(cerr << "Marking " << btp->name() << " (" << btp 
                 << ") as translated." << endl);
    }

    new_vars.push_back(btp); 

    return new_vars;
}

BaseType *
NCAccess::find_child_sequence()
{
    return 0;
}


// $Log: NCAccess.cc,v $
// Revision 1.14  2005/06/03 16:33:21  jimg
// Added instrumentation using to find bugs found while working on bug 773.
//
// Revision 1.13  2005/04/11 18:38:20  jimg
// Fixed a problem with NCSequence where nested sequences were not flagged
// but instead were translated. The extract_values software cannot process a
// nested sequence yet. Now the code inserts an attribute that notes that a
// nested sequence has been elided.
//
// Revision 1.12  2005/04/07 23:35:36  jimg
// Changed the value of the translation attribute from "translated" to "flatten".
//
// Revision 1.11  2005/04/06 20:50:07  jimg
// Fixed a problem reported by Kevin O'Brien where Grids were not processed
// correctly. The fix is in NCConnect::var_info(). The case where a netCDF array
// is an Opendap Grid was not handled.
//
// Revision 1.10  2005/03/05 00:16:58  jimg
// checkpoint: working on memory leaks found using unit tests
//
// Revision 1.9  2005/03/04 18:10:49  jimg
// At this point valgrind runs the Unidata tests for both local and remote access
// and shows no errors or leaks. There are 8 bytes still reachable from an
// exception, but that's it.
//
// Revision 1.8  2005/03/02 17:51:50  jimg
// Considerable reduction in memory leaks and fixed all errant memory
// accesses found with nc_test. OPeNDAP error codes and Error object
// message strings are now reported using the nc_strerrror() function!
//
// Revision 1.7  2005/02/26 00:43:20  jimg
// Check point: This version of the CL can now translate strings from the
// server into char arrays. This is controlled by two things: First a
// compile-time directive STRING_AS_ARRAY can be used to remove/include
// this feature. When included in the code, only Strings associated with
// variables created by the translation process will be turned into char
// arrays. Other String variables are assumed to be single character strings
// (although there may be a bug with the way these are handled, see
// NCAccess::extract_values()).
//
// Revision 1.6  2005/02/17 23:44:13  jimg
// Modifications for processing of command line projections combined
// with the limit stuff and projection info passed in from the API. I also
// consolodated some of the code by moving d_source from various
// classes to NCAccess. This may it so that DODvario() could be simplified
// as could build_constraint() and store_projection() in NCArray.
//
// Revision 1.5  2005/01/26 23:25:51  jimg
// Implemented a fix for Sequence access by row number when talking to a
// 3.4 or earlier server (which contains a bug in is_end_of_rows()).
//
// Revision 1.4  2004/11/30 22:11:35  jimg
// I replaced the flatten_*() functions with a flatten() method in
// NCAccess. The default version of this method is in NCAccess and works
// for the atomic types; constructors must provide a specialization.
// Then I removed the code that copied the variables from vectors to
// lists. The translation code in NCConnect was modified to use the
// new method.
//
// Revision 1.3  2004/11/05 17:06:35  jimg
// Fiddled with the comments a bit...
//
// Revision 1.2  2004/10/22 21:51:34  jimg
// More massive changes: Translation of Sequences now works so long as the
// Sequence contains only atomic types.
//
// Revision 1.1  2004/09/08 22:09:48  jimg
// Added
//
