
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libnc-dods, A C++ implementation of netCDF 3 which
// supports reading from local files and OPeNDAP servers.

// Copyright (c) 2004 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.

#include "ClientParams.h"

#include <sstream>

#include "debug.h"

using namespace std;

static int
parse_int(stringstream &iparams, const string &param)
{
    string i_string;
    iparams >> ws >> i_string;
    if (i_string.empty())
        throw Error(string("Could not read a value for the '") + param
                    + string("' parameter."));

    int i;
    char *end = NULL;
    char **endptr = &end;
    i = strtol(i_string.c_str(), endptr, 0);
    if (**endptr != '\0')
        throw Error(string("Could not read a positive integer value for '")
                    + param + string("'. Instead I read: '")
                    + i_string + string("'."));
    return i;
}


/** Initialize an instance using the information in a URL.
    @param url The DAP URL with client-side parameters. */
ClientParams::ClientParams(const string &url) throw(Error)
    : d_url(url), d_preload(false), d_default_limit(0)
{
    // Get the params off the front of the URL and replace []= with spaces.
    string params(d_url.substr(0, d_url.find("http://")));

    string::size_type pos;
    while ((pos = params.find_first_of("[]=")) != string::npos)
        params.replace(pos, 1, " ");
        
    stringstream iparams(params);
    string param;

    while ((iparams >> ws >> param) && !iparams.eof()) {
        DBG(cerr << "param: '" << param << "'" << endl);
        if (param == "limit") {
            d_default_limit = parse_int(iparams, "limit");
        }
        else if (param == "preload") {
            string bool_val;
            iparams >> ws >> bool_val;
            DBG(cerr << "bool_val: '" << bool_val << "'" << endl);
            if (bool_val == "true")
                d_preload = true;
            else if (bool_val == "false")
                d_preload = false;
            else
                throw Error("The value: '" + bool_val 
                            + "' is not a valid boolean value. "
                            + "Please use 'true' or 'false'");
        }
        else if (param.find("limit-") != string::npos) {
            int val = parse_int(iparams, param);
            string var_name = param.substr(param.find("-") + 1);
            d_limits[var_name] = val;
        }
        else {
            throw Error("The client-side parameter: '" + param
                        + "' is not valid.\n"
                        + "Supported parameters are: 'limit', 'preload' and 'limit-<varname>'");
        }
    }
}

/** Get the value of the \c limit parameter for the variable \e var_name. 
    Return the default value if no specific value of \c limit was set for 
    \e var_name.
    @param var_name The name of the variable.
    @return The value of \c limit. */
int
ClientParams::get_limit(const string &var_name) const
{
    map<string,int>::const_iterator pos = d_limits.find(var_name);
    if (pos == d_limits.end())
        return d_default_limit;
    else
        return pos->second;
}

/** Given a url which may be prefixed with client-side params, return the 
    part of the URL without the parameters. This depends on a DAP URL
    beginning with "http://".
    @param url The URL
    @return The URL minus any client-side parameters. */
string
ClientParams::url_without_params() const
{
    string::size_type p = d_url.find("http://");
    if (p == string::npos)
        return d_url;
    else
        return d_url.substr(p);
}

// $Log: ClientParams.cc,v $
// Revision 1.3  2004/11/16 23:19:53  jimg
// Bug fix: URLs which are really local file names broke the
// url_without_params() method.
//
// Revision 1.2  2004/10/28 16:38:19  jimg
// Added support for error handling to ClientParams. Added use of
// ClientParams to NCConnect, although that's not complete yet. NCConnect
// now has an instance of ClientParams. The instance is first built and
// then passed into NCConnect's ctor which stores a const reference to the CP
// object.
//
// Revision 1.1  2004/10/27 19:28:22  jimg
// Added
//
