
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libnc-dods, A C++ implementation of netCDF 3 which
// supports reading from local files and OPeNDAP servers.

// Copyright (c) 2004 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.
 
#include <cppunit/TextTestRunner.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/extensions/HelperMacros.h>

// Tests for NCConnect. 02/24/04 jhrg

#include "netcdf.h"

#include "NCConnect.h"
#include "NCSequence.h"
#include "NCStructure.h"
#include "NCArray.h"
#include "NCByte.h"
#include "NCGrid.h"
#include "nc_util.h"

#include "debug.h"

using namespace CppUnit;
using namespace std;

class NCConnectTest : public TestFixture {
private:
    NCConnect *ncc;
    ClientParams *cp;
public: 
    NCConnectTest() {}
    ~NCConnectTest() {}

    void setUp() {
        cp = new ClientParams("[string_as_char_array=true]http://localhost/");
        ncc = new NCConnect("http://localhost/", cp);
    }

    void tearDown() {
	   delete ncc; ncc = 0;
    }

    CPPUNIT_TEST_SUITE( NCConnectTest );
    CPPUNIT_TEST(parse_string_dims_test1);
    CPPUNIT_TEST(parse_string_dims_test2);
    CPPUNIT_TEST(store_ce_test);
    CPPUNIT_TEST(flatten_attributes_test);
    CPPUNIT_TEST(global_attribute_processing_test);
    CPPUNIT_TEST(translate_dds_test1);
    CPPUNIT_TEST(translate_dds_test2);
    CPPUNIT_TEST(translate_dds_test3);
    CPPUNIT_TEST(translate_dds_test4);
    CPPUNIT_TEST(translate_dds_test5);
    CPPUNIT_TEST(translate_dds_test6);
    CPPUNIT_TEST_SUITE_END();
    
    void parse_string_dims_test1() {
        // read in DDS
        ncc->d_constrained_dds.parse("testsuite/string1.dds");
        CPPUNIT_ASSERT(ncc->d_ndims == 0);
        
        // Set the translated propery and call parse_string_dims.
        dynamic_cast<NCAccess&>(**ncc->d_constrained_dds.var_begin()).set_translated(true);
        ncc->parse_string_dims(ncc->d_constrained_dds);
        CPPUNIT_ASSERT(ncc->d_ndims == 1);
        CPPUNIT_ASSERT(ncc->d_dim_name[0] == "s1-chars");
        CPPUNIT_ASSERT(ncc->d_dim_size[0] == 128);
    }

    // Really testing parse_array_dims with an array of strings.
    void parse_string_dims_test2() {
        ncc->d_constrained_dds.parse("testsuite/string2.dds");
        CPPUNIT_ASSERT(ncc->d_ndims == 0);
        dynamic_cast<NCAccess&>(**ncc->d_constrained_dds.var_begin()).set_translated(true);
        ncc->parse_array_dims(ncc->d_constrained_dds);
        DBG(cerr << "ncc->d_ndims: " << ncc->d_ndims << endl);
        //CPPUNIT_ASSERT(ncc->d_ndims == 2);
        
        DBG(cerr << ncc->d_dim_name[0] << endl);
        DBG(cerr << ncc->d_dim_size[0] << endl);
        DBG(cerr << ncc->d_dim_name[1] << endl);
        DBG(cerr << ncc->d_dim_size[1] << endl);
        
        CPPUNIT_ASSERT(ncc->d_dim_name[0] == "s2_0");
        CPPUNIT_ASSERT(ncc->d_dim_size[0] == 256);

        CPPUNIT_ASSERT(ncc->d_dim_name[1] == "s2-chars");
        CPPUNIT_ASSERT(ncc->d_dim_size[1] == 128);
    }
    
    void store_ce_test() {
        ncc->store_ce("");
        CPPUNIT_ASSERT(ncc->d_proj_ce == "" && ncc->d_sel_ce == "");
        
        ncc->store_ce("x,y");
        CPPUNIT_ASSERT(ncc->d_proj_ce == "x,y" && ncc->d_sel_ce == "");
        
        ncc->store_ce("x[10:17]");
        CPPUNIT_ASSERT(ncc->d_proj_ce == "x[10:17]" && ncc->d_sel_ce == "");
        
        ncc->store_ce("&y<10");
        CPPUNIT_ASSERT(ncc->d_proj_ce == "" && ncc->d_sel_ce == "&y<10");
        
        ncc->store_ce("x[0:9]&y<10&z<20");
        CPPUNIT_ASSERT(ncc->d_proj_ce == "x[0:9]" && ncc->d_sel_ce == "&y<10&z<20");
    }
    
    void flatten_attributes_test() {
        DAS das;
        AttrTable *attr;
        AttrTable::Attr_iter i;
        try {
            das.parse("testsuite/fnoc1.nc.das");
            // Get NC_GLOBAL, flatten and test.
            attr = das.find_container("NC_GLOBAL");
            AttrTable *at = ncc->flatten_attributes(attr);
            // delete attr; Normally, you'd delete this, but in this case
            // attr points into an object that will be deleted later on.
            attr = at;
            i = attr->attr_begin();
            CPPUNIT_ASSERT(attr->get_name(i) == "base_time");
            CPPUNIT_ASSERT(attr->get_name(++i) == "title");        
            delete at; at = 0;
            
            das.parse("testsuite/3B42.980909.5.HDF.das");
            // Get CoreMetadata, flatten and test.
            attr  = das.find_container("CoreMetadata");
            DBG2(stderr << "CoreMetadata table before flatten:" << endl);
            DBG2(attr->print(stderr));
            at = ncc->flatten_attributes(attr);
            // delete attr;
            attr = at;
            DBG2(stderr << "CoreMetadata table after flatten:" << endl);
            DBG2(attr->print(stderr));
            i = attr->attr_begin();
            DBG(cerr << "Attr name: " << attr->get_name(i) << endl);
            CPPUNIT_ASSERT(attr->get_name(i) == "OrbitNumber:Value");
            CPPUNIT_ASSERT(attr->get_attr_num(i) == 1);
            CPPUNIT_ASSERT(attr->get_name(++i) == "OrbitNumber:Data_Location");
            CPPUNIT_ASSERT(attr->get_attr_num(i) == 1);
            ++i;
            CPPUNIT_ASSERT(attr->get_name(++i) == "RangeBeginningDate:Value");
            CPPUNIT_ASSERT(attr->get_attr_num(i) == 1);
            delete at; at = 0;
            
            das.parse("testsuite/nested.das");
            attr = das.find_container("Main");
            DBG(cerr << "Main table before flatten:" << endl);
            DBGN(attr->print(stderr));
            
            at = ncc->flatten_attributes(attr);
            // delete attr;
            attr = at;
            
            DBG(cerr << "Main table after flatten:" << endl);
            DBGN(attr->print(stderr));
            i = attr->attr_begin();
            DBG(cerr << "Attr name: " << attr->get_name(i) << endl);
            CPPUNIT_ASSERT(attr->get_name(i) == "first_value");
            CPPUNIT_ASSERT(attr->get_attr_num(i) == 1);
            ++i;
            DBG(cerr << "Attr name: " << attr->get_name(i) << endl);
            CPPUNIT_ASSERT(attr->get_name(i) == "Inner_1:value");
            CPPUNIT_ASSERT(attr->get_attr_num(i) == 1);
            ++i;
            DBG(cerr << "Attr name: " << attr->get_name(i) << endl);
            CPPUNIT_ASSERT(attr->get_name(i) == "Inner_1:Inner_2:value_2");
            CPPUNIT_ASSERT(attr->get_attr_num(i) == 1);
            delete at; at = 0;
        }
        catch (Error &e) {
             cerr << "Error: " << e.get_error_message() << endl;
             CPPUNIT_ASSERT(!"Caught an Error");
        }
    }

    void global_attribute_processing_test() {
        try {
            // Kludge together the initial configuration...
            ncc->d_constrained_dds.parse("testsuite/3B42.980909.5.HDF.dds");
            
            DAS das;
            das.parse("testsuite/3B42.980909.5.HDF.das");
    
            // Code from init_remote_source() starts here...
            
            // Combine the Attributes with the variables in the DDS.
            ncc->d_constrained_dds.transfer_attributes(&das);
        
            // Search for global attributes. For netCDF the global attributes must
            // all be atomic; no containers within the global attr container.
            ncc->set_global_attributes();
            
            DBG2(cerr << "Global attributes before flattening: " 
                 << ncc->d_global_attributes->get_name() << endl);
            DBG2(ncc->d_global_attributes->print(cerr));
            AttrTable *at = ncc->flatten_attributes(ncc->d_global_attributes);
            delete ncc->d_global_attributes;
            ncc->d_global_attributes = at;

            DBG2(cerr << "Global attributes after flatten call:" 
                 << ncc->d_global_attributes->get_name() << endl);
            DBG2(ncc->d_global_attributes->print(cerr));
            

            AttrTable *attr = ncc->d_global_attributes;
            AttrTable::Attr_iter i = attr->attr_begin();
            CPPUNIT_ASSERT(attr->get_name(i) == "OrbitNumber:Value");
            CPPUNIT_ASSERT(attr->get_attr_num(i) == 1);
            CPPUNIT_ASSERT(attr->get_name(++i) == "OrbitNumber:Data_Location");
            CPPUNIT_ASSERT(attr->get_attr_num(i) == 1);
            ++i;
            CPPUNIT_ASSERT(attr->get_name(++i) == "RangeBeginningDate:Value");
            CPPUNIT_ASSERT(attr->get_attr_num(i) == 1);
        }
        catch (Error &e) {
             cerr << "Error: " << e.get_error_message() << endl;
             CPPUNIT_ASSERT(!"Error!");
        }
    }
    
    void translate_dds_test1() {
        ncc->d_constrained_dds.parse("testsuite/atomic_vars.dds");
        if (!ncc->d_global_attributes)
            ncc->d_global_attributes = new AttrTable;

        ncc ->translate_dds();
        DBG(ncc->d_translated_dds.print(stderr));
        DDS::Vars_iter i = ncc->d_translated_dds.var_begin();
        CPPUNIT_ASSERT((*i++)->name() == "b1");
        CPPUNIT_ASSERT((*i++)->name() == "i2");
        CPPUNIT_ASSERT(i == ncc->d_translated_dds.var_end());
    }
    
    void translate_dds_test2() {
        ncc->d_constrained_dds.parse("testsuite/structure_vars.dds");
        if (!ncc->d_global_attributes)
            ncc->d_global_attributes = new AttrTable;

        ncc ->translate_dds();
        DBG(ncc->d_translated_dds.print(stderr));
        DDS::Vars_iter i = ncc->d_translated_dds.var_begin();
        CPPUNIT_ASSERT((*i++)->name() == "t.i5");
        CPPUNIT_ASSERT((*i++)->name() == "t.s6");
        CPPUNIT_ASSERT(i == ncc->d_translated_dds.var_end());
    }

    void translate_dds_test3() {
        ncc->d_constrained_dds.parse("testsuite/sequence_vars.dds");
        if (!ncc->d_global_attributes)
            ncc->d_global_attributes = new AttrTable;

        ncc ->translate_dds();
        DBG(ncc->d_translated_dds.print(stderr));
        DDS::Vars_iter i = ncc->d_translated_dds.var_begin();
        CPPUNIT_ASSERT((*i++)->name() == "s.f3");
        CPPUNIT_ASSERT((*i++)->name() == "s.i4");
        CPPUNIT_ASSERT(i == ncc->d_translated_dds.var_end());
    }

    void translate_dds_test4() {
        ncc->d_constrained_dds.parse("testsuite/array_vars.dds");
        if (!ncc->d_global_attributes)
            ncc->d_global_attributes = new AttrTable;

        ncc->translate_dds();
        DBG(ncc->d_translated_dds.print(stderr));
        DDS::Vars_iter i = ncc->d_translated_dds.var_begin();
        CPPUNIT_ASSERT((*i++)->name() == "b8");
        CPPUNIT_ASSERT((*i++)->name() == "points.lat");
        CPPUNIT_ASSERT((*i++)->name() == "points.lon");
        CPPUNIT_ASSERT(i == ncc->d_translated_dds.var_end());
    }

    void translate_dds_test5() {
        ncc->d_constrained_dds.parse("testsuite/complex_vars.dds");
        if (!ncc->d_global_attributes)
            ncc->d_global_attributes = new AttrTable;
        
        ncc->translate_dds();
        DBG(ncc->d_translated_dds.print(stderr));
        DDS::Vars_iter i = ncc->d_translated_dds.var_begin();
        CPPUNIT_ASSERT((*i++)->name() == "struct.ii");
        CPPUNIT_ASSERT((*i++)->name() == "struct.ss.iii");
        CPPUNIT_ASSERT((*i++)->name() == "struct.ss.jjj");
        CPPUNIT_ASSERT((*i++)->name() == "struct.seq.name");
        CPPUNIT_ASSERT((*i++)->name() == "struct.seq.temp");
        CPPUNIT_ASSERT((*i++)->name() == "struct.values");
        CPPUNIT_ASSERT((*i++)->name() == "sequence.ii");
        CPPUNIT_ASSERT((*i++)->name() == "sequence.values");
        CPPUNIT_ASSERT((*i++)->name() == "sequence.ss.iii");
        CPPUNIT_ASSERT((*i++)->name() == "sequence.ss.jjj");
        // CPPUNIT_ASSERT((*i++)->name() == "sequence.seq.name");
        // CPPUNIT_ASSERT((*i++)->name() == "sequence.seq.temp");
        CPPUNIT_ASSERT(i == ncc->d_translated_dds.var_end());
    }

    void translate_dds_test6() {
        ncc->d_constrained_dds.parse("testsuite/grid_vars.dds");
        if (!ncc->d_global_attributes)
            ncc->d_global_attributes = new AttrTable;

        ncc ->translate_dds();
        DBG(ncc->d_translated_dds.print(stderr));
        DDS::Vars_iter i = ncc->d_translated_dds.var_begin();
        CPPUNIT_ASSERT((*i++)->name() == "b1");
        CPPUNIT_ASSERT((*i)->name() == "g");
        NCGrid *g = dynamic_cast<NCGrid*>(*i);
        CPPUNIT_ASSERT(g->array_var()->name() == "v");
        Grid::Map_iter j = g->map_begin();
        CPPUNIT_ASSERT((*j++)->name() == "x");
        CPPUNIT_ASSERT((*j++)->name() == "y");
        CPPUNIT_ASSERT(j == g->map_end());
        // (*i)->print_decl(cerr);
        ++i;
        CPPUNIT_ASSERT((*i++)->name() == "i2");
        CPPUNIT_ASSERT(i == ncc->d_translated_dds.var_end());
    }

    void set_recdim_test() {
    }
};

CPPUNIT_TEST_SUITE_REGISTRATION(NCConnectTest);

int 
main( int argc, char* argv[] )
{
    CppUnit::TextTestRunner runner;
    runner.addTest( CppUnit::TestFactoryRegistry::getRegistry().makeTest() );

    bool wasSuccessful = runner.run( "", false ) ;

    return wasSuccessful ? 0 : 1;
}
