!******************************************************************************
!  2003-10-01
!    const_set subroutine is added. 
!------------------------------------------------------------------------------
!  2003-09-07
!******************************************************************************

  module const_module

    use vtype_module

    implicit none


    ! pi
    ! pix2     : 2 * pi
    !
    ! pi2      : same as pix2 (this is remained for the purpose of compativility)
    !
    ! d2r      : pi / 180
    ! r2d      : 180 / pi
    ! c        : speed of light (m/s)
    ! amu      : atomic mass unit
    ! planc    : planck constant
    ! boltz    : boltzmann constant
    ! ugasc    : universal gas constant
    ! gconst   : gravitational constant
    !
    real(dp), parameter :: &
         pi     = 3.1415926535897932385d0, &
         pix2   = 2.0d0 * pi             , &
         pi2    = pix2                   , &
         d2r    = pi / 180.0d0           , &
         r2d    = 180.0d0 / pi           , &
         c      = 2.99792458d8           , &
         amu    = 1.6605655d-27          , &
         planc  = 6.6260755d-34          , &
         boltz  = 1.380658d-23           , &
         ugasc  = 8.314510d0             , &
         gconst = 6.67260d-11


    !
    ! sboltz   : Stefan-Boltzmann constant
    !
    real(dp), parameter :: sboltz &
         = 2.0d0*pi*pi*pi*pi*pi*boltz*boltz*boltz*boltz &
         /(15.0d0*c*c*planc*planc*planc)

    !
    ! dayearth  : 1 day of the Earth (sec)
    !
    real(dp), parameter :: dayearth = 60.0d0 * 60.0d0 * 24.0d0


    !**************************************************************************

    ! co2lh     : Latent heat of CO2 (J kg^{-1})
    !           : This value is obtained from Pollack et al. [1981] and 
    !           : Forget et al. [1998]. 
    !
    real(dp), parameter :: co2lh = 5.9d5

    !**************************************************************************

    !
    ! pradi    : planetary radius (m)
    ! pradisq  : planetary radius squared (m^2)
    !
    ! pradi2   : same as pradisq (this is remained for the purpose of 
    !          : compativility)
    !
    ! grav     : gravitational acceleration (m/s^2)
    !
    real(dp), save :: pradi
    real(dp), save :: pradisq
    real(dp), save :: pradi2
    real(dp), save :: grav


    ! hour     : one solar hour (sec)
    ! day      : one solar day (sec)
    ! omega    : planetary angular velocity (rad/sec)
    !
    real(dp), save :: hour
    real(dp), save :: day
    real(dp), save :: omega

    ! cp       : specific heat at constant pressure
    !
    real(dp), save :: cp


    data pradi     / 1.0d100 /
    data pradisq   / 1.0d100 /
    data pradi2    / 1.0d100 /
    data grav      / 1.0d100 /
    data hour      / 1.0d100 /
    data day       / 1.0d100 /
    data omega     / 1.0d100 /

    data cp        / 1.0d100 /


    !**************************************************************************

    public :: pi, pix2, pi2, c, amu, planc, boltz, ugasc, sboltz
    public :: pradi, pradisq, pradi2, grav, hour, day, omega
    public :: co2lh

    public :: cp

    public :: const_set

    !**************************************************************************

  contains

    !**************************************************************************
    ! planetid  : Planet ID
    !           : 0  : the Earth
    !           : 1  : Mars
    !**************************************************************************

    subroutine const_set( planetid, planetname )

      character(len=*), intent(in), optional :: planetname
      integer(i4b)    , intent(in), optional :: planetid


      ! Local variables
      !
      integer(i4b) :: pid


      if( present( planetname ) ) then

         if( ( planetname .eq. 'Earth' ) &
              .or. ( planetname .eq. 'EARTH' ) ) then
            pid = 0
         else if( ( planetname .eq. 'Mars' ) &
              .or. ( planetname .eq. 'MARS' ) ) then
            pid = 1
         else
            write( 6, * ) 'Unexpected planet name : ', trim( planetname ), '.'
            stop
         end if

      else

         if( present( planetid ) ) then
            pid = planetid
         else
            pid = 0
         end if

      end if


      select case( pid )
      case( 0 )

         !
         ! for the Earth
         !
         pradi = 6371.0d3
         grav  = 9.81d0
!         omega = pix2 / dayearth
         omega = 7.292d-5
         day   = 1.0d0 * dayearth
         hour  = day / 24.0d0

      case( 1 )

         !
         ! for Mars
         !
         pradi = 3382.946d3
         omega = pix2 / ( 22.663d0 + 60.0d0 * ( 37.0d0 + 60.0d0 * 24.0d0 ) )
         day   = 1.0274912d0 * dayearth
         hour  = day / 24.0d0
         grav  = 3.72d0

      case default

         write( 6, * ) "Unexpected planet ID, ", pid

      end select

      pradisq = pradi * pradi
      pradi2  = pradisq

!      omega=0.0d0


    end subroutine const_set

    !**************************************************************************

  end module const_module
