/*
 * File:
 *
 *	view.c
 *
 * Author:
 *
 *	Brad Rullman
 *	Department of Computer Science  FR-35
 *	University of Washington
 *	Seattle, Washington  98155
 *	email: ecola@cs.washington.edu
 *
 *	Copyright @ March, 1987 - This program, or any subset of the functions
 *	herein, is not to be redistributed, or used for personal or commercial
 *	gain in the form of fame, fortune, or shorter working hours, without
 *	the consent of the author.
 *
 * Function:
 *
 *	A file of functions which implement the View subwindow.
 *	Also defines FontTool's drawing op's, which are actually
 *	performed in the ViewSW first, then scaled up to the PaintSW.
 *
 * Contents:
 *
 *   Public:
 *
 *	InitViewSW		Creates the View subwindow.
 *	FixViewSW		Resets the size of the ViewSW.
 *	DrawDot			Paints a dot (cell of pixels) on the screen.
 *	StartLine		Paints the first dot in a set of lines.
 *	DrawLine		Draws a line based on most recent mouse event
 *	FillRegion		Fills enclosed area designated by last click.
 *	MoveRefPoint		Moves the reference point to a new location.
 *	Undo			Undoes the previous paint window operation.
 * 
 *   Private Internal:
 *
 *	doFill			Paints a region of pixels on the screen.
 */

#include "global.h"

/*
 * Statics for when mouse is drawing lines.
 */
static struct pr_pos startPt;	/* startpoint of the current line */
static struct pr_pos endPt;	/* endpoint of the current line   */



/*
 * InitViewSW
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Creates the View subwindow, which is a canvas subwindow.
 */
void
InitViewSW()
{
    ViewSW = window_create(FontToolFrame, CANVAS,
			WIN_WIDTH,		MINVIEWSWWIDTH,
			WIN_HEIGHT, 		MINVIEWSWHEIGHT,
			WIN_CURSOR,		MainCursor,
			CANVAS_FAST_MONO,	TRUE,
			0);
    ViewPW = canvas_pixwin(ViewSW);
}



/*
 * FixViewSW
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Makes sure the ViewSW is sized correctly, in case the user has
 *	resized the main tool window, or the ViewSW has changed in size.
 */
void
FixViewSW()
{
    Rect r;

    r = (*(Rect *) window_get(FontToolFrame, WIN_RECT));
    ViewRect.r_width = MIN(ViewPR->pr_size.x, r.r_width - 2*FRAMEMARGIN);
    ViewRect.r_height = MIN(
	ViewPR->pr_size.y,
	r.r_height - (int) window_get(FontToolFrame, WIN_TOP_MARGIN)
	    - MessageRect.r_height - SWSPACING - FRAMEMARGIN);
    r = ViewRect;
    window_set(ViewSW, WIN_RECT, &r, 0);
}



/*
 * DrawDot
 *
 * Input: 
 *	xInput, yInput: ViewSW coordinates of a dot to be drawn.
 * Output: 
 *	none.
 * Action: 
 *	Draw a dot on the appropriate data structures, including the screen.
 */
void
DrawDot(xInput, yInput)
int xInput, yInput;
{
    pr_put(ViewPR, xInput, yInput, PaintColor);
    DisplayPR(xInput, yInput, 1, 1);
}



/*
 * StartLine
 *
 * Input: 
 *	xInput, yInput: starting point of a line in the ViewSW.
 * Output: 
 *	none.
 * Action: 
 *	This routine is called if we're in drawlines mode and the left mouse
 *	button has been pressed, indicating the start point of a new line.
 */
void
StartLine(xInput, yInput)
int xInput, yInput;
{
    startPt.x = endPt.x = xInput;
    startPt.y = endPt.y = yInput;
    ResetRect(&UndoRect, startPt.x, startPt.y, 1, 1);
    COPYPR(ViewPR, UndoPR);
    DrawDot(startPt.x, startPt.y);
}



/*
 * DrawLine
 *
 * Input:
 *	xInput, yInput: the ViewSW endpoint of the line.
 * Output:
 *	none.
 * Action:
 *	Proceeds to draw a rubber band line based on the given endpoint
 *	and the already saved startpoint.
 */
void
DrawLine(xInput, yInput)
int xInput, yInput;
{
    /*
     * If the new mouse position is in the same PaintSW cell
     * as last time, don't bother.
     */
    if ( (xInput == endPt.x) && (yInput == endPt.y) ) {
	return;
    }

    endPt.x = xInput;
    endPt.y = yInput;

    /*
     * Undo the previously drawn (unanchored) line, and expand
     * UndoRect to enclose the line to be drawn.
     */
    pr_rop(ViewPR, UndoRect.r_left, UndoRect.r_top,
	   UndoRect.r_width, UndoRect.r_height, PIX_SRC, 
	   UndoPR, UndoRect.r_left, UndoRect.r_top);
    ExpandRect(&UndoRect, endPt.x, endPt.y);

    /*
     * Draw a vector between the two points in the ViewPR, scale it
     * up and copy it into the PaintPR, and display it on screen.
     */
    pr_vector(ViewPR, startPt.x, startPt.y, endPt.x, endPt.y,
	      (PaintColor == BLACK) ? PIX_SET : PIX_CLR, 1);
    DisplayPR(UndoRect.r_left, UndoRect.r_top,
	      UndoRect.r_width, UndoRect.r_height);

    /*
     * The old line has now been erased in the PaintSW, and the new line
     * drawn.  Reset UndoRect to contain JUST the new line.
     */
    ResetRect(&UndoRect, startPt.x, startPt.y, 1, 1);
    ExpandRect(&UndoRect, endPt.x, endPt.y);
}



/*
 * FillRegion
 *
 * Input: 
 *	xInput, yInput: ViewSW coordinates of starting point of the fill op.
 * Output: 
 *	none.
 * Action: 
 *	Fills the closed region pointed to by the current (x, y) mouse
 *	position after resetting the appropriate undo structures.
 */
void
FillRegion(xInput, yInput)
int xInput, yInput;
{
    static void doFill();

    SETCURSORWATCH;
    ResetRect(&UndoRect, xInput, yInput, 1, 1);
    COPYPR(ViewPR, UndoPR);
    doFill(xInput, yInput, !PaintColor, PaintColor);
    DisplayPR(UndoRect.r_left, UndoRect.r_top,
	      UndoRect.r_width, UndoRect.r_height);
    SETCURSORNORMAL;
}



/*
 * doFill
 *
 * Input: 
 *	x, y        : ViewSW coordinates of starting point of fill operation.
 *	oldIntensity: 0 if old color to be re-painted is white, 1 if black.
 *	newIntensity: 1 or 0.
 * Output: 
 *	none.
 * Action: 
 *	Writes pixels of color newIntensity into ViewPR recursively using
 *	a 4-connected policy until no more pixels of oldIntensity can be
 *	reached.  Keeps track of modified area in ViewPR by constantly
 *	updating UndoRect.
 */
static void
doFill(x, y, oldIntensity, newIntensity)
int x, y;
int oldIntensity, newIntensity;
{
    if (pr_get(ViewPR, x, y) == oldIntensity) {
	pr_put(ViewPR, x, y, newIntensity);
	/*
	 * Expand the undo rectangle accordingly so we know what region
	 * to write to the screen when we are done.
	 */
	ExpandRect(&UndoRect, x, y);

	doFill(x + 1, y, oldIntensity, newIntensity);
	doFill(x - 1, y, oldIntensity, newIntensity);
	doFill(x, y + 1, oldIntensity, newIntensity);
	doFill(x, y - 1, oldIntensity, newIntensity);
    }
}



/*
 * MoveRefPoint
 *
 * Input:
 *	xInput, yInput: new location of the reference point in the PaintSW.
 * Output:
 *	none.
 * Action:
 *	Erases the reference point in the PaintSW, resets the current 
 *	reference point location, and redraws it in the PaintSW.
 */
void
MoveRefPoint(xInput, yInput)
int xInput, yInput;
{
    pw_write(PaintPW, ReferenceXY.x, ReferenceXY.y,
	     Magnification, Magnification, PIX_SRC, PaintPR,
	     ReferenceXY.x, ReferenceXY.y);
    ReferenceXY.x = xInput;
    ReferenceXY.y = yInput;
    DrawRefPoint();
}



/*
 * Undo
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Copies a bounding box (defined by UndoRect) in UndoPR into ViewPR
 *	(actually done by merely swapping pixrect pointers).  This is
 *	supposedly restoring ViewPR to it's state before the last major
 *	PaintSW operation.  Since the "Move Reference Point" option
 *	involves changing a variable containing coordinates rather than
 *	changing the ViewPR, we first check if this was the previous paint
 *	operation performed.
 */
void
Undo()
{
    struct pixrect *pr;
    struct pr_pos p;

    if (UndoReferenceXY.x > -1) {
	/*
	 * Undo the "Move Reference Point", and switch ReferenceXY and
	 * UndoReferenceXY in case "Undo" is called again.
	 */
	p.x = ReferenceXY.x;
	p.y = ReferenceXY.y;
	MoveRefPoint(UndoReferenceXY.x, UndoReferenceXY.y);
	UndoReferenceXY.x = p.x;
	UndoReferenceXY.y = p.y;
	return;
    }
    pr = UndoPR;
    UndoPR = ViewPR;
    ViewPR = pr;
    DisplayPR(UndoRect.r_left, UndoRect.r_top,
	      UndoRect.r_width, UndoRect.r_height);
}
