/*
 * File:
 *
 *	main.c
 *
 * Author:
 *
 *	Brad Rullman
 *	Department of Computer Science  FR-35
 *	University of Washington
 *	Seattle, Washington  98155
 *	email: ecola@cs.washington.edu
 *
 *	Copyright @ March, 1987 - This program, or any subset of the functions
 *	herein, is not to be redistributed, or used for personal or commercial
 *	gain in the form of fame, fortune, or shorter working hours, without
 *	the consent of the author.
 *
 * Description:
 *
 *	A SunView 3.2 application for editing TeX fonts.
 *
 * Contents:
 *
 *   Public:
 *
 *	main			The main driver.
 *	FixSubWindows		Resets sizes and positions of subwindows.
 *	RedisplayFrame		Redisplays frame and all subwindows.
 *	AbortWithError		Aborts execution, with a message.
 *
 *   Private:
 *
 *	initCursors		Initializes cursors used by FontTool.
 *	initFrame		Creates the main frame window.
 *	initPixrects		Initializes global pixrects.
 *	initRects		Initializes global rects.
 *	catchResize		Intercepts frame resize events.
 *
 * General coding rules:
 *
 *	Names of global variables (those in global.c) are capitalized - 
 *		all others are not.
 *	Functions that are used only within the same module as they are
 *		defined (private internal functions) are declared static.
 *		Names of static (private internal) functions start with a
 *		lower case letter.
 *	Names of all other (global) functions start with an upper case
 *		letter.
 *	For ease in differentiating ft variable and function names from
 *		SunView variable and function names, the former use 
 *		capital letters rather than underscore characters, e.g.
 *		OpenPxlFile vs. open_pxl_file.
 */

#include "global.h"

static short iconImage[] = {		/* FontTool icon */
#include "fonttool.icon"
};
DEFINE_ICON_FROM_IMAGE(fonttoolIcon, iconImage);



/*
 * Main driver.
 *
 */
main(argc, argv)
int argc;
char *argv[];
{
    static void initCursors(), initFrame(), initPixrects(), initRects();
    static Notify_value catchResize();

    initCursors();
    initPixrects();
    initFrame();
    InitMessageSW();
    InitViewSW();
    InitOptionSW();
    InitPaintSW();
    window_fit(FontToolFrame);
    initRects();

    /*
     * Insert an interposer so that whenever the frame gets resized
     * we can handle it ourselves.
     */
    (void) notify_interpose_event_func(
			FontToolFrame, catchResize, NOTIFY_SAFE);

    window_main_loop(FontToolFrame);

    window_destroy(FontToolFrame);
}



/*
 * initCursors
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Initializes the mouse cursors used in FontTool.  Normally the
 *	cursor will always be black, but in the PaintSW, where there
 *	may be large regions painted black, we want the cursor to be
 *	black on white paint and white on black paint.
 */
static void
initCursors()
{
    MainCursor = cursor_create(
			CURSOR_IMAGE,	&CursorPR,
			CURSOR_OP, 	PIX_SRC | PIX_DST,
			0);
    PaintCursor = cursor_create(
			CURSOR_IMAGE,	&CursorPR,
			CURSOR_OP, 	PIX_SRC ^ PIX_DST,
			0);
    WatchCursor = cursor_create(
			CURSOR_IMAGE,	&WatchCursorPR,
			CURSOR_OP, 	PIX_SRC | PIX_DST,
			0);
    PaintWatchCursor = cursor_create(
			CURSOR_IMAGE,	&WatchCursorPR,
			CURSOR_OP, 	PIX_SRC ^ PIX_DST,
			0);
}



/*
 * initFrame
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Create the main frame.
 */
static void
initFrame()
{
    FontToolFrame = window_create(NULL, FRAME,
			WIN_X,				100,
			WIN_Y,				50,
			FRAME_LABEL,			"FontTool",
			FRAME_SUBWINDOWS_ADJUSTABLE,	FALSE,
			FRAME_ICON,			&fonttoolIcon,
			0);
}



/*
 * initPixrects
 *
 * Input:
 *	none.
 * Output: 
 *	none.
 * Action:
 *	Creates and initializes the global pixrects.
 *	PaintPR maps the PaintSW.
 *	ViewPR maps the ViewSW, which is a "true-size" view of the PaintSW.
 *	UndoPR maps the ViewSW one major action previous.
 *	ReferencePoint contains a description of how the reference point
 *	  should be drawn at the default magnification.
 */
static void
initPixrects()
{
    PaintPR = mem_create(SCALEUP(MINVIEWSWWIDTH), SCALEUP(MINVIEWSWHEIGHT),1);
    ViewPR = mem_create(MINVIEWSWWIDTH, MINVIEWSWHEIGHT, 1);
    UndoPR = mem_create(MINVIEWSWWIDTH, MINVIEWSWHEIGHT, 1);
    ReferenceXY.x = SCALEUP(((int) (.2 * ViewPR->pr_size.x)));
    ReferenceXY.y = SCALEUP(((int) (.667 * ViewPR->pr_size.y)));
    CLEARUNDOREFPT;
    ReferencePoint = mem_create(MAXMAG, MAXMAG, 1);
    pr_vector(
	ReferencePoint, 0, 0, MAXMAG-1, MAXMAG-1, PIX_SET, 1);
    pr_vector(
	ReferencePoint, 0, MAXMAG-1, MAXMAG-1, 0, PIX_SET, 1);
}



/*
 * initRects
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Initializes the rect structures that, for convenience, keep track of 
 *	the sizes and positions of the subwindows.
 */
static void
initRects()
{
    Rect *r;

    r = (Rect *) window_get(MessageSW, WIN_RECT);
    MessageRect = *r;
    r = (Rect *) window_get(ViewSW, WIN_RECT);
    ViewRect = *r;
    r = (Rect *) window_get(OptionSW, WIN_RECT);
    OptionRect = *r;
    OptionSWHeight = r->r_height;
    r = (Rect *) window_get(PaintSW, WIN_RECT);
    PaintRect = *r;
}



/*
 * catchResize
 *
 * Input:
 *	frame: The frame from which the resize event was initiated.
 *	event: The struct containing the event description.
 *	arg  : "An optional argument containing data pertinent to the event."
 *	type : Type of notification.
 * Output:
 *	A Notify_value, as required.
 * Action:
 *	Interposed function which checks all input events passed to the frame
 *	for resize events; if it finds one, the subwindows are resized
 *	and redisplayed.  Note that checking is done AFTER the frame
 *	processes the event because if the frame changes its size due to
 *	this event (because the window has been opened or closed for instance)
 *	we want to fit the subwindows to the new size.
 */
static Notify_value
catchResize(frame, event, arg, type)
Frame frame;
Event *event;
Notify_arg arg;
Notify_event_type type;
{
    Notify_value value;
    void FixSubwindows();

    if (event_id(event) == WIN_RESIZE) {
	/*
	 * If the window is not iconic, go ahead and fix things up.
	 */
	if ((int)window_get(frame, FRAME_CLOSED) == 0) {
	    FixSubwindows();
	    RedisplayPRs();
	}
    }
    value = notify_next_event_func(frame, event, arg, type);
    return(value);
}



/*
 * FixSubwindows
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Compute and set the sizes of any subwindows that may be affected by
 *	a resizing of the frame or of the ViewSW.
 */
void
FixSubwindows()
{
    FixViewSW();
    FixOptionSW();
    FixPaintSW();
}



/*
 * RedisplayFrame
 *
 * Input:
 *	none.
 * Output:
 *	none.
 * Action:
 *	Forces the frame and its contents to be redisplayed, to remove
 *	garbage left behind after subwindows have been resized or repositioned.
 */
void
RedisplayFrame()
{
    int framefd;	/* file descriptor for the FontTool frame */

    framefd = (int) window_get(FontToolFrame, WIN_FD);
    wmgr_refreshwindow(framefd);
}



/*
 * AbortWithError
 *
 * Input:
 *	A string to be printed on standard error.
 * Output:
 *	none.
 * Action:
 *	Prints the passed message, and aborts the tool with an error code.
 */
void
AbortWithError(string)
char *string;
{
	fprintf(stderr, string);
	exit(-1);
}
