/* kyodev.c The kyocera output driver for dvi2kyo program.
 * Copyright 1987 State University Groningen, Netherlands
 * Author: kees@guvaxin.uucp
 */

#include <stdio.h>
#include "dev.h"
#include "kyo.h"

#define DEV_DPI             300 /* Device dots per inch */
#define DEV_XOFFSET         200
#define DEV_YOFFSET         300

FILE *dev_out;
long dev_h,dev_v;
unsigned long dev_options;

/* Map the ASCII codes 0-060 to 0240-0320
 * because the Kyocera can not print symbols with
 * codes 0-040 in each emulation mode, and the ASCII
 * sequence !R! brings it into PRESCRIBE command mode
 */
unsigned char visual(ch)
     unsigned long ch;
{
  if (ch < 060) ch += 0240;
  return((unsigned char)ch);
}

/* Print symbol "ch" from current font
 * and update the horizontal cursor position
 */
dev_setc(ch,xsize)
     unsigned long ch;
     int xsize;
{
  putc(visual(ch), dev_out);
  dev_h += xsize;
}

/* Print symbol "c" by sending the pixel raster 
 * The raster info is converted to a vectorlike image
 * and the symbol is printed by drawing lines
 */
dev_raster(c)
     struct kyochar *c;
{
  int i,w,row,l,s,bw;
  unsigned short *bp, x;

  debug("Output raster info\n");
  bp = c->kc_glyph.p;
  bw = (c->kc_width + 15) / 16;
  fprintf(dev_out, "!R! SCP;MRP %1d,%1d;", -(c->kc_xoffset), -(c->kc_yoffset));
  for (row=0; row<c->kc_height; row++) {
    l = 0; s = 0;
    fprintf(dev_out, "SCP;MRP 0,%1d;", row);
    debug("row: %d\n", row);
    for (w=0; w<bw; w++) {
      x = *bp++;
      if ((w % 8) == 0) debug("\n");
      debug("%o ", x);
      for (i=15; i>=0; i--)
	if (x & (1 << i)) {
	  if (s) {
	    fprintf(dev_out, "MRP %1d,0;", s);
	    s = 0;
	  }
	  l++;
	} else {
	  if (l) {
	    fprintf(dev_out, "DRP %1d,0;", l);
	    l = 0;
	  }
	  s++;
	}
    }
    if (l) fprintf(dev_out, "DRP %1d,0;", l);
    fprintf(dev_out, "RPP;");
  }
  fprintf(dev_out, "EXIT;");
}

/* Put the cursor on position "(x,y)" */
dev_position(x,y)
     long x,y;
{
  if ((dev_h != x) || (dev_v != y))
      fprintf(dev_out, "!R! MAP %1d,%1d;EXIT;", x, y);
  dev_h = x; dev_v = y;
}

/* Draw a filled rectangele */
dev_draw_box(h,w)
     long h,w;
{
  fprintf(dev_out, "!R! BLK %1d,%1d;EXIT;", w, -h);
}

/* End of page */
dev_eop()
{
  fprintf(dev_out,"!R! PAGE;EXIT;");
  fflush(dev_out);
  dev_h = dev_v = 0;
}

/* Initialize Kyocera */
hard_init()
{
  fprintf(dev_out, "!R! RES;UNIT D;SLM %1d;STM %1d;SPD 1;EXIT;"
		 , DEV_XOFFSET, DEV_YOFFSET);
  page_init();
}

/* Initialization at each page begin */
page_init()
{
  dev_h = dev_v = 0;
}

/* Initialization routine called by device independent part */
dev_init(f,options,dpi)
     FILE *f;
     unsigned long options;
     long *dpi;
{
  dev_out = f;
  *dpi = DEV_DPI;
  dev_options = options;
  hard_init();
}

/* Terminate use of device */
dev_term()
{
  fprintf(dev_out, "!R! PAGE;RES; EXIT;!R! RES; EXIT;");
  fflush(dev_out);
}

/* Print error log */
dev_print_log(f)
     FILE *f;
{
  register int ch;

  if (dev_h || dev_v) putc('\f',dev_out);
  fprintf(dev_out, "!R! FONT 1;EXIT;");
  while (!ferror(f) && (ch = getc(f)) != EOF) {
    if (ch == '\n') putc('\r',dev_out);
    putc(ch,dev_out);
  }
  putc('\f',dev_out);
  fflush(dev_out);
}

/* Handle special DVI commands
 * "nchars" are copied from file "f"
 * directly to the Kyocera
 */
dev_special(nchars,f)
     int nchars;
     FILE *f;
{
  while (nchars-- > 0) putc(getc(f),dev_out);
  page_init();
}

