/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/* Generation of pdr file stuff.  */

#include <stdio.h>
#include <sys/file.h>
#include  <math.h>
#include "units.h"
#include "defs.h"
#include "char.h"
#include "pdr.h"
#include "tfm.h"
#include "extfil.h"

extern int Verbose;
extern CAFM AfmChar[MAX_CHAR_PS];
extern int EncodingClass;
extern char * EmulateFontName;
/* Vice versa: floating point number to 32.20 bit integer. */
#define	FLOAT_TO_FIXES(x) ((int)((x+0.0)*(double)(1<<20)))

/*
 * Variables, which have to do with reading in the .pfd file.
 * This is done in a procedure, so we can reinitialize and then
 * process more than one .pfd file.
 */

/* Name of the .pfd file read in. */
char *PfdFileName;
extern char * OutputFileNameBase;
/* Basefont of the font being discussed. */
char * BaseFontName;
/* HRatio (Horizontal Ratio). */
double HRatio;

/* Width Delta: unit is percentage of the design size of the font,
 * where the design size is normally 10pt. */
double WidthDelta;
/* TiltAngle */
double TiltAngle;
/* Outline Factor: 0.0 is no outline font. */
double OutlineFactor;
/* Monocase fonts: ratio of lower and upper chars, 0.0: no mono case */
double MonoCase;
/* Whether the font is downloadable or not. */
int DownLoadable;
/* Is it an invisible Slitex font? */
int SlitexInvisible;

/* Design size of the font */
double DesignSize, DesignSizeDef;

/* Slant, stretch, shrink, regular space */
int Slant, SlantDef;
int Stretch, StretchDef;
int Shrink, ShrinkDef;
int Space, SpaceDef;
int ExtraSpace, ExtraSpaceDef;
int Quad, QuadDef;
int XHeight, XHeightDef;

/* Design size of the font */
double DesignSize, DesignSizeDef;

/* Name of file where all \TeX definitions are stored */
char * TeXMacroDefs;

/* Parameter list (\fontdimen ...), up to 22, initialized to -1 */
int FontDimenArray[23];

/* File name and file pointer of .pdr file */
EX_FILES ExPdrFile;

/*
 * Maximum character code we will use -- character with character
 * code greater than this value will not be written to the tfm and
 * the pdr file. The value of MaxCharCode must be either 127 or 255.
 */
int MaxCharCode;

/*
 * PfdDefaults
 * ***********
 * This procedure is called in the beginning, before a pfd
 * file is read in. It sets all variables to their defaults.
 */
void
PfdDefaults ()
{
  int i;

  /* Variables which can be set in the .pfd file */
  PfdFileName = NULL;
  OutputFileNameBase = NULL;
  BaseFontName = NULL;
  HRatio = 1.0;
  WidthDelta = 0.0;
  TiltAngle = 0.0;
  OutlineFactor = 0.0;
  SlitexInvisible = FALSE;
  MonoCase = 0.0;
  DownLoadable = FALSE;
  TeXMacroDefs = NULL;    
  EncodingClass = 0;
  for (i=0; i<23; i++)
    FontDimenArray[i] = -1;
  MaxCharCode = 255;

  /*
   * The following are a bunch of parameters, where the defaults
   * vary upon the font type. Again---these parameters can be
   * overwritten by commands in the .pfd file. We set them here to
   * negative numbers to indicate initial values (the defaults are
   * not known yet).
   */
  DesignSize = -1;
  Slant = -1;
  Stretch = -1;
  Shrink = -1;
  Space = -1;
  ExtraSpace = -1;
  Quad = -1;
  XHeight = -1;
}

/*
 * WritePdrFile
 * ************
 * Write the pdr file.
 *
 * tp: pointer to tfm data.
 */
void
WritePdrFile (tp)
     TFM_S_P tp;
{
  double	a, b, c, d, t_x, t_y;
  int i, count;

  if (Verbose > V_SOME)
    fprintf (stderr, "WritePdrFile(): Starting to write driver file \"%s\"\n",
	     ExPdrFile.ef_fn);

  if (MaxCharCode != 127 && MaxCharCode != 255)
    Fatal("WritePdrFile(): MaxCharCode wrong");

  FExOpen(&ExPdrFile, EFT_WRITE, EFQ_FILE_NAME_LOADED, NULL, NULL);

  WriteBytes (EX_FP(ExPdrFile), PDR_FILE_ID, 2); /* fid */
  WriteString (EX_FP(ExPdrFile), 1, Strlen(BaseFontName), BaseFontName); /* l, bna */
  WriteBytes (EX_FP(ExPdrFile), tp->tfm_cs, 4); /* checksum */
  WriteBytes (EX_FP(ExPdrFile), MaxCharCode, 2); /* hcc */
  WriteBytes (EX_FP(ExPdrFile), tp->tfm_designsize, 4);  /* ds */
  WriteBytes (EX_FP(ExPdrFile), tp->tfm_designsize/6, 4);  /* space */

  /* Compute the font matrix. */
  b = 0.0;
  t_x = 0.0;
  t_y = 0.0;
  a = 1.0;
  d = HRatio;
  /* Positive tilting is tilting to the right: see section 93, page 96,
   * of the PostScript Cookbook. */
  c = a * tan(TiltAngle/180.0 * PI);
  WriteBytes (EX_FP(ExPdrFile), FLOAT_TO_FIXES(a), 4); /* a */
  WriteBytes (EX_FP(ExPdrFile), FLOAT_TO_FIXES(b), 4);
  WriteBytes (EX_FP(ExPdrFile), FLOAT_TO_FIXES(c), 4);
  WriteBytes (EX_FP(ExPdrFile), FLOAT_TO_FIXES(d), 4);
  WriteBytes (EX_FP(ExPdrFile), FLOAT_TO_FIXES(t_x),4);
  WriteBytes (EX_FP(ExPdrFile), FLOAT_TO_FIXES(t_y),4); /* t_y */

  WriteBytes (EX_FP(ExPdrFile), FLOAT_TO_FIXES(OutlineFactor), 4); /* o */
  WriteBytes (EX_FP(ExPdrFile), 0, 4); /* mc (mono case), none right now */
  WriteBytes (EX_FP(ExPdrFile), Strlen(EmulateFontName)==0 ? 0:1, 1);
  WriteBytes (EX_FP(ExPdrFile), SlitexInvisible==FALSE? 0:1, 1);

  count = 0; /* Compute length of encoding array. */
  for (i=0; i<=MaxCharCode; i++) {
    if (AfmChar[i].c_used)
      count++;
  }

  /* Write the length out to this file. */
  WriteBytes (EX_FP(ExPdrFile), count, 1); /* lec */

  /* In a loop write the character information to the .pdr file. */
  for (i=0; i<=MaxCharCode; i++) {
    if (!AfmChar[i].c_used)
      continue;
#ifdef DEBUG
    fprintf (stderr, "%% WritePdrFile(): '%o, type = %d\n",
	     i, AfmChar[i].c_type);
#endif
    if (AfmChar[i].c_type == CT_NONE)
      Fatal3("WritePdrFile(): CT_ illegal, character '%o, type %d",
	     i, AfmChar[i].c_type);
    WriteBytes  (EX_FP(ExPdrFile), i, 1); /* Character code */
    WriteBytes  (EX_FP(ExPdrFile), AfmChar[i].c_type, 1); /* Type of character */
    if (AfmChar[i].c_type == CT_EXECPS) /* Class Execps char */
      WriteBytes (EX_FP(ExPdrFile), AfmChar[i].c_execps_type, 1);
    else
      WriteBytes (EX_FP(ExPdrFile), 0, 1);

    /* Tfm width of character. */
    WriteBytes (EX_FP(ExPdrFile), AfmChar[i].c_tfm, 4);

    /* Name of character or ExecPS procedure of character; nothing
     * if the character is a "width only" character. */
    WriteString (EX_FP(ExPdrFile), 2, Strlen(AfmChar[i].c_string), AfmChar[i].c_string);
  }

  FExClose (&ExPdrFile);

  if (Verbose > V_SOME)
    fprintf (stderr, "WritePdrFile(): Done with writing \"%s\"\n", ExPdrFile.ef_fn);
}
