/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/*
 * Code to write pl (property list) files for tfm files. Then we
 * execute pltotf to generate the tfm file.
 */

#include <stdio.h>
#include <sys/wait.h>
#include <sys/file.h>
#include "defs.h"
#include "pfd2tfm.h"
#include "char.h"
#include "extfil.h"

extern char * StrcpyAlloc();
extern int Verbose;

extern char * ProgName;

extern CAFM AfmChar[1];

extern int IsFixedPitch;
extern int XHeight;
extern double HRatio;
extern double WidthDelta;
extern int MaxCharCode;
extern char *BaseFontName;
extern char *bindir_pltotf_default;
extern char *tmpdir_default;
extern int Verbose;

/* The not default variables are initialized with a negative value.
 * Use defaults only, if values are negative. */
extern double DesignSize, DesignSizeDef;
extern int Slant, SlantDef;
extern int Stretch, StretchDef;
extern int Shrink, ShrinkDef;
extern int Space, SpaceDef;
extern int ExtraSpace, ExtraSpaceDef;
extern int Quad, QuadDef;
extern int XHeight, XHeightDef;
extern int FontDimenArray[1];

/* File file for writing the property list to. */
EX_FILES ExPlFile;

/*
 * WritePlTfmFile
 * **************
 * Property list writing for a TFM file. At the end pltotf is executed
 * and all together we get a TFM file.
 */
void
WritePlTfmFile(tfm_fn)
     char *tfm_fn;
{
  int	childpid;
  union	wait wait_status;
  char pltotf_file_name[STRINGLENGTH];

  int i;
  KPX_P kpxp;	
  LIG_P ligp;
  int right_index, lig_index;
  char buffer[256];

  /* Open a temporay file. */
  FExOpen(&ExPlFile, EFT_TEMP, EFQ_TEMP_MODE_2, NULL, "pl");
  if (Verbose > V_QUIET)
    printf ("WritePlTfmFile(): writing property list to \"%s\"\n", ExPlFile.ef_fn);
  
  /* Default value business */
  DesignSizeDef = 10.0;

  /*
   * What follows depends on the font: text fonts are
   * treated differently from the symbol font.
   */
  if (strcmp (BaseFontName, "Symbol") == 0) {
    SpaceDef = 0;
    SlantDef = 0;
    StretchDef = 0;
    ShrinkDef = 0;
    ExtraSpaceDef = 0;
    QuadDef = 0;
    XHeight = 0;
  } else {
    SpaceDef = AfmChar[32].c_wx;
    SlantDef = 0;
    if (IsFixedPitch) { /* Fixed pitch font */
      StretchDef = 0;
      ShrinkDef = 0;
      ExtraSpaceDef = SpaceDef; /* double spaces after the period */
      QuadDef = 0;
      XHeightDef = XHeight;
    } else {		/* Variable pitch font */
      StretchDef = AfmChar[32].c_wx/2;
      ShrinkDef =  AfmChar[32].c_wx/3;
      ExtraSpaceDef = AfmChar[32].c_wx/3;
      QuadDef = 1000;
      XHeightDef = XHeight;
    }
  }
  /* Now set all the defaults out, unless there have been values specified
   * by the user. */
  if (DesignSize < 0.0) DesignSize = DesignSizeDef;
  if (Space < 0) Space = SpaceDef;
  if (Slant < 0) Slant = SlantDef;
  if (Stretch < 0) Stretch = StretchDef;
  if (Shrink < 0)  Shrink = ShrinkDef;
  if (ExtraSpace < 0) ExtraSpace = ExtraSpaceDef;
  if (Quad < 0) Quad = QuadDef;
  if (XHeight < 0) XHeight = XHeightDef;

  /* Writing the propertly list starts here ! */
  fprintf (EX_FP(ExPlFile), "(COMMENT Property list generated by %s)\n", ProgName);

  fprintf (EX_FP(ExPlFile), "(COMMENT so what)\n");
  fprintf (EX_FP(ExPlFile), "(DESIGNSIZE R %5.3lf)\n", DesignSize);
  fprintf (EX_FP(ExPlFile), "(FONTDIMEN\n");
  fprintf (EX_FP(ExPlFile), "    (SLANT        R %5.3lf)\n", Slant/1000.0);
  fprintf (EX_FP(ExPlFile), "    (SPACE        R %5.3lf)\n", Space/1000.0);
  fprintf (EX_FP(ExPlFile), "    (STRETCH      R %5.3lf)\n", Stretch/1000.0);
  fprintf (EX_FP(ExPlFile), "    (SHRINK       R %5.3lf)\n", Shrink/1000.0);
  fprintf (EX_FP(ExPlFile), "    (XHEIGHT      R %5.3lf)\n", XHeight/1000.0);
  fprintf (EX_FP(ExPlFile), "    (QUAD         R %5.3lf)\n", Quad/1000.0);
  fprintf (EX_FP(ExPlFile), "    (EXTRASPACE   R %5.3lf)\n", ExtraSpace/1000.0);
  for (i=8; i<23; i++) {
    if (FontDimenArray[i] >= 0)
      fprintf (EX_FP(ExPlFile), "    (PARAMETER D %d R %5.3lf)\n",
	       i, FontDimenArray[i]/1000.0);
  }
  fprintf (EX_FP(ExPlFile), ")\n");

  /* Ligature and kerning table written here. */
  fprintf (EX_FP(ExPlFile), "(LIGTABLE\n");
  for (i=0; i<=MaxCharCode; i++) {
    if (!AfmChar[i].c_used)
      continue;
    kpxp = AfmChar[i].c_kern;
    ligp = AfmChar[i].c_lig;
    if ((ligp==NULL) && (kpxp==NULL))
      continue; /* neither ligature, nor kerning */
    
    if (Verbose > V_SOME)
      fprintf (stderr, "WritePlTfmFile(): Kern and lig for \"%s\" ['%o]\n",
	       AfmChar[i].c_string, i);
    
    fprintf (EX_FP(ExPlFile), "    (LABEL D %d)\n", i);

    /* Ligatures ? */
    while (ligp != NULL) {
      if (CharNameToIndex (ligp->l_cleft) != i)
	Fatal ("WritePlTfmFile(): inconsistency in lig stuff");
      right_index  = CharNameToIndex (ligp->l_cright);
      lig_index =    CharNameToIndex (ligp->l_lig_name);
      fprintf (EX_FP(ExPlFile), "    (LIG D %d D %d)\n", right_index, lig_index);
      ligp = ligp->l_next;
    }

    /* Kerning first */
    while (kpxp != NULL) {
      fprintf (EX_FP(ExPlFile), "    (KRN D %d R %6.5lf)\n",
	       CharNameToIndex(kpxp->k_cright), (double)kpxp->k_dist/1000.0);
      kpxp = kpxp->k_next;
    }
    
    /* Done with this character */
    fprintf (EX_FP(ExPlFile), "    (STOP)\n");
  }
  fprintf (EX_FP(ExPlFile), "    )\n");
  
  /* Write character sizes here: they are relative to the design size */
  for (i=0; i<=MaxCharCode; i++) {
    if (!AfmChar[i].c_used)
      continue;
    fprintf (EX_FP(ExPlFile), "(CHARACTER D %d\n", i);
    fprintf (EX_FP(ExPlFile), "    (CHARWD R %6.5lf)\n",
	     (double)AfmChar[i].c_wx/1000.0 + WidthDelta/1000.0);
    fprintf (EX_FP(ExPlFile), "    (CHARHT R %6.5lf)\n",
	     HRatio*AfmChar[i].c_ury/1000.0);
    fprintf (EX_FP(ExPlFile), "    (CHARDP R %6.5lf)\n",
	     -HRatio*AfmChar[i].c_lly/1000.0);
    fprintf (EX_FP(ExPlFile), "    )\n");
  }
  if (Verbose > V_QUIET)
    fprintf (stderr, "WritePlTfmFile(): Ligature and kerning table\
 done\n");
  
  /* Done: close file, fork off pltotf to generate .tfm file. */
  FExClose (&ExPlFile);

  if (Verbose > V_QUIET)
    fprintf (stderr, "WritePlTfmFile(): execute pltotf now (following output by pltotf)\n");
  if ((childpid=fork()) == 0)	{
    /* Child */
    sprintf (buffer, "%s-CHILD", ProgName);
    ProgName = StrcpyAlloc(buffer);
    InitProgName(ProgName, tmpdir_default);
    sprintf (pltotf_file_name, "%s/%s", bindir_pltotf_default, "pltotf");
    if (execl (pltotf_file_name, "pltotf", ExPlFile.ef_fn, tfm_fn, NULL) == -1) {
      perror ("pltotf");
      Fatal ("WritePlTfmFile() [child]: execl of pltotf failed.");
    }
    exit(0);
  }

  if (childpid == -1)
    Fatal ("WritePlTfmFile(): fork() of pltotf failed.");

  /* Parent: wait on completion of child. */
  if (wait(&wait_status) == -1)
    Fatal ("WritePlTfmFile(): parent wait() error no child.");

  if ((wait_status.w_status>>8) & 0177)
    Fatal ("WritePlTfmFile(): status of child != 0");

  if (Verbose > V_QUIET)
    fprintf (stderr, "WritePlTfmFile(): execute pltotf done.\n");

  FExClose (&ExPlFile); /* That really removes it. */
}
