/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/* main program of pfd2tfm */

#include <stdio.h>
#include <sys/file.h>
#include <strings.h>
#include "defs.h"
#include "pfd2tfm.h"
#include "char.h"
#include "tfm.h"
#include "extfil.h"

extern char * StrcpyAlloc();
extern EX_FILES ExPdrFile;
extern char *LocateFileWithPath();
extern char * getenv();
extern char * HandleFileNameExtension();
extern TFM_S_P ReadTfmFile();
extern void Done();
extern char * optarg;
extern int optind;
extern char * ps_def_encodings_dir;
extern char * tmpdir_default;

/* A bunch of file names etc. */
EX_FILES ExPfdFile; /* Pfd file. */
char * OutputFileNameBase; /* Basename (without extensions) of the TFM
			      and PDR file we generate. */
extern char * EmulateFontName; /* Name of font being emulated. */

EX_FILES ExAfmFile; /* Afm file */

/* BaseFontName parameter (or directly from AFM file name). No extension,
   of course. */
extern char *BaseFontName;

extern CAFM AfmChar[MAX_CHAR_PS];
extern char *afm_path_default;
extern char *texfonts_default;

extern char *MapFileName; /* If empty, then none is established */
extern char *HashLongToShort;

/* Verbose level is any number between 0 [quiet] and 2 [verbose] */
int Verbose = V_SOME;

/* Program name */
char *ProgName;

void Usage();

/*
 * main
 * ****
 */
main (argc, argv)
    int argc;
   char * argv[];
{
  int i, c;
  char tmp[256];
  char *tmp2;
  TFM_S_P tp; /* Pointer to tfm data structure */
  /* Are we reading in an AFM file directly, i.e., without a PFD file. */
  int afmdirect;
  char *tfm_fn; /* tfm file name */
  char ** p_ptr; /* Temporary */
  int index;

  ProgName = argv[0];
  InitProgName (ProgName, tmpdir_default);

  if (argc == 1) {
    Usage();
    exit(1);
  }

  /* Some typical initializations we have to do. */
  FExInit();
  InitSignalHandling();

  Report ("Starting");

  /* Options */
  while ((c=getopt(argc, argv, "qvm:p:")) != EOF) {
    switch (c) {
      /* -q: quiet. */
      case 'q':
        Verbose = V_QUIET;
	break;

      /* -v: verbose. */
      case 'v':
	Verbose = V_A_LOT;
	break;

      /* -m map-file-name: use the specfied file for mapping full
	 length AFM file names to short versions. */
      case 'm':
	SetUpFileNameMapping (optarg);
	break;

      /* -p directory: use specified directory instead of default to locate
	 PostScript default encodings. */
      case 'p':
	ps_def_encodings_dir = StrcpyAlloc(optarg); /* Overwrites default in "defaults.c" */
	break;
      case '?':
	Fatal ("main(): Illegal option.");
	break;

      default:
	Fatal ("main(): defaults.");
      } /* switch */
  } /* while */

  Report ("Done with options");

  LoadDefEVectors();

  if (optind == argc)
    Fatal ("main(): no PFD or AFM files provided.");

  /* In a loop go through all the PFD (or AFM) files */
  for (index=optind; index<argc; index++) {
    PfdDefaults(); /* Set up the defaults for PFD files */
    AfmInit(); /* AFM array initialisation */
    KernAndLigInit(); /* Kerning and Ligature table initialisations */
    InitializeFixFontWidths(); /* Fixing font widths in PFD file */
    EmulateInit(); /* Initialize font emulation */

    /* If an .afm file specified? If so, afmdirect is TRUE. */
    i = strlen(argv[index]) - strlen (".afm");
    if (i < 0) i = 0;
    afmdirect = (strcmp (argv[index]+i, ".afm") == 0);

    if (! afmdirect) {
      Report ("Processing based on a PFD file");
      /* Extension stuff, PFD file, strip it off. BaseFontName is specified
         inside the PDF file. */
      OutputFileNameBase = HandleFileNameExtension (0, argv[index], "pfd");
    } else { /* AFM file: prepend PS_FONTS_PREFIX */
      Report ("Processing based on AFM file.");
      BaseFontName = HandleFileNameExtension (0, argv[index], "afm");
      /* Compute the base name of the PFD and TFM file
	 we are going to generate. May have to shorten it (-m option) */
      if (Strlen(MapFileName) != 0) {
	p_ptr = (char **)LookUpKeyInHashTableE (HashLongToShort, BaseFontName,
						"main(): tried to map");
	sprintf (tmp, "%s%s", PS_FONTS_PREFIX, *p_ptr);
      } else {
	sprintf (tmp, "%s%s", PS_FONTS_PREFIX, BaseFontName);
      }
      OutputFileNameBase = StrcpyAlloc(tmp);
    }

    /* Remove any previously generate PDR and TFM files for this font
       if such files exist. */
    sprintf (tmp, "%s.%s", OutputFileNameBase, "tfm");
    UnLinkIfFileExists(tmp);

    sprintf (tmp, "%s.%s", OutputFileNameBase, "pdr");
    ExPdrFile.ef_fn = StrcpyAlloc(tmp);
    UnLinkIfFileExists(ExPdrFile.ef_fn);

    if (! afmdirect) {
      Report ("Reading PFD file");

      /* Open the PFD file, set up lex, start yacc to read it in. */
      FExOpen(&ExPfdFile, EFT_READ, 0, OutputFileNameBase, "pfd");

      /* Get lexical analyser for reading PFD file ready, start
       * yacc on the PFD file */
      SetupLex (&ExPfdFile, PFD_LEX, FALSE);
      if (YYparse_pfd() != 0)
	Fatal2 ("main(): error parsing PFD file \"%s.pfd\"", OutputFileNameBase);
      FExClose (&ExPfdFile);
      Report ("PFD file read in");

      if (Strlen(BaseFontName) == 0)
	Fatal2 ("main(): PFD file \"%s.pfd\" misses BaseFontName specification",
		OutputFileNameBase);
    }
    InitEncodingClass(BaseFontName);
	
    /* Is a font being emulated, or is this a regular PS font? */
    if (Strlen(EmulateFontName) == 0) {
      /* No emulation. Read AFM file. */
      Report ("Reading AFM file now");
      if (Strlen(MapFileName) != 0) {
	p_ptr = (char **) LookUpKeyInHashTableE (HashLongToShort, BaseFontName,
				     "main(): tried to map to find AFM file");
	sprintf (tmp, "%s.afm", *p_ptr);
      } else {
	sprintf (tmp, "%s.afm", BaseFontName);
      }
      /* Locate afm file. */
      if ((tmp2 = LocateFileWithPath(tmp, getenv("AFMPATH"), afm_path_default, Verbose > V_SOME)) == NULL)
	Fatal2 ("main(): could not locate \"%s\"", tmp);
      FExOpen(&ExAfmFile, EFT_READ, 0, tmp2, "afm");
      /* Set up lexical analyzer, and start parser. */
      SetupLex (&ExAfmFile, AFM_LEX, FALSE);
      if (yyparse_afm() != 0)
	Fatal2 ("main(): Parsing \"%s\" failed", ExAfmFile.ef_fn);
      FExClose (&ExAfmFile);

      /* Build now the linked lists, which contain the ligature
       * and kerning information for each character. */
      Report ("Handling kerning now");
      BuildKernStructure(); /* Build kerning structure .*/
      Report ("Handling ligatures now");
      BuildLigatureStructure (); /* Build the ligature structures now. */
      Report ("Fixing widths now");
      FixFontWidths(); /* Incorporate font width changes from PFD file */

      Report ("Building TFM file now");
      sprintf (tmp, "%s.tfm", OutputFileNameBase);
      tfm_fn = StrcpyAlloc(tmp);
      WritePlTfmFile(tfm_fn);     /* Write .pl file, generate TFM file using pltotf. */
      tp = ReadTfmFile(tfm_fn); /* Read TFM file to get the widths. */
      EnterTfmDataIntoAfmCharArray (tp); /* Enter data into afm array */
      Report ("Write PDR file now");
      WritePdrFile(tp); /* Write PDR file */
      Report ("Write TeX macros if enabled");
      WriteTeXMacros(); /* Write tex macros for character access. */
      ReportUsageDefCodes();
      Report ("Done");
    } else {
      /* Emulating font: the font metric information is imported
       * from another TFM file. */
      Report ("Emulating font business");
      /* Read afm file. */
      Report ("Read AFM file");
      sprintf (tmp, "%s.afm", BaseFontName);
      if ((tmp2 = LocateFileWithPath (tmp, getenv("AFMPATH"), afm_path_default, Verbose > V_SOME)) == NULL)
	Fatal2 ("main(): could not locate \"%s\"", tmp);
      FExOpen(&ExAfmFile, EFT_READ, 0, tmp2, "afm");
      /* Now use lexical analyzer and parser to read in .afm file. */
      SetupLex (&ExAfmFile, AFM_LEX, FALSE);
      yyparse_afm();
      FExClose (&ExAfmFile);

      /* Find TFM file of font being emulated. */
      Report ("Look at TFM file of font being emulated");
      sprintf (tmp, "%s.tfm", EmulateFontName);
      if ((tmp2 = LocateFileWithPath(tmp, getenv("TEXFONTS"), texfonts_default, Verbose > V_SOME)) == NULL)
	Fatal2("main(): Emulating fonts: could not locate \"%s\"", tmp);

      /* This makes a duplicate of the TFM file being emulated
       * under the name of the new font we compute here. */
      sprintf (tmp, "%s.tfm", OutputFileNameBase);
      tfm_fn = StrcpyAlloc(tmp);
      FileCopy(tmp2, tfm_fn);
      
      /* Read in tfm file to replace the width information in the character
	 array. */
      tp = ReadTfmFile(tfm_fn);
      EnterTfmDataIntoAfmCharArray (tp);

      Report ("Write PDR file");
      WritePdrFile(tp);	/* Write the PDR file */
      WriteTeXMacros();
    } /* if: emulate font */

    if (++index != argc)
      Report ("NEXT FONT");

  } /* while */

  Done();
}

