/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/*
 * Tpic support routines
 *
 *      Tim Morgan, UC Irvine
 *	Fletcher Mattox, fletcher@sally.utexas.edu, (adapted to dvi2ps)
 *      Stephan Bechtolsheim made some changes (more to conform to
 *      the conventions in this driver than anything else).
 * \special{pn pensize}			Tpic_set_pen_size(pensize)
 * \special{tx count pattern}		not implemented
 * \special{pa x y}			Tpic_add_path(x, y)
 * \special{fp}				Tpic_flush_path()
 * \special{dt length}			Tpic_flush_dashed(length, 1)
 * \special{da length}			Tpic_flush_dashed(length, 0)
 * \special{ar x y xrad yrad start end} arc(x, y, xrad, yrad, start, end)
 * \special{sh}				Tpic_shade_last(grey)
 * \special{wh}				Tpic_shade_last(white)
 * \special{bl}				Tpic_shade_last(black)
 *
 * At the time these routines are called, the values of hhh and vvv should
 * have been updated to the upper left corner of the graph (the position
 * the \special appears at in the dvi file).  Then the coordinates in the
 * graphics commands are in terms of a virtual page with axes oriented:
 *
 *                      0,0
 *                       +-----------> +x
 *                       |
 *                       |
 *                       |
 *                      \ /
 *                       +y
 *
 * Angles are measured in the conventional way, from +x towards +y.
 * Unfortunately, that reverses the meaning of "counterclockwise"
 * from what it's normally thought of.
 */

#include <stdio.h>
#include <math.h>
#include "defs.h"
#include "dvitps.h"
#include "emit.h"

#define	MAXPOINTS	300	/* Max points in a path */

extern int IntFromLex();
extern double DoubleFromLex();
extern int PssColumn;

static int xx[MAXPOINTS], yy[MAXPOINTS]; /* Path in milli-inches,
					    absolute coordinates. */
static int path_len = 0;	/* # points in current path */
int pen_size = 1;		/* Pixel width of lines drawn */

extern int DriverMag; /* Magnification used for this dvi file */
extern int Resolution;	/* Pixels per Inch */

/*
 * These two macros scale from milli-inches to pixel coords, but do
 * not apply the necessary offsets for absolute display positioning.
 * Because of overflow problems has to be done with "double".
 */
#define	xsc(x)	(int) (((x)*(double)Resolution*(double)DriverMag/1000.0\
			+ 500.0) / 1000)
#define	ysc(y)	(int) (((y)*(double)Resolution*(double)DriverMag/1000.0\
			+ 500.0) / 1000)

/*
 *  These two macros scale from milli-inches to pixel coords, and add
 *  the necessary offsets for absolute device positions.
 */
extern int  		hhh, vvv;	/* device coordinates in pixels */
#define	xconv(x)	(xsc(x) + hhh)
#define	yconv(y)	(ysc(y) + vvv)

int shade = FALSE;		/* do we shade the last object? */

extern int IntFromLex();
extern double FloatFromLex();

/*
 * Tpic_shade_last
 * ***************
 * Expects a `color' between 0 (black) and 1 (white).
 * it saves the current color, (probably black), which is then
 * restored in Tpic_shade_it();
 */
void
Tpic_shade_last(color)
     double color;
{
  char buffer[256];

  if (color < 0.0 || color > 1.0) {
    Warning2 ("Tpic_shade_last(): Illegal shade %f", color);
    return;
  }
  shade = TRUE;

  PssSendCommand ("currentgray");		/* save it */
  sprintf (buffer, "%4.2lf ", color);
  PssSendCommand (buffer);
  PssSendCommand ("setgray");
}

void
Tpic_shade_it()
{
  if (!shade) {
    PssSendCommand("stroke");
  } 
  else {
    PssSendCommand("fill");
    PssSendCommand("setgray");	/* restore it */
    shade = FALSE;
  }
}

/*
 *  Tpic \specials must be careful not to forget what the current state
 *  is, so that it can restore it later on.  I think `current state'
 *  == `current point'.  The driver expects an rmoveto to work after
 *  a \special, so the current point must be saved (at least).
 */
void
Tpic_save_state()
{
  PssSendCommand("currentpoint");
}

/*
 *  Pop the currentpoint off the stack.  Note that Tpic_save_state()
 *  and Tpic_restore_state() must come in pairs, or we are in big trouble.
 */
void
Tpic_restore_state()
{
  PssSendCommand("moveto");
}

/*
 *  Set the size of the virtual pen used to draw in milli-inches
 */
void 
Tpic_set_pen_size()
{
  int ps;
  
  ps = IntFromLex();
  pen_size = xsc(ps);
  PssSendInteger (pen_size);
  PssSendCommand ("setlinewidth");
}


/*
 *  Print the line defined by previous path commands
 */
void 
Tpic_flush_path()
{
  register int i;
  int k;

#ifdef ABC
  for (;;) {
    printf ("Give x: ");
    scanf ("%d", &k);
    printf (":%d:\n", xconv(k));
  }
#endif

#ifdef TESTING
  EMIT_NEW_LINE;
#endif
  Tpic_save_state();
  PssSendCommand("newpath");
  PssSendInteger (xconv(xx[1]));
  PssSendInteger (yconv(yy[1]));
  PssSendCommand ("moveto");

  for (i=1; i<path_len; i++)  {
#ifdef TESTING
    EMIT_NEW_LINE;
#endif
    PssSendInteger(xconv(xx[i+1]));
    PssSendInteger(yconv(yy[i+1]));
    PssSendCommand("lineto");
  }
  Tpic_shade_it();
  Tpic_restore_state();
#ifdef TESTING
  EMIT_NEW_LINE;
#endif
  path_len = 0;
}

/*
 *  Draw a dot -- the call to Tpic_dot_at() must be protected
 *  by a Tpic_save_state().
 */
Tpic_dot_at(x, y)
     int x,y;
{
  int radius = 1;		/* in pixels */

  PssSendCommand ("newpath");
  PssSendInteger(xconv(x));
  PssSendInteger(yconv(y));
  PssSendInteger(radius);
  PssSendInteger(0);
  PssSendInteger(360);
  PssSendCommand("arc");
  PssSendCommand("fill");
}

/*
 * Print a dashed line along the previously defined path, with
 * the dashes/inch defined.
 */
void 
Tpic_flush_dashed(dotted)
     int dotted;
{
  int i, numdots, x0, y0, x1, y1;
  int cx0, cy0;
  double inchesperdash;
  double d, spacesize, a, dx, dy, milliperdash;

  inchesperdash = DoubleFromLex();
  if (path_len <= 1 || inchesperdash <= 0.0) {
    Warning("Illegal conditions for dotted/dashed line");
    return;
  }
  milliperdash = inchesperdash * 1000.0;
  x0 = xx[1];	
  y0 = yy[1];
  x1 = xx[2];	
  y1 = yy[2];
  dx = x1 - x0;
  dy = y1 - y0;
  if (dotted) {
    numdots = sqrt(dx*dx + dy*dy) / milliperdash + 0.5;
    Tpic_save_state();
    for (i=0; i <= numdots; i++) {
      a = (double) i / (double) numdots;
      cx0 = x0 + a*dx + 0.5;
      cy0 = y0 + a*dy + 0.5;
      Tpic_dot_at(cx0, cy0);
    }
    Tpic_restore_state();
  }
  else {
    d = sqrt(dx*dx + dy*dy);
    numdots = d / (2.0*milliperdash) + 1.0;
    if (numdots == 1)
      spacesize = 0;
    else
      spacesize = (d - numdots * milliperdash) / (numdots - 1);
    PssSendCommand ("[");
    PssSendInteger (xsc(milliperdash));
    PssSendInteger (xsc(spacesize));
    PssSendCommand ("] 0 setdash");
    Tpic_flush_path();
    PssSendCommand("[] 0 setdash");
  }
  path_len = 0;
}

/*
 *  Add a point to the current path
 */
void 
Tpic_add_path()
{
  if (++path_len >= MAXPOINTS)
    Fatal("Tpic_add_path(): Too many points.");
  xx[path_len] = IntFromLex();
  yy[path_len] = IntFromLex();
}

/*
 * Draw an arc
 */
void 
Tpic_arc()
{
  int xc, yc, xrad, yrad;
  double start_angle, end_angle;
  char buffer[256];

  xc = IntFromLex();
  yc = IntFromLex();
  xrad = IntFromLex();
  yrad = IntFromLex();
  start_angle = DoubleFromLex();
  end_angle = DoubleFromLex();
  while (start_angle < 0.0)  start_angle += TWOPI;
  while (end_angle < 0.0)  end_angle += TWOPI;
  while (start_angle > TWOPI)  start_angle -= TWOPI;
  while (end_angle > TWOPI)  end_angle -= TWOPI;
  start_angle *= 360.0/TWOPI;
  end_angle *= 360.0/TWOPI;

  Tpic_save_state();
  PssSendCommand("newpath");
  PssSendInteger(xconv(xc));
  PssSendInteger(yconv(yc));
  PssSendInteger(xsc(xrad));
  PssSendInteger(ysc(yrad));
  /* To use PssSendCommand for the following is a little bit of
     cheating but it works. */
  sprintf (buffer, "%5.2lf", start_angle);
  PssSendCommand (buffer);
  sprintf (buffer, "%5.2lf", end_angle);
  PssSendCommand (buffer);
  PssSendCommand ("@ellipse");

  Tpic_shade_it();
  Tpic_restore_state();
}

/*
 *  Draw a Chaikin spline along the previously defined path
 *  @ChaikinSplineDraw strokes the path, so we don't do it here.
 */
void 
Tpic_flush_spline()
{
  int i;
  
  Tpic_save_state();
  for (i=1; i<=path_len; i++) {
    PssSendInteger(xconv(xx[i]));
    PssSendInteger(yconv(yy[i]));
  }
  PssSendInteger(path_len);
  PssSendCommand ("@ChaikinSplineDraw");
  Tpic_restore_state();
  path_len = 0;
}
