/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/* Contains routines which form the font names used to locate font files.
   It really belongs to "search.c", but to simplify modifications
   of this software the routines below are stored in a separate file. */

#include <stdio.h>
#include <strings.h>
#include "units.h"
#include "dvitps.h"
#include "extfil.h"
#include "fontd.h"

extern int Resolution;
extern int DriverMag;

/* Try this font name. */
char TryThisFontFileName[256];

/* On some machines font files with the same extension are collected
   in the same directory. In case this is so you have to change
   the code below in such a way that this string is loaded properly.
   The default is to set is simply an empty string. If set it
   must contain a trailing '/'. */
char TryThisFontFileDir[256];

/*
 * SetFontFileName
 * ***************
 * Construct various font file names (we don't know the type yet)
 * and leave the result in "TryThisFontFileName".
 * Remember for the following, that the font magnification already
 * incorporates the global magnification, when it was set!
 *
 * control: CF_.... (font file type)
 * f_mag: font magnification
 * f_n: font name;
 */
void
SetFontFileName (control, f_mag, f_n)
     int control;
     double f_mag;
     char *f_n;
{
  int i;
  double f;

  /* f = 1.0 if no global and font magnification used. */
  f = f_mag * (DriverMag/1000.0);

  switch (control) {
    case CF_PXL:
      /* The basis of the file name extension of a pxl file is a
       * printer of 200 dots/inch---the numerical extension then is
       * the magnification to be used. A document magnified by 1500
       * on a 300 dots/inch printer comes out at the original size.
       * The following formula will therefore evaluate to 1500 in case
       * no global or font magnification is used. */
      f *= 1000 * Resolution / 200.0;
      i = PxlCluge(f);
      sprintf (TryThisFontFileName, "%s.%dpxl", f_n, i);
      sprintf (TryThisFontFileDir, "%s", "");
      break;
    case CF_PK:
    /* For pk and gf files, the resolution of the printer is used as
     * numerical extension, in case there is no magnification going on */
      f *= Resolution;
      i = GfPkCluge(f);
      sprintf (TryThisFontFileName, "%s.%dpk", f_n, i);
      sprintf (TryThisFontFileDir, "%s", "");
      break;
    case CF_GF:
      f *= Resolution;
      i = GfPkCluge(f);
      sprintf (TryThisFontFileName, "%s.%dgf", f_n, i);
      sprintf (TryThisFontFileDir, "%s", "");
      break;
    case CF_PDR:
      /* pdr files are unaffected by any magnification. */
      sprintf (TryThisFontFileName, "%s.pdr", f_n);
      sprintf (TryThisFontFileDir, "%s", "");
      break;
    default:
      Fatal ("SetFontFileName(): illegal CF_");
    } /* switch */
#ifdef DEBUG
    fprintf (stderr, "%% SetFontFileName(): \"%s\"\n", TryThisFontFileName);
    fprintf (stderr, "%%\tDir: \"%s\"\n", TryThisFontFileDir);
#endif
}
