/*
 * dvipage: DVI Previewer Program for Suns
 *
 * Neil Hunt (hunt@spar.slb.com)
 *
 * This program is based, in part, upon the program dvisun,
 * distributed by the UnixTeX group, extensively modified by
 * Neil Hunt at the Schlumberger Palo Alto Research Laboratories
 * of Schlumberger Technologies, Inc.
 *
 * Copyright (c) 1988 Schlumberger Technologies, Inc 1988.
 * Anyone can use this software in any manner they choose,
 * including modification and redistribution, provided they make
 * no charge for it, and these conditions remain unchanged.
 *
 * This program is distributed as is, with all faults (if any), and
 * without any warranty. No author or distributor accepts responsibility
 * to anyone for the consequences of using it, or for whether it serves any
 * particular purpose at all, or any other reason.
 *
 * $Log:	message.c,v $
 * Revision 1.2  88/12/07  18:48:27  hunt
 * Fixed typo left from previous mods.
 * 
 * Revision 1.1  88/11/28  18:41:13  hunt
 * Initial revision
 * 
 * Stripped from dvipage.c 1.4.
 */

#include <stdio.h>
#include <strings.h>
#include <varargs.h>
#include <sys/param.h>		/* For MAXPATHLEN */
#include <suntool/sunview.h>
#include <suntool/canvas.h>
#include <suntool/panel.h>
#include "dvipage.h"

/*
 * Forward functions.
 * =================
 */

forward void		mess_done();

/*
 * These are here rather than in dvipage.h because
 * they cause problems unless a whole raft of other includes
 * go in front.
 */
extern Frame		disp_frame;
extern Canvas		disp_canvas;

Frame			mess_frame;		/* Frame for message window. */
Panel			mess_panel;		/* Panel for message window. */

static int		errors_displayed = 0;

/*
 * message:
 *	Pops up a message window (if one is not already displayed)
 *	Formats a message (using fmt and args) and displays it into the window.
 *	Repeated calls to this function result in multiple lines of messages.
 */

void
message(fmt, va_alist)
char *fmt;
va_dcl
{
	va_list args;
	static char string[100];

	if(silent)
		return;

	va_start(args);
	vsprintf(string, fmt, args);
	va_end(args);

	/*
	 * If there are no errors displayed,
	 * then build a window..
	 */
	if(errors_displayed == 0)
	{
		mess_frame = window_create(disp_frame, FRAME,
		  FRAME_NO_CONFIRM, TRUE,
		  0);
		mess_panel = window_create(mess_frame, PANEL,
		  0);
		panel_create_item(mess_panel, PANEL_BUTTON,
		  PANEL_LABEL_IMAGE,
		    panel_button_image(mess_panel, "OK", 0, 0),
		  PANEL_ITEM_X, ATTR_COL(1),
		  PANEL_ITEM_Y, ATTR_ROW(0),
		  PANEL_NOTIFY_PROC, mess_done,
		  0);
	}
	else
		window_set(mess_frame,
		  WIN_SHOW, FALSE,
		  0);

	panel_create_item(mess_panel, PANEL_MESSAGE,
	  PANEL_LABEL_STRING, string,
	  PANEL_ITEM_X, ATTR_COL(1),
	  PANEL_ITEM_Y, ATTR_ROW(++errors_displayed)+5,
	  0);

	window_fit(mess_panel);
	window_fit(mess_frame);

	window_set(mess_frame,
	  WIN_SHOW, TRUE,
	  0);
}

/*
 * mess_done:
 *	This function is called when the OK button is pressed on the message
 *	window; the window is closed, and displayed messages are cleared.
 */

void
mess_done()
{
	window_destroy(mess_frame);

	errors_displayed = 0;
}

/*
 * Pop up prompts:
 * ==============
 */

static bool				prompt_return_value = TRUE;

static Pixrect				*ok_button = NULL;
static Pixrect				*abort_button = NULL;

static Panel_setting			prompt_ok_proc();
static Panel_setting			prompt_abort_proc();
static Panel_setting			prompt_notify_proc();

/*
 * strings_prompt: eg.
 *	strings_prompt(1152/2, 900/2,
 *	  "Directory: ", &dir[0],
 *	  "Filename: ", &fname[0],
 *	  0);
 */

#define MAX_STRING_ITEMS	10

bool
strings_prompt(x, y, va_alist)
int x, y;
va_dcl
{
	Frame frame;
	Panel panel;
	Panel_item string_item[MAX_STRING_ITEMS], ok_item, abort_item;
	Event *event;
	int w, h;
	int i;
	char *prompt;
	char *value;
	va_list ap;

	/*
	 * Create the frame and panel.
	 */
	frame = window_create(NULL, FRAME,
	  FRAME_SHOW_LABEL, FALSE,
	  FRAME_NO_CONFIRM, TRUE,
	  0);

	panel = window_create(frame, PANEL,
	  PANEL_ITEM_X_GAP, 1000, /* Enforces vertical layout */
	  0);

	va_start(ap);
	{
		for(i = 0; i < MAX_STRING_ITEMS; i++)
		{
			/*
			 * Stop if no more strings.
			 */
			if((prompt = va_arg(ap, char *)) == NULL ||
			   (value = va_arg(ap, char *)) == NULL)
				break;

			/*
			 * Create a text item for the string.
			 */
			string_item[i] = panel_create_item(panel, PANEL_TEXT,
			  PANEL_LABEL_STRING, prompt,
			  PANEL_VALUE, value,
			  PANEL_VALUE_DISPLAY_LENGTH, 20,
			  PANEL_VALUE_STORED_LENGTH, 132,
			  PANEL_NOTIFY_LEVEL, PANEL_ALL,
			  PANEL_NOTIFY_PROC, prompt_notify_proc,
			  0);
		}
	}
	va_end(ap);

	/*
	 * Create an OK button.
	 */
	if(ok_button == NULL)
		ok_button = panel_button_image(panel, "OK", 0, 0);

	ok_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, ok_button,
	  PANEL_NOTIFY_PROC, prompt_ok_proc,
	  0);

	/*
	 * Create an abort button.
	 */
	if(abort_button == NULL)
		abort_button = panel_button_image(panel, "Abort", 0, 0);

	abort_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, abort_button,
	  PANEL_NOTIFY_PROC, prompt_abort_proc,
	  0);

	/*
	 * Make them windows fit.
	 */
	window_fit(panel);
	window_fit(frame);

	/*
	 * Centre the prompt
	 */
	w = (int)window_get(frame, WIN_WIDTH);
	h = (int)window_get(frame, WIN_HEIGHT);
	x -= w/2;
	y -= h/2;
	window_set(frame,
	  WIN_X, x,
	  WIN_Y, y,
	  0);

	/*
	 * Set the flag to TRUE to indicate not aborted.
	 */
	prompt_return_value = TRUE;

	/*
	 * Display and wait for done.
	 */
	(void)window_loop(frame);

	/*
	 * If OK, then retrieve the window values and store them.
	 */
	if(prompt_return_value)
	{
		/*
		 * Loop through the items copying back values.
		 */
		va_start(ap);
		{
			for(i = 0; i < MAX_STRING_ITEMS; i++)
			{
				/*
				 * Stop if no more strings.
				 */
				if((prompt = va_arg(ap, char *)) == NULL ||
				   (value = va_arg(ap, char *)) == NULL)
					break;

				/*
				 * Get the value of this item.
				 */
				strcpy(value, panel_get_value(string_item[i]));
			}
		}
		va_end(ap);
	}

	/*
	 * Destroy the window.
	 */
	window_destroy(frame);

	return prompt_return_value;
}

/*
 * integers_prompt: eg.
 *	integers_prompt(1152/2, 900/2,
 *	  "Width: ", &w,
 *	  "Height: ", &h,
 *	  0);
 */

#define MAX_INTEGER_ITEMS	MAX_STRING_ITEMS

bool
integers_prompt(x, y, va_alist)
int x, y;
va_dcl
{
	Frame frame;
	Panel panel;
	Panel_item integer_item[MAX_INTEGER_ITEMS], ok_item, abort_item;
	Event *event;
	int w, h;
	int i;
	char *prompt;
	int *value;
	va_list ap;
	char value_string[20];

	/*
	 * Create the frame and panel.
	 */
	frame = window_create(NULL, FRAME,
	  FRAME_SHOW_LABEL, FALSE,
	  FRAME_NO_CONFIRM, TRUE,
	  0);

	panel = window_create(frame, PANEL,
	  PANEL_ITEM_X_GAP, 1000, /* Enforces vertical layout */
	  0);

	va_start(ap);
	{
		for(i = 0; i < MAX_INTEGER_ITEMS; i++)
		{
			/*
			 * Stop if no more strings.
			 */
			if((prompt = va_arg(ap, char *)) == NULL ||
			   (value = va_arg(ap, int *)) == NULL)
				break;

			/*
			 * Create a text item for the string.
			 */
			sprintf(value_string, "%d", *value);
			integer_item[i] = panel_create_item(panel, PANEL_TEXT,
			  PANEL_LABEL_STRING, prompt,
			  PANEL_VALUE, value_string,
			  PANEL_VALUE_DISPLAY_LENGTH, 20,
			  PANEL_VALUE_STORED_LENGTH, 20,
			  PANEL_NOTIFY_STRING, "\033",
			  PANEL_NOTIFY_LEVEL, PANEL_ALL,
			  PANEL_NOTIFY_PROC, prompt_notify_proc,
			  0);

		}
	}
	va_end(ap);

	/*
	 * Create an OK button.
	 */
	if(ok_button == NULL)
		ok_button = panel_button_image(panel, "OK", 0, 0);

	ok_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, ok_button,
	  PANEL_NOTIFY_PROC, prompt_ok_proc,
	  0);

	/*
	 * Create an abort button.
	 */
	if(abort_button == NULL)
		abort_button = panel_button_image(panel, "Abort", 0, 0);

	abort_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, abort_button,
	  PANEL_NOTIFY_PROC, prompt_abort_proc,
	  0);

	/*
	 * Make them windows fit.
	 */
	window_fit(panel);
	window_fit(frame);

	/*
	 * Centre the prompt
	 */
	w = (int)window_get(frame, WIN_WIDTH);
	h = (int)window_get(frame, WIN_HEIGHT);
	x -= w/2;
	y -= h/2;
	window_set(frame,
	  WIN_X, x,
	  WIN_Y, y,
	  0);

	/*
	 * Set the flag to TRUE to indicate not aborted.
	 */
	prompt_return_value = TRUE;

	/*
	 * Display and wait for done.
	 */
	(void)window_loop(frame);

	/*
	 * If OK, then retrieve the window values and store them.
	 */
	if(prompt_return_value)
	{
		/*
		 * Loop through the items copying back values.
		 */
		va_start(ap);
		{
			for(i = 0; i < MAX_INTEGER_ITEMS; i++)
			{
				/*
				 * Stop if no more strings.
				 */
				if((prompt = va_arg(ap, char *)) == NULL ||
				   (value = va_arg(ap, int *)) == NULL)
					break;

				/*
				 * Get the value of this item.
				 */
				*value = atoi(panel_get_value(integer_item[i]));
			}
		}
		va_end(ap);
	}

	/*
	 * Destroy the window.
	 */
	window_destroy(frame);

	return prompt_return_value;
}

/*
 * doubles_prompt: eg.
 *	doubles_prompt(1152/2, 900/2,
 *	  "Width: ", &w,
 *	  "Height: ", &h,
 *	  0);
 */

#define MAX_DOUBLE_ITEMS	MAX_INTEGER_ITEMS

bool
doubles_prompt(x, y, va_alist)
int x, y;
va_dcl
{
	Frame frame;
	Panel panel;
	Panel_item double_item[MAX_DOUBLE_ITEMS], ok_item, abort_item;
	Event *event;
	int w, h;
	int i;
	char *prompt;
	double *value;
	va_list ap;
	char value_string[20];

	/*
	 * Create the frame and panel.
	 */
	frame = window_create(NULL, FRAME,
	  FRAME_SHOW_LABEL, FALSE,
	  FRAME_NO_CONFIRM, TRUE,
	  0);

	panel = window_create(frame, PANEL,
	  PANEL_ITEM_X_GAP, 1000, /* Enforces vertical layout */
	  0);

	va_start(ap);
	{
		for(i = 0; i < MAX_DOUBLE_ITEMS; i++)
		{
			/*
			 * Stop if no more strings.
			 */
			if((prompt = va_arg(ap, char *)) == NULL ||
			   (value = va_arg(ap, double *)) == NULL)
				break;

			/*
			 * Create a text item for the string.
			 */
			sprintf(value_string, "%g", *value);
			double_item[i] = panel_create_item(panel, PANEL_TEXT,
			  PANEL_LABEL_STRING, prompt,
			  PANEL_VALUE, value_string,
			  PANEL_VALUE_DISPLAY_LENGTH, 20,
			  PANEL_VALUE_STORED_LENGTH, 40,
			  PANEL_NOTIFY_STRING, "\033",
			  PANEL_NOTIFY_LEVEL, PANEL_ALL,
			  PANEL_NOTIFY_PROC, prompt_notify_proc,
			  0);
		}
	}
	va_end(ap);

	/*
	 * Create an OK button.
	 */
	if(ok_button == NULL)
		ok_button = panel_button_image(panel, "OK", 0, 0);

	ok_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, ok_button,
	  PANEL_NOTIFY_PROC, prompt_ok_proc,
	  0);

	/*
	 * Create an abort button.
	 */
	if(abort_button == NULL)
		abort_button = panel_button_image(panel, "Abort", 0, 0);

	abort_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, abort_button,
	  PANEL_NOTIFY_PROC, prompt_abort_proc,
	  0);

	/*
	 * Make them windows fit.
	 */
	window_fit(panel);
	window_fit(frame);

	/*
	 * Centre the prompt
	 */
	w = (int)window_get(frame, WIN_WIDTH);
	h = (int)window_get(frame, WIN_HEIGHT);
	x -= w/2;
	y -= h/2;
	window_set(frame,
	  WIN_X, x,
	  WIN_Y, y,
	  0);

	/*
	 * Set the flag to FALSE to indicate not aborted.
	 */
	prompt_return_value = FALSE;

	/*
	 * Display and wait for done.
	 */
	(void)window_loop(frame);

	/*
	 * If OK, then retrieve the window values and store them.
	 */
	if(prompt_return_value)
	{
		/*
		 * Loop through the items copying back values.
		 */
		va_start(ap);
		{
			for(i = 0; i < MAX_DOUBLE_ITEMS; i++)
			{
				/*
				 * Stop if no more strings.
				 */
				if((prompt = va_arg(ap, char *)) == NULL ||
				   (value = va_arg(ap, double *)) == NULL)
					break;

				/*
				 * Get the value of this item.
				 */
				*value = atof(panel_get_value(double_item[i]));
			}
		}
		va_end(ap);
	}

	/*
	 * Destroy the window.
	 */
	window_destroy(frame);

	return prompt_return_value;
}

/*
 * prompt_notify_proc:
 */

static Panel_setting
prompt_notify_proc(item, event)
Panel_item item;
Event *event;
{
	if(event_id(event) == ESC)
	{
		prompt_return_value = TRUE;
		window_return(NULL);
		return PANEL_NONE;
	}
	else if(event_id(event) == Control('C'))
	{
		prompt_return_value = FALSE;
		window_return(NULL);
		return PANEL_NONE;
	}
	else
		return panel_text_notify(item, event);
}

/*
 * prompt_ok_proc:
 *	Normal return from string prompt.
 */

static Panel_setting
prompt_ok_proc(item, event) 
Panel_item item; 
Event *event;
{ 
	prompt_return_value = TRUE;

	window_return(NULL);

	return PANEL_NONE;
}

/*
 * prompt_abort_proc:
 *	Return from prompt with null string.
 */

static Panel_setting
prompt_abort_proc(item, event)
Panel_item item;
Event *event;
{
	prompt_return_value = FALSE;

	window_return(NULL);

	return PANEL_NONE;
}
