/*
 * DVI previewer for X.
 *
 * Eric Cooper, CMU, September 1985.
 *
 * Code derived from dvi-imagen.c.
 *
 * Modification history:
 * 1/1986	Modified for X.10 by Bob Scheifler, MIT LCS.
 * 7/1988	Modified for X.11 by Mark Eichin, MIT
 * 12/1988	Added 'R' option, toolkit, magnifying glass
 *			--Paul Vojta, UC Berkeley.
 * 2/1989	Added tpic support	--Jeffrey Lee, U of Toronto
 *
 *	Compilation options:
 *	X10	compile for X10
 *	MSBITFIRST	store bitmaps internally in with significant bit first
 *	BMSHORT	store bitmaps in shorts instead of bytes
 *	BMLONG	store bitmaps in longs instead of bytes
 *	MAXCHARS set to 256 for 256 character fonts (default = 128)
 */
#ifndef lint
#include "patchlevel.h"
static	struct {char	a[36], b, c;}
#ifndef X10
	dv_c = {"$Header: xdvi.c (X11), patchlevel = ", '0' + PATCHLEVEL, 0};
#else X10
	dv_c = {"$Header: xdvi.c (X10), patchlevel = ", '0' + PATCHLEVEL, 0};
#endif X10
#endif	lint

#ifndef X10

#undef Boolean
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#ifdef OLD_X11_TOOLKIT
#include <X11/Atoms.h>
#else not OLD_X11_TOOLKIT
#include <X11/Xatom.h>
#include <X11/StringDefs.h>
#endif not OLD_X11_TOOLKIT
#include <X11/Shell.h>	/* needed for def. of XtNiconX */
#include <X11/Viewport.h>
#include <X11/Simple.h>
#include <X11/cursorfont.h>
#include "xdvi.icon"

#else X10

#include <X/Xlib.h>

#endif X10

#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include <signal.h>
#include "xdvi.h"

#ifndef X10
static	Display	*DISP;
#define	DPY	DISP,
static	Screen	*SCRN;
static	Cursor	redraw_cursor, ready_cursor;
#define	SetCursor(x)	XDefineCursor(DISP, mane.win, x)
#define	ClearPage()	XClearWindow(DISP, mane.win);
#define	Flush()		XFlush(DISP)
static	Boolean	unmapped = True;
#ifndef X11HEIGHT
#define	X11HEIGHT	8	/* Height of server default font */
#endif X11HEIGHT
#else X10
#define	DPY
#define	GC		int
#define	XtNumber(arr)	(sizeof(arr)/sizeof(arr[0]))
#define	XtPending	XPending
typedef	int		Position;
typedef	int		Dimension;
#define	SetCursor(x)
#define	ClearPage()	XClear(mane.win);
#define	XBell(a,b)	XFeep(b/10-1)
#define	Flush()		XFlush()
#define	ConnectionNumber(DISP)	(_XlibCurrentDisplay->fd)
#ifndef X10FONT
#define	X10FONT	"helv10b"	/* Font for X10 error messages */
#define	X10HEIGHT	10
#endif X10FONT
#endif X10

#define	MAGBORD	1	/* border size for magnifier */
char	*font_path;
char	default_font_path[]	= DEFAULT_FONT_PATH;

/*
 * Command line flags.
 */
int	debug = 0;
Boolean	list_fonts = False;

int	density = 40;
int	pixels_per_inch = 300;
static	char	*margins, *sidemargin, *topmargin;
static	Boolean	reverse;
static	Dimension	bwidth	= 2;
static	int	bak_shrink;
static	char	*debug_arg;
static	int	mg_size[5] = {200, 350, 600, 900, 1200};

char	*dvi_name = NULL;
FILE	*dvi_file;				/* user's file */
char	*prog;
char	*curr_page = NULL;

#ifndef X10
static	double	specialConv;
		/* fg and bg colors */
static	Arg	fore_args = {XtNforeground,	(XtArgVal) 0};
static	Arg	back_args = {XtNbackground,	(XtArgVal) 0};
static	XColor	hl_Color, cr_Color;
#endif X10

static	char	*fore_color;
static	char	*back_color;
static	char	*high_color;
static	char	*curs_color;
static	GC	foreGC, highGC;
#ifndef X10
static	GC	ruleGC;
static	GC	foreGC2;
#else X10
#define	ruleGC	foreGC
#endif X10

int	page_w, page_h;
static	int	screen_w, screen_h;
static	Dimension	window_w, window_h;
static	int	home_x, home_y;
static	int	min_x, max_x, min_y, max_y;

struct WindowRec mane	= {NULL, 4, 0, 0, MAXINT, 0, MAXINT, 0};
struct WindowRec alt	= {NULL, 1, 0, 0, MAXINT, 0, MAXINT, 0};
/*	curr is temporary storage except for within redraw() */
struct WindowRec curr	= {NULL, 4, 0, 0, MAXINT, 0, MAXINT, 0};

/*
 *	Mechanism to keep track of the magnifier window.  The problems are,
 *	(a) if the button is released while the window is being drawn, this
 *	could cause an X error if we continue drawing in it after it is
 *	destroyed, and
 *	(b) creating and destroying the window too quickly confuses the window
 *	manager, which is avoided by waiting for an expose event before
 *	destroying it.
 */
static	short	alt_stat;	/* 1 = wait for expose, */
				/* -1 = destroy upon expose */
static	Boolean	alt_canit;	/* stop drawing this window */

int	pageno_correct	= 1;

/*
 *	Data for buffered events.
 */

static	Boolean	canit		= False,
		arg		= False;
static	short	event_counter	= 0,
		event_freq	= 70;
static	int	number		= 0,
		sign		= 1;
static	jmp_buf	canit_env;

static	void	can_exposures(), read_events();

char	*malloc(), *index(), *rindex(), *sprintf(), *getenv(),
	*strcpy(), *strcat();

double	atof();

#ifndef X10
/* Things we need from spec_draw, unfortunately */

/* (ignored for now)
extern int pen_size, blacken, whiten, shade;
*/

#define	toint(x)	((int) ((x) + 0.5))
#define	xconv(x)	(toint(specialConv*(x))/shrink_factor + PXL_H)
#define	yconv(y)	(toint(specialConv*(y))/shrink_factor + PXL_V)

/*
 *	Draw a line from (fx,fy) to (tx,ty).
 *	Right now, we ignore pen_size.
 */
void
line_btw(fx, fy, tx, ty)
int fx, fy, tx, ty;
{
	register int	fcx = xconv(fx),
			tcx = xconv(tx),
			fcy = yconv(fy),
			tcy = yconv(ty);

	if ((fcx < max_x || tcx < max_x) && (fcx >= min_x || tcx >= min_x) &&
	    (fcy < max_y || tcy < max_y) && (fcy >= min_y || tcy >= min_y))
		XDrawLine(DISP, curr.win, ruleGC,
		    fcx - curr.base_x, fcy - curr.base_y,
		    tcx - curr.base_x, tcy - curr.base_y);
}

/*
 *	Draw a dot at (x,y)
 */
void
dot_at(x, y)
{
	register int	cx = xconv(x),
			cy = yconv(y);

	if (cx < max_x && cx >= min_x && cy < max_y && cy >= min_y)
	    XDrawPoint(DISP, curr.win, ruleGC,
		cx - curr.base_x, cy - curr.base_y);
}

/*
 *	Apply the requested attributes to the last path (box) drawn.
 *	Attributes are reset.
 *	(Not currently implemented.)
 */
	/* ARGSUSED */
void
do_attribute_path(last_min_x, last_max_x, last_min_y, last_max_y)
int last_min_x, last_max_x, last_min_y, last_max_y;
{
}
#else X10
/*
 *	Specials are not implemented in X10.
 */

void
applicationDoSpecial(cmd)
char	*cmd;
{
	if (spec_warn) Fprintf(stderr, "special ``%s'' not implemented\n", cmd);
}
#endif X10

#ifndef X10
static	Widget	top_level, vport_widget, draw_widget, clip_widget;
static	XImage	*image;

static	Arg	vport_args[] = {
	{XtNallowHoriz,	(XtArgVal) True},
	{XtNallowVert,	(XtArgVal) True},
};

/*	Note:  Argument order in the following is important! */

static	Arg	draw_args[] = {
	{XtNwidth,	(XtArgVal) 0},
	{XtNheight,	(XtArgVal) 0},
	{XtNx,		(XtArgVal) 0},
	{XtNy,		(XtArgVal) 0},
	{XtNlabel,	(XtArgVal) ""},
};

static	void	set_draw_args();
#else X10
static	int	GXfunc;
static	int	backpix, backmap, bdrmap;
/*
 * Cursor and mask for valid cursor
 */
#include "xdvi_curs.h"
#include "xdvi_mask.h"
#endif X10

#ifdef lint
#ifndef X10
WidgetClass	viewportWidgetClass, simpleWidgetClass;
char	xdvi_bits[288];
#else X10
short	xdvi_bits[15], xdvi_mask_bits[15];
Display	*_XlibCurrentDisplay;
#endif X10
#endif lint

/**
 **	Put a rectangle on the screen.  hl determines the GC.
 **/

put_rectangle(x, y, w, h, hl)
	int x, y, w, h;
	Boolean hl;
{
	if (x < max_x && x + w >= min_x && y < max_y && y + h >= min_y) {
		if (--event_counter == 0) read_events(False);
#ifndef X10
		XFillRectangle(DISP, curr.win, hl ? highGC : ruleGC,
			       x - curr.base_x, y - curr.base_y, w?w:1, h?h:1);
#else X10
		XPixSet(curr.win, x - curr.base_x, y - curr.base_y,
			w?w:1, h?h:1, hl ? highGC : ruleGC);
#endif X10
	}
}

put_bitmap(bitmap, x, y)
	register struct bitmap *bitmap;
	register int x, y;
{

	if (debug & DBG_BITMAP)
		Printf("X(%d,%d)\n", x-curr.base_x, y-curr.base_y);
	if (x < max_x && x + bitmap->w >= min_x &&
	    y < max_y && y + bitmap->h >= min_y) {
		if (--event_counter == 0) read_events(False);
#ifndef X10
		image->width = bitmap->w;
		image->height = bitmap->h;
		image->data = (char *)bitmap->bits;
		image->bytes_per_line = bitmap->bytes_wide;
		XPutImage(DISP, curr.win, foreGC, image,
			0, 0,
			x - curr.base_x, y - curr.base_y,
			bitmap->w, bitmap->h);
		if (foreGC2)
		    XPutImage(DISP, curr.win, foreGC2, image,
			0, 0,
			x - curr.base_x, y - curr.base_y,
			bitmap->w, bitmap->h);
#else X10
		XBitmapBitsPut(curr.win, x - curr.base_x, y - curr.base_y,
			bitmap->w, bitmap->h, (char *) bitmap->bits,
			foreGC, backpix, NULL, GXfunc, AllPlanes);
#endif X10
	}
}

put_border(x, y, w, h, t)
	int x, y, w, h, t;
{
	put_rectangle(x, y, w, t, True);
	put_rectangle(x, y, t, h, True);
	put_rectangle(x, y + h - t, w, t, True);
	put_rectangle(x + w - t, y, t, h, True);
}

#ifndef X10
/*
 *	routines for X11 toolkit
 */

static	Arg	arg_wh[] = {
	{XtNwidth,	(XtArgVal) &window_w},
	{XtNheight,	(XtArgVal) &window_h},
};

#define	get_wh(widget)	XtGetValues(widget, arg_wh, 2)

static	Position	window_x, window_y;
static	Arg		arg_xy[] = {
	{XtNx,		(XtArgVal) &window_x},
	{XtNy,		(XtArgVal) &window_y},
};

#define	get_xy()	XtGetValues(draw_widget, arg_xy, 2)

#define	mane_base_x	0
#define	mane_base_y	0

static	Boolean
scroll(horizontal, percent)
	Boolean	horizontal;
	float	percent;
{
	register Widget	widget;

	widget = XtNameToWidget(vport_widget,
		horizontal ? "horizontal" : "vertical");
	if (!widget) return False;
	XtGetValues(clip_widget, horizontal ? &arg_wh[0] : &arg_wh[1], 1);
	XtCallCallbacks(widget, XtNscrollProc,
		(int) (percent*(horizontal ? window_w : window_h)));
	return True;
}

/*
 *	We unmap the window so that it does not generate expose events when
 *	moving things around.  I have found that compress_exposure does not do
 *	this.
 */

static
unmap()
{
	if (unmapped) return;
	XUnmapWindow(DISP, mane.win);
	unmapped = True;
}

static
home()
{
	register Widget	widget;
	register int coord;

	unmap();
	get_xy();
	get_wh(clip_widget);
	widget = XtNameToWidget(vport_widget, "horizontal");
	if (widget) {
	    coord = 0;
	    if (page_w > window_w) {
		coord = (page_w - window_w) / 2;
		if (coord > home_x / mane.shrinkfactor)
		    coord = home_x / mane.shrinkfactor;
	    }
	    XtCallCallbacks(widget, XtNscrollProc, window_x + coord);
	}
	widget = XtNameToWidget(vport_widget, "vertical");
	if (widget) {
	    coord = 0;
	    if (page_h > window_h) {
		coord = (page_h - window_h) / 2;
		if (coord > home_y / mane.shrinkfactor)
		    coord = home_y / mane.shrinkfactor;
	    }
	    XtCallCallbacks(widget, XtNscrollProc, window_y + coord);
	}
}

static	void
center(x, y)
	int x, y;
{
	register Widget widget;

/*	We use the clip widget here because it gives a more exact value. */
	get_wh(clip_widget);
	x -= window_w/2;
	y -= window_h/2;
	widget = XtNameToWidget(vport_widget, "horizontal");
	if (widget) XtCallCallbacks(widget, XtNscrollProc, x);
	widget = XtNameToWidget(vport_widget, "vertical");
	if (widget) XtCallCallbacks(widget, XtNscrollProc, y);
	XWarpPointer(DISP, None, None, 0, 0, 0, 0, -x, -y);
}

/*
 *	callback routines
 */

/* The following callback routine should never be called. */
	/*ARGSUSED*/
static	void
handle_key(widget, junk, event)
	Widget	widget;
	caddr_t	junk;
	XEvent	*event;
{
	XBell(DISP, 20);
}
#else X10
static
home()
{
	mane.base_x = 0;
	if (page_w > window_w) {
	    mane.base_x = (page_w - window_w) / 2;
	    if (mane.base_x > home_x / mane.shrinkfactor)
		mane.base_x = home_x / mane.shrinkfactor;
	}
	mane.base_y = 0;
	if (page_h > window_h) {
	    mane.base_y = (page_h - window_h) / 2;
	    if (mane.base_y > home_y / mane.shrinkfactor)
		mane.base_y = home_y / mane.shrinkfactor;
	}
}

#define	unmap()
#define	get_wh(widget)
#define	get_xy()
#define	window_x 0
#define	window_y 0
#define	mane_base_x	mane.base_x
#define	mane_base_y	mane.base_y
#endif X10

#ifndef X10
	/*ARGSUSED*/
static	void
handle_button(widget, junk, event)
	Widget	widget;
	caddr_t	junk;
	XButtonEvent *event;
#else X10
static	void
handle_button(event, mag_size)
	XButtonPressedEvent *event;
	int mag_size;
#endif X10
{
	int x, y;
#ifndef X10
	int mag_size = mg_size[event->button - 1];
	XSetWindowAttributes attr;
#endif X10

	if (alt.win != NULL || mane.shrinkfactor == 1 || mag_size <= 0)
	    XBell(DISP, 20);
	else {
#ifndef X10
	    x = event->x_root - mag_size/2;
	    if (x > WidthOfScreen(SCRN) - mag_size - 2*MAGBORD)
		x = WidthOfScreen(SCRN) - mag_size - 2*MAGBORD;
	    if (x < 0) x = 0;
	    y = event->y_root - mag_size/2;
	    if (y > HeightOfScreen(SCRN) - mag_size - 2*MAGBORD)
		y = HeightOfScreen(SCRN) - mag_size - 2*MAGBORD;
	    if (y < 0) y = 0;
#else X10
	    x = event->x - mag_size/2;
	    if (x > window_w - mag_size - 2*MAGBORD)
		x = window_w - mag_size - 2*MAGBORD;
	    if (x < 0) x = 0;
	    y = event->y - mag_size/2;
	    if (y > window_h - mag_size - 2*MAGBORD)
		y = window_h - mag_size - 2*MAGBORD;
	    if (y < 0) y = 0;
#endif X10
	    alt.base_x = (event->x + mane_base_x) * mane.shrinkfactor -
		mag_size/2;
	    alt.base_y = (event->y + mane_base_y) * mane.shrinkfactor -
		mag_size/2;
#ifndef X10
	    attr.save_under = True;
	    attr.border_pixel = fore_args.value;
	    attr.background_pixel = back_args.value;
	    alt.win = XCreateWindow(DISP, RootWindowOfScreen(SCRN),
			x, y, mag_size, mag_size, MAGBORD,
			0,	/* depth from parent */
			InputOutput, CopyFromParent,
			CWSaveUnder|CWBorderPixel|CWBackPixel, &attr);
	    XSetTransientForHint(DISP, alt.win, XtWindow(top_level));
	    XSelectInput(DISP, alt.win, ExposureMask);
#else X10
	    alt.win = XCreateWindow(mane.win,
			x, y, mag_size, mag_size, MAGBORD,
			bdrmap, backmap);
	    XSelectInput(alt.win, ExposeRegion);
#endif X10
	    XMapWindow(DPY alt.win);
	    alt_stat = 1;	/* waiting for exposure */
	}
}

#ifndef X10
	/*ARGSUSED*/
static	void
handle_release(widget, junk, event)
	Widget	widget;
	caddr_t	junk;
	XButtonEvent *event;
#else X10
static	void
handle_release()
#endif X10
{
	if (alt.win)
	    if (alt_stat) alt_stat = -1;	/* destroy upon expose */
	    else {
		XDestroyWindow(DPY alt.win);
		if (curr.win == alt.win) alt_canit = True;
		alt.win = NULL;
		can_exposures(&alt);
	    }
}

#ifndef X10
	/*ARGSUSED*/
static	void
handle_exp(widget, windowrec, event)
	Widget	widget;
	struct WindowRec *windowrec;
	register XExposeEvent *event;
{
	if (windowrec == &alt)
	    if (alt_stat < 0) {	/* destroy upon exposure */
		alt_stat = 0;
		handle_release(widget, (caddr_t) NULL, (XButtonEvent *) event);
		return;
	    }
	    else
		alt_stat = 0;
	if (windowrec->min_x > event->x) windowrec->min_x = event->x;
	if (windowrec->max_x < event->x + event->width)
	    windowrec->max_x = event->x + event->width;
	if (windowrec->min_y > event->y) windowrec->min_y = event->y;
	if (windowrec->max_y < event->y + event->height)
	    windowrec->max_y = event->y + event->height;
}
#endif X10

#ifndef X10
#define	TRSIZE	100
#endif X10
static	void
read_events(wait)
	Boolean	wait;
{
	char	ch;
	Boolean	arg0;
	int	number0;
	XEvent	event;
#ifndef X10
	char	trbuf[TRSIZE];
#endif X10
	char	*string;
	int	nbytes;
	int	next_page;

	alt_canit = False;
	for (;;) {
	    ch = '\0';
	    event_counter = event_freq;
	    /*
	     * If we get a hit at this point, then we'll just end up making
	     * an extra call.
	     * Also, watch out, if we destroy the magnifying glass while
	     * writing it.
	     */
	    if (!XtPending() && (!wait || canit || mane.min_x < MAXINT ||
		    alt.min_x < MAXINT))
		if (alt_canit) longjmp(canit_env, 1);
		else return;
#ifndef X10
	    XtNextEvent(&event);
	    if (event.xany.window == alt.win &&
		    event.type == Expose) {
		handle_exp((Widget) NULL, &alt, &event.xexpose);
		continue;
	    }
	    if (event.type != KeyPress) {
		XtDispatchEvent(&event);
		continue;
	    }
	    string = trbuf;
	    nbytes = XLookupString(&event, string, TRSIZE, NULL, NULL);
	    if (nbytes > 1) goto bad;
	    if (nbytes != 0) ch = *string;
#else X10
	    XNextEvent(&event);
	    switch (event.type) {
	    case ExposeWindow:
		if (event.window == mane.win) {
		    window_h = ((XExposeEvent *)(&event))->height;
		    window_w = ((XExposeEvent *)(&event))->width;
		    home();
		    ch = '\f';
		    break;
		}
		/* otherwise control passes through */

	    case ExposeRegion:
		    /* check in case we already destroyed the window */
		if (event.window == mane.win || alt.win != NULL) {
		    struct WindowRec *wr =
			(event.window == mane.win ? &mane : &alt);
		    if (wr == &alt)
			if (alt_stat < 0) { /* destroy upon exposure */
			    alt_stat = 0;
			    handle_release();
			    break;
			}
			else
			    alt_stat = 0;
#define	ev		((XExposeEvent *)(&event))
		    if (wr->min_x > ev->x) wr->min_x = ev->x;
		    if (wr->max_x < ev->x + ev->width)
			wr->max_x = ev->x + ev->width;
		    if (wr->min_y > ev->y) wr->min_y = ev->y;
		    if (wr->max_y < ev->y + ev->height)
			wr->max_y = ev->y + ev->height;
#undef	ev
		}
		break;

	    case ButtonPressed: {
		    int n = 0;
		    switch (((XButtonPressedEvent *) (&event))->detail &
			    ValueMask) {
			case LeftButton:  n=0; break;
			case MiddleButton:  n=1; break;
			case RightButton:  n=2; break;
		    }
		    handle_button((XButtonPressedEvent *) (&event), mg_size[n]);
		}
		break;
	    case ButtonReleased:
		handle_release();
		break;
	    case KeyPressed:
		string = XLookupMapping (&event, &nbytes);
		if (nbytes > 1) goto bad;
		if (nbytes != 0) ch = *string;
		break;
	    }
#endif X10
	    if (ch == '\0') continue;
	    if (ch >= '0' && ch <= '9') {
		arg = True;
		number = number * 10 + sign * (ch - '0');
		continue;
	    }
	    else if (ch == '-') {
		arg = True;
		sign = -1;
		number = 0;
		continue;
	    }
	    arg0 = arg;
	    arg = False;
	    number0 = number;
	    number = 0;
	    sign = 1;
	    next_page = current_page;

	    switch (ch) {
		case 'q':
		case '\003':	/* control-C */
		case '\004':	/* control-D */
		    exit(0);
		case 'n':
		case 'f':
		case ' ':
		case '\r':
		case '\n':
		    /* scroll forward; i.e. go to relative page */
		    next_page = current_page + (arg0 ? number0 : 1);
		    break;
		case 'p':
		case 'b':
		case '\b':
		case '\177':	/* Del */
		    /* scroll backward */
		    next_page = current_page - 1;
		    break;
		case 'g':
		    /* go to absolute page */
		    next_page = (arg0 ? number0 - pageno_correct :
			total_pages - 1);
		    break;
		case 'P':		/* declare current page */
		    pageno_correct = arg0 * number0 - current_page;
		    continue;
		case '\f':
		    /* redisplay current page */
		    break;
		case '^':
		    home();
		    break;
#ifndef X10
		case 'u':
		    if (!scroll(False, -0.67)) goto bad;
		    continue;
		case 'd':
		    if (!scroll(False, 0.67)) goto bad;
		    continue;
		case 'l':
		    if (!scroll(True, -0.67)) goto bad;
		    continue;
		case 'r':
		    if (!scroll(True, 0.67)) goto bad;
		    continue;
		case 'c':
		    center(event.xkey.x, event.xkey.y);
		    continue;
		case 'M':
		    XTranslateCoordinates(DISP, event.xkey.window, mane.win,
			    event.xkey.x, event.xkey.y, &home_x, &home_y,
			    &number0);	/* throw away last argument */
		    home_x *= mane.shrinkfactor;
		    home_y *= mane.shrinkfactor;
		    continue;
		case '\020':	/* Control P */
		    Printf("Unit = %d, bitord = %d, byteord = %d\n",
			BitmapUnit(DISP), BitmapBitOrder(DISP),
			ImageByteOrder(DISP));
		    continue;
#else X10
		case 'u':
		    if (mane.base_y == 0) goto bad;
		    mane.base_y -= window_h;
		    if (mane.base_y < 0)
			mane.base_y = 0;
		    break;
		case 'd':
		    if (mane.base_y >= page_h - window_h) goto bad;
		    mane.base_y += window_h;
		    if (mane.base_y > page_h - window_h)
			mane.base_y = page_h - window_h;
		    break;
		case 'l':
		    if (mane.base_x == 0) goto bad;
		    mane.base_x -= window_w;
		    if (mane.base_x < 0)
			mane.base_x = 0;
		    break;
		case 'r':
		    if (mane.base_x >= page_w - window_w) goto bad;
		    mane.base_x += window_w;
		    if (mane.base_x > page_w - window_w)
			mane.base_x = page_w - window_w;
		    break;
		case 'c':
#define	ev		((XKeyPressedEvent *) (&event))
		    mane.base_x += ev->x - window_w/2;
		    mane.base_y += ev->y - window_h/2;
		    XWarpMouse(mane.win, window_w/2, window_h/2, 3);
		    break;
		case 'M':
		    home_x = (ev->x + mane.base_x) * mane.shrinkfactor;
		    home_y = (ev->y + mane.base_y) * mane.shrinkfactor;
		    continue;
#undef	ev
#endif X10
		case 's':
		    if (!arg0) {
			long fac1, fac2;
			shrink_factor = 1;
			get_wh(vport_widget);
			fac1 = ROUNDUP(PAPER_WIDTH, window_w);
			fac2 = ROUNDUP(PAPER_HEIGHT, window_h);
			if (fac1 < fac2)
			    number0 = fac2;
			else
			    number0 = fac1;
		    }
		    if (number0 <= 0) goto bad;
		    if (number0 == mane.shrinkfactor) continue;
		    shrink_factor = mane.shrinkfactor = number0;
		    unmap();
		    init_page();
		    if (number0 != 1 && number0 != bak_shrink) {
			bak_shrink = number0;
			reset_fonts();
		    }
#ifndef X10
		    set_draw_args();
		    XtSetValues(draw_widget, draw_args, 2);
#endif X10
		    home();
		    break;
		case 'S':
		    if (!arg0) goto bad;
		    if (number0 < 0) goto bad;
		    if (number0 == density) continue;
		    density = number0;
		    reset_fonts();
		    if (mane.shrinkfactor == 1) continue;
		    unmap();
		    break;
		case 'R':
		    /* reread DVI file */
		    --dvi_time;	/* then it will notice a change */
		    break;
		default:
		    goto bad;
	    }
	    if (0 <= next_page && next_page < total_pages) {
		if (current_page != next_page) {
		    current_page = next_page;
		    spec_warn = True;
		    home();
		}
		canit = True;
		Flush();
		longjmp(canit_env, 1);
	    }
	    bad:  XBell(DISP, 10);
	}
}

static
redraw(windowrec)
	struct WindowRec *windowrec;
{
	char	*errtext;
#ifdef X10
	static FontInfo *font = 0;
#endif X10

	curr = *windowrec;
	min_x = curr.min_x + curr.base_x;
	min_y = curr.min_y + curr.base_y;
	max_x = curr.max_x + curr.base_x;
	max_y = curr.max_y + curr.base_y;
	can_exposures(windowrec);

	if (debug & DBG_EVENT)
	    Printf("Redraw %d x %d at (%d, %d) (base=%d,%d)\n", max_x - min_x,
		max_y - min_y, min_x, min_y, curr.base_x, curr.base_y);
	SetCursor(redraw_cursor);
	if (errtext = (char *) setjmp(dvi_env)) {
	    ClearPage();
#ifndef X10
	    get_xy();
	    XDrawString(DISP, mane.win, foreGC,
		5 - window_x, 5 + X11HEIGHT - window_y,
		errtext, strlen(errtext));
#else X10
	    if (!font) font = XOpenFont(X10FONT);
	    XTextMask(mane.win, 5, 5 + X10HEIGHT, errtext, strlen(errtext),
		font->id, foreGC);
#endif X10
	    if (dvi_file) {
		Fclose(dvi_file);
		dvi_file = NULL;
	    }
	}
	else {
	    draw_page();
	    spec_warn = False;
	}
}

redraw_page()
{
	if (debug & DBG_EVENT) fputs("Redraw page:  ", stdout);
	get_wh(clip_widget);
	get_xy();
	ClearPage();
	mane.min_x = -window_x;
	mane.max_x = -window_x + window_w;
	mane.min_y = -window_y;
	mane.max_y = -window_y + window_h;
	redraw(&mane);
}

/*
 *	Interrupt system for receiving events.  The program sets a flag
 *	whenever an event comes in, so that at the proper time (i.e., when
 *	reading a new dvi item), we can check incoming events to see if we
 *	still want to go on printing this page.  This way, one can stop
 *	displaying a page if it is about to be erased anyway.  We try to read
 *	as many events as possible before doing anything and base the next
 *	action on all events read.
 *	Note that the Xlib and Xt routines are not reentrant, so the most we
 *	can do is set a flag in the interrupt routine and check it later.
 *	Also, sometimes the interrupts are not generated (some systems only
 *	guarantee that SIGIO is generated for terminal files, and on the system
 *	I use, the interrupts are not generated if I use "(xdvi foo &)" instead
 *	of "xdvi foo").  Therefore, there is also a mechanism to check the
 *	event queue every 70 drawing operations or so.  This mechanism is
 *	disabled if it turns out that the interrupts do work.
 *	For a fuller discussion of some of the above, see xlife in
 *	comp.sources.x.
 */

static	void
can_exposures(windowrec)
	struct WindowRec *windowrec;
{
	windowrec->min_x = windowrec->min_y = MAXINT;
	windowrec->max_x = windowrec->max_y = 0;
}

static	int
handle_intr() {
	event_counter = 1;
	event_freq = -1;	/* forget Plan B */
}

static	void
enable_intr() {
	int	socket	= ConnectionNumber(DISP);
	if (!isatty(0)) {
	    puts("trying...");
	    if (dup2(socket, 0) == -1) perror(prog);
	    socket = 0;
	}
	(void) signal(SIGIO, handle_intr);
	(void) fcntl(socket, F_SETOWN, getpid());
	(void) fcntl(socket, F_SETFL, fcntl(socket, F_GETFL, 0) | FASYNC);
}

static
do_pages()
{
	if (debug & DBG_BATCH) {
#ifndef X10
	    while (mane.min_x == MAXINT) read_events(True);
#else X10
	    while (mane.min_x == MAXINT)
		if (setjmp(canit_env)) break;
		else read_events(True);
#endif X10
	    for (current_page = 0; current_page < total_pages; ++current_page)
		redraw_page();
	    exit(0);
	}
	else {
	    enable_intr();
	    (void) setjmp(canit_env);
	    for (;;) {
		SetCursor(ready_cursor);
		read_events(True);
		if (canit) {
		    canit = False;
		    can_exposures(&mane);
		    can_exposures(&alt);
#ifndef X10
		    if (unmapped) {
				/* this creates a redraw event */
			XMapWindow(DISP, mane.win);
			unmapped = False;
		    }
		    else
#endif
			redraw_page();
		}
		else if (alt.min_x < MAXINT) redraw(&alt);
		else if (mane.min_x < MAXINT) redraw(&mane);
		Flush();
	    }
	}
}

static
usage() {
#ifndef X10
	fputs("\
Usage: xdvi [+[<page>]] [-s <shrink>] [-S <density>] [-p <pixels>] [-l] [-rv]\n\
	[-fg <color>] [-bg <color>] [-hl <color>] [-bd <color>] \
[-cr <color>]\n\
	[-margins <inches>] [-sidemargin <inches>] [-topmargin <inches>]\n\
	[-mgs[n] <size>] [-geometry <geometry>]  [#<geometry>]\n\
	[-display <host:display>] dvi_file\n", stderr);
#else X10
	fputs("\
Usage: xdvi [+[<page>]] [-s <shrink>] [-S <density>] [-p <pixels>] [-l] [-rv]\n\
	[-fg <color>] [-bg <color>] [-hl <color>] [-bd <color>] \
[-cr <color>]\n\
	[-margins <inches>] [-sidemargin <inches>] [-topmargin <inches>]\n\
	[-mgs[n] <size>] [-geometry <geometry> | =<geometry>]\n\
	[-display <host:display> | host:display] dvi_file\n", stderr);
#endif X10
	exit(1);
}

/**
 **	Main programs start here.
 **/

#ifndef X10
static	char	*icon_geometry;
static	Boolean	thorough;

static	XrmOptionDescRec	options[] = {
{"-d",		".debugLevel",	XrmoptionSepArg,	(caddr_t) NULL},
{"+",		".gotoPage",	XrmoptionStickyArg,	(caddr_t) NULL},
{"-s",		".shrinkFactor", XrmoptionSepArg,	(caddr_t) NULL},
{"-S",		".densityPercent", XrmoptionSepArg,	(caddr_t) NULL},
{"-p",		".pixelsPerInch", XrmoptionSepArg,	(caddr_t) NULL},
{"-margins",	".margins",	XrmoptionSepArg,	(caddr_t) NULL},
{"-sidemargin",	".sideMargin",	XrmoptionSepArg,	(caddr_t) NULL},
{"-topmargin",	".topMargin",	XrmoptionSepArg,	(caddr_t) NULL},
{"-l",		".listFonts",	XrmoptionNoArg,		(caddr_t) "on"},
{"+l",		".listFonts",	XrmoptionNoArg,		(caddr_t) "off"},
{"-fg",		".foreground",	XrmoptionSepArg,	(caddr_t) NULL},
{"-foreground",	".foreground",	XrmoptionSepArg,	(caddr_t) NULL},
{"-bg",		".background",	XrmoptionSepArg,	(caddr_t) NULL},
{"-background",	".background",	XrmoptionSepArg,	(caddr_t) NULL},
{"-hl",		".highlight",	XrmoptionSepArg,	(caddr_t) NULL},
{"-cr",		".cursorColor",	XrmoptionSepArg,	(caddr_t) NULL},
{"#",		".iconGeometry",XrmoptionStickyArg,     (caddr_t) NULL},
{"-thorough",	".thorough",	XrmoptionNoArg,		(caddr_t) "on"},
{"+thorough",	".thorough",	XrmoptionNoArg,		(caddr_t) "off"},
{"-mgs",	".magnifierSize1",XrmoptionSepArg,	(caddr_t) NULL},
{"-mgs1",	".magnifierSize1",XrmoptionSepArg,	(caddr_t) NULL},
{"-mgs2",	".magnifierSize2",XrmoptionSepArg,	(caddr_t) NULL},
{"-mgs3",	".magnifierSize3",XrmoptionSepArg,	(caddr_t) NULL},
{"-mgs4",	".magnifierSize4",XrmoptionSepArg,	(caddr_t) NULL},
{"-mgs5",	".magnifierSize5",XrmoptionSepArg,	(caddr_t) NULL},
};

static	XtResource	resources[] = {
{"debugLevel", "DebugLevel", XtRString, sizeof(char *),
  (Cardinal) &debug_arg, XtRString, NULL},
{"gotoPage", "GotoPage", XtRString, sizeof(char *),
  (Cardinal) &curr_page, XtRString, NULL},
{"shrinkFactor", "ShrinkFactor", XtRInt, sizeof(int),
  (Cardinal) &shrink_factor, XtRInt, (caddr_t) &shrink_factor},
{"densityPercent", "DensityPercent", XtRInt, sizeof(int),
  (Cardinal) &density, XtRInt, (caddr_t) &density},
{"pixelsPerInch", "PixelsPerInch", XtRInt, sizeof(int),
  (Cardinal) &pixels_per_inch, XtRInt, (caddr_t) &pixels_per_inch},
{"margins", "Margin", XtRString, sizeof(char *),
  (Cardinal) &margins, XtRString, NULL},
{"sideMargin", "Margin", XtRString, sizeof(char *),
  (Cardinal) &sidemargin, XtRString, NULL},
{"topMargin", "Margin", XtRString, sizeof(char *),
  (Cardinal) &topmargin, XtRString, NULL},
{"listFonts", "ListFonts", XtRBoolean, sizeof(Boolean),
  (Cardinal) &list_fonts, XtRBoolean, (caddr_t) &list_fonts},
{"reverseVideo", "ReverseVideo", XtRBoolean, sizeof(Boolean),
  (Cardinal) &reverse, XtRBoolean, (caddr_t) &reverse},
{"foreground", "Foreground", XtRPixel, sizeof(Pixel),
  (Cardinal)&fore_args.value, XtRPixel, (caddr_t) &fore_args.value},
{"foreground", "Foreground", XtRString, sizeof(char *),
  (Cardinal)&fore_color, XtRString, NULL},
{"background", "Background", XtRPixel, sizeof(Pixel),
  (Cardinal)&back_args.value, XtRPixel, (caddr_t) &back_args.value},
{"background", "Background", XtRString, sizeof(char *),
  (Cardinal)&back_color, XtRString, NULL},
{"highlight", "Highlight", XtRColor, sizeof(XColor),
  (Cardinal)&hl_Color, XtRColor, (caddr_t) &hl_Color},
{"highlight", "Highlight", XtRString, sizeof(char *),
  (Cardinal)&high_color, XtRString, NULL},
{"cursorColor", "CursorColor", XtRColor, sizeof(XColor),
  (Cardinal)&cr_Color, XtRColor, (caddr_t) &cr_Color},
{"cursorColor", "CursorColor", XtRString, sizeof(char *),
  (Cardinal)&curs_color, XtRString, NULL},
{"iconGeometry", "IconGeometry", XtRString, sizeof(char *),
  (Cardinal)&icon_geometry, XtRString, (caddr_t) NULL},
{"thorough", "Thorough", XtRBoolean, sizeof(Boolean),
  (Cardinal)&thorough, XtRBoolean, (caddr_t) &thorough},
{"magnifierSize1", "MagnifierSize", XtRInt, sizeof(int),
  (Cardinal) &mg_size[0], XtRInt, (caddr_t) &mg_size[0]},
{"magnifierSize2", "MagnifierSize", XtRInt, sizeof(int),
  (Cardinal) &mg_size[1], XtRInt, (caddr_t) &mg_size[1]},
{"magnifierSize3", "MagnifierSize", XtRInt, sizeof(int),
  (Cardinal) &mg_size[2], XtRInt, (caddr_t) &mg_size[2]},
{"magnifierSize4", "MagnifierSize", XtRInt, sizeof(int),
  (Cardinal) &mg_size[3], XtRInt, (caddr_t) &mg_size[3]},
{"magnifierSize5", "MagnifierSize", XtRInt, sizeof(int),
  (Cardinal) &mg_size[4], XtRInt, (caddr_t) &mg_size[4]},
};

static	void
set_draw_args() {
	draw_args[0].value = (XtArgVal) page_w;
	draw_args[1].value = (XtArgVal) page_h;
}

static	Arg	temp_args1[] = {
	{XtNiconX,	(XtArgVal) 0},
	{XtNiconY,	(XtArgVal) 0},
};

static	Arg	temp_args2 = {XtNborderWidth,	(XtArgVal) &bwidth};

static	Arg	temp_args3[] = {
	{XtNwidth,	(XtArgVal) 0},
	{XtNheight,	(XtArgVal) 0},
	{XtNiconPixmap,	(XtArgVal) 0},
};

/*
 *	X11 main program
 */

main(argc, argv)
	int argc;
	char **argv;
{
	prog = *argv;
	if (*prog == '/') prog = rindex(prog, '/') + 1;

	top_level = XtInitialize(prog, "XDvi", options, XtNumber(options),
		&argc, argv);
	if (argc != 2) usage();
	dvi_name = argv[1];

	XtGetApplicationResources(top_level, (caddr_t) NULL, resources,
		XtNumber(resources), NULL, 0);
	if (shrink_factor <= 0 || density <= 0 || pixels_per_inch <= 0) usage();
	if (shrink_factor != 1) bak_shrink = shrink_factor;
	mane.shrinkfactor = shrink_factor;
	specialConv = pixels_per_inch / 1000.0;
	if (debug_arg != NULL)
	    debug = isdigit(*debug_arg) ? atoi(debug_arg) : DBG_ALL;
	if (margins) home_x = home_y = atof(margins) * pixels_per_inch;
	if (sidemargin) home_x = atof(sidemargin) * pixels_per_inch;
	if (topmargin) home_y = atof(topmargin) * pixels_per_inch;
		/* The following code is lifted from Xterm */
	DISP = XtDisplay(top_level);
	SCRN = XtScreen(top_level);
	if (icon_geometry != NULL) {
	    int scr, junk;

	    for(scr = 0;	/* yyuucchh */
		SCRN != ScreenOfDisplay(DISP, scr);
		scr++);

	    temp_args1[0].name = XtNiconX;
	    temp_args1[1].name = XtNiconY;
	    XGeometry(DISP, scr, icon_geometry, "", 0, 0, 0, 0, 0,
		      &temp_args1[0].value, &temp_args1[1].value, &junk, &junk);
	    XtSetValues(top_level, temp_args1, 2);
	}

	if ((font_path = getenv(FONT_PATH)) == NULL)
	    font_path = default_font_path;
	else if (*font_path == ':')
		/*concatenate default_font_path before font_path */
	    font_path = strcat(strcpy(malloc((unsigned)
		strlen(default_font_path) + strlen(font_path) + 1),
		default_font_path), font_path);

	open_dvi_file();
	if (curr_page) {
		current_page = (*curr_page ? atoi(curr_page) : total_pages) - 1;
		if (current_page < 0 || current_page >= total_pages) usage();
	}

		/* Set default window size and icon */
	XtGetValues(top_level, &temp_args2, 1);	/* get border width */
	screen_w = WidthOfScreen(SCRN) - 2*bwidth;
	screen_h = HeightOfScreen(SCRN) - 2*bwidth;
	temp_args3[0].value = (XtArgVal) (page_w<screen_w ? page_w : screen_w);
	temp_args3[1].value = (XtArgVal) (page_h<screen_h ? page_h : screen_h);
	temp_args3[2].value = (XtArgVal) (XCreateBitmapFromData(DISP,
				RootWindowOfScreen(SCRN),
				xdvi_bits, xdvi_width, xdvi_height));
	XtSetValues(top_level, temp_args3, 3);

	vport_widget = XtCreateManagedWidget("vport", viewportWidgetClass,
		top_level, vport_args, XtNumber(vport_args));
	clip_widget = XtNameToWidget(vport_widget, "clip");
	set_draw_args();
	draw_widget = XtCreateManagedWidget("drawing", simpleWidgetClass,
		vport_widget, draw_args, XtNumber(draw_args));
	XtAddEventHandler(vport_widget, KeyPressMask, 0, handle_key,
		(caddr_t) NULL);
	XtAddEventHandler(draw_widget, ExposureMask, GraphicsExpose, handle_exp,
		(caddr_t) &mane);
	XtAddEventHandler(draw_widget, ButtonPressMask, 0, handle_button,
		(caddr_t) NULL);
	XtAddEventHandler(draw_widget, ButtonReleaseMask, 0, handle_release,
		(caddr_t) NULL);
	XtRealizeWidget(top_level);
	curr.win = mane.win = XtWindow(draw_widget);
	/* unmapped = True;	(it was initialized this way) */
	home();			/* no need to unmap at this stage */
	unmapped = False;

	if (reverse) {
	    if (!fore_color) fore_args.value = WhitePixelOfScreen(SCRN);
	    if (!back_color) back_args.value = BlackPixelOfScreen(SCRN);
	    fore_color = back_color = (char *)1;	/* nonzero */
	} else {
	    if (!fore_color) fore_args.value = BlackPixelOfScreen(SCRN);
	    if (!back_color) back_args.value = WhitePixelOfScreen(SCRN);
	}
	if (fore_color) XtSetValues(draw_widget, &fore_args, 1);
	if (back_color) {
	    XtSetValues(draw_widget, &back_args, 1);
	    XtSetValues(clip_widget, &back_args, 1);
	}
	{
	    XGCValues	values;
	    Pixel	set_bits = (Pixel) (fore_args.value & ~back_args.value);
	    Pixel	clr_bits = (Pixel) (back_args.value & ~fore_args.value);
#define	MakeGC(fcn, fg, bg)	(values.function = fcn, values.foreground=fg,\
		values.background=bg,\
		XCreateGC(DISP, RootWindowOfScreen(SCRN),\
			GCFunction|GCForeground|GCBackground, &values))

	    if (set_bits && clr_bits)
		ruleGC = MakeGC(GXcopy, fore_args.value, back_args.value);
	    if (!thorough && ruleGC) {
		foreGC = ruleGC;
		puts("Note:  overstrike characters may be incorrect.");
	    }
	    else {
		if (set_bits) foreGC = MakeGC(GXor, set_bits, 0);
		if (clr_bits)
		    *(foreGC ? &foreGC2 : &foreGC) =
			MakeGC(GXandInverted, clr_bits, 0);
		if (!ruleGC) ruleGC = foreGC;
	    }
	    highGC = ruleGC;
	    if (high_color)
		highGC = MakeGC(GXcopy, hl_Color.pixel, back_args.value);
	}

	ready_cursor = XCreateFontCursor(DISP, XC_cross);
	redraw_cursor = XCreateFontCursor(DISP, XC_watch);
	if (!curs_color)
	    if (high_color) cr_Color = hl_Color;
	    else {
		cr_Color.pixel = fore_args.value;
		XQueryColor(DISP, DefaultColormapOfScreen(SCRN), &cr_Color);
	    }
	{
	    XColor bg_Color;
	    bg_Color.pixel = back_args.value;
	    XQueryColor(DISP, DefaultColormapOfScreen(SCRN), &bg_Color);
	    XRecolorCursor(DISP, ready_cursor, &cr_Color, &bg_Color);
	    XRecolorCursor(DISP, redraw_cursor, &cr_Color, &bg_Color);
	}

	image = XCreateImage(DISP, DefaultVisualOfScreen(SCRN), 1, XYBitmap, 0,
			     (char *)NULL, 0, 0, BITS_PER_BMUNIT, 0);
	image->bitmap_unit = BITS_PER_BMUNIT;
#ifndef	MSBITFIRST
	image->bitmap_bit_order = LSBFirst;
#else	MSBITFIRST
	image->bitmap_bit_order = MSBFirst;
#endif	MSBITFIRST
	{
	    short endian = (MSBFirst << 8) + LSBFirst;
	    image->byte_order = *((char *) &endian);
	}

	do_pages();
}

#else X10
static	char	*display;
static	char	*brdr_color;
static	char	*geometry;

static	struct option {
	char	*name;
	char	*resource;
	enum	{FalseArg, TrueArg, StickyArg, SepArg} argclass;
	enum	{BooleanArg, StringArg, NumberArg} argtype;
	caddr_t	address;
}	options[] = {
		/* the display option MUST be first */
{"-display",	NULL,		SepArg,	StringArg,	(caddr_t) &display},
{"-d",		"debugLevel",	SepArg,	StringArg,	(caddr_t) &debug_arg},
{"+",		NULL,		StickyArg, StringArg,	(caddr_t) &curr_page},
{"-s",		"shrinkFactor", SepArg, NumberArg,	(caddr_t) &shrink_factor},
{"-S",		"densityPercent", SepArg, NumberArg,	(caddr_t) &density},
{"-p",		"pixelsPerInch", SepArg, NumberArg,	(caddr_t) &pixels_per_inch},
{"-margins",	"margins",	SepArg,	StringArg,	(caddr_t) &margins},
{"-sidemargin",	"sideMargin",	SepArg,	StringArg,	(caddr_t) &sidemargin},
{"-topmargin",	"topMargin",	SepArg,	StringArg,	(caddr_t) &topmargin},
{"-l",		"listFonts",	TrueArg, BooleanArg,	(caddr_t) &list_fonts},
{"+l",		NULL,		FalseArg, BooleanArg,	(caddr_t) &list_fonts},
{"-rv",		"reverseVideo",	TrueArg, BooleanArg,	(caddr_t) &reverse},
{"+rv",		NULL,		FalseArg, BooleanArg,	(caddr_t) &reverse},
{"-bw",		"borderWidth",	SepArg,	NumberArg,	(caddr_t) &bwidth},
{"-borderwidth", NULL,		SepArg,	NumberArg,	(caddr_t) &bwidth},
{"-fg",		"foreground",	SepArg,	StringArg,	(caddr_t) &fore_color},
{"-foreground",	NULL,		SepArg,	StringArg,	(caddr_t) &fore_color},
{"-bg",		"background",	SepArg,	StringArg,	(caddr_t) &back_color},
{"-background",	NULL,		SepArg,	StringArg,	(caddr_t) &back_color},
{"-hl",		"highlight",	SepArg,	StringArg,	(caddr_t) &high_color},
{"-bd",		"borderColor",	SepArg,	StringArg,	(caddr_t) &brdr_color},
{"-cr",		"cursorColor",	SepArg,	StringArg,	(caddr_t) &curs_color},
{"-geometry",	"geometry",	SepArg,	StringArg,	(caddr_t) &geometry},
{"=",		NULL,		StickyArg, StringArg,	(caddr_t) &geometry},
{"-mgs",	NULL,		SepArg, NumberArg,	(caddr_t) &mg_size[0]},
{"-mgs1",	"magnifierSize1",SepArg, NumberArg,	(caddr_t) &mg_size[0]},
{"-mgs2",	"magnifierSize2",SepArg, NumberArg,	(caddr_t) &mg_size[1]},
{"-mgs3",	"magnifierSize3",SepArg, NumberArg,	(caddr_t) &mg_size[2]},
};

/*
 *	X10 main program
 */

main(argc, argv)
	int argc;
	char **argv;
{
	char	**arg;
	char	**argvend = argv + argc;
	char	*optstring;
	caddr_t	addr;
	struct option *opt, *candidate;
	int	len1, len2, matchlen;
	OpaqueFrame frame;
	char	def[32];
	int	mouspix;
	Color	cdef;

	prog = *argv;
	if (*prog == '/') prog = rindex(prog, '/') + 1;
/*
 *	Process the option table.  This is not guaranteed for all possible
 *	option tables, but at least it works for this one.
 */
	for (arg = argv + 1; arg < argvend; ++arg) {
	    len1 = strlen(*arg);
	    candidate = NULL;
	    matchlen = 0;
	    for (opt = options; opt < options + XtNumber(options); ++opt) {
		len2 = strlen(opt->name);
		if (opt->argclass == StickyArg) {
		    if (matchlen <= len2 && !strncmp(*arg, opt->name, len2)) {
			candidate = opt;
			matchlen = len2;
		    }
		}
		else if (len1 <= len2 && matchlen <= len1 &&
		    !strncmp(*arg, opt->name, len1)) {
		    if (len1 == len2) {
			candidate = opt;
			break;
		    }
		    if (matchlen < len1) candidate = opt;
		    else if (candidate && candidate->argclass != StickyArg)
			candidate = NULL;
		    matchlen = len1;
		}
	    }
	    if (candidate == NULL) {
		if (**arg == '-') usage();
		if (index(*arg, ':') != NULL) {	/* display */
		    --arg;
		    candidate = options;
		}
		else if (dvi_name) usage();
		else {
		    dvi_name = *arg;
		    continue;
		}
	    }
	    addr = candidate->address;
	    for (opt = options; opt < options + XtNumber(options); ++opt)
		if (opt->address == addr) opt->resource = NULL;
	    switch (candidate->argclass) {
		case FalseArg:	*((Boolean *) addr) = False; break;
		case TrueArg:	*((Boolean *) addr) = True; break;
		case StickyArg:	optstring = *arg + strlen(candidate->name);
		    break;
		case SepArg:
		    ++arg;
		    if (arg >= argvend) usage();
		    optstring = *arg;
		    break;
	    }
	    switch (candidate->argtype) {
		case StringArg:	*((char **) addr) = optstring; break;
		case NumberArg:	*((int *) addr) = atoi(optstring); break;
	    }
	}

	if (XOpenDisplay(display) == NULL)
	    oops("Can't open display\n");
	for (opt = options; opt < options + XtNumber(options); ++opt)
	    if (opt->resource && (optstring = XGetDefault(prog, opt->resource)))
		switch (opt->argtype) {
		    case StringArg:
			*((char **) opt->address) = optstring;
			break;
		    case NumberArg:
			*((int *) opt->address) = atoi(optstring);
			break;
		    case BooleanArg:
			*((Boolean *) opt->address) =
			    (strcmp(optstring, "on") == 0);
		}

	if (shrink_factor <= 0 || density <= 0 || pixels_per_inch <= 0 ||
		dvi_name == NULL) usage();
	if (shrink_factor != 1) bak_shrink = shrink_factor;
	mane.shrinkfactor = shrink_factor;
	if (debug_arg != NULL)
	    debug = isdigit(*debug_arg) ? atoi(debug_arg) : DBG_ALL;
	if (margins) home_x = home_y = atof(margins) * pixels_per_inch;
	if (sidemargin) home_x = atof(sidemargin) * pixels_per_inch;
	if (topmargin) home_y = atof(topmargin) * pixels_per_inch;

	if ((font_path = getenv(FONT_PATH)) == NULL)
	    font_path = default_font_path;
	else if (*font_path == ':')
		/*concatenate default_font_path before font_path */
	    font_path = strcat(strcpy(malloc((unsigned)
		strlen(default_font_path) + strlen(font_path) + 1),
		default_font_path), font_path);

	open_dvi_file();
	if (curr_page) {
		current_page = (*curr_page ? atoi(curr_page) : total_pages) - 1;
		if (current_page < 0 || current_page >= total_pages) usage();
	}

	if (reverse) {
		foreGC = WhitePixel;
		highGC = WhitePixel;
		backpix = BlackPixel;
		backmap = BlackPixmap;
		bdrmap = WhitePixmap;
		mouspix = WhitePixel;
		GXfunc = GXor;
	} else {
		foreGC = BlackPixel;
		highGC = BlackPixel;
		backpix = WhitePixel;
		backmap = WhitePixmap;
		bdrmap = BlackPixmap;
		mouspix = BlackPixel;
		GXfunc = GXand;
	}
	if (DisplayCells() > 2) {
		if (fore_color && XParseColor(fore_color, &cdef) &&
			XGetHardwareColor(&cdef))
			foreGC = cdef.pixel;
		if (back_color && XParseColor(back_color, &cdef) &&
			XGetHardwareColor(&cdef)) {
			backpix = cdef.pixel;
			backmap = XMakeTile(backpix);
		}
		if (high_color && XParseColor(high_color, &cdef) &&
			XGetHardwareColor(&cdef))
			highGC = cdef.pixel;
		if (brdr_color && XParseColor(brdr_color, &cdef) &&
			XGetHardwareColor(&cdef))
			bdrmap = XMakeTile(cdef.pixel);
		if (curs_color && XParseColor(curs_color, &cdef) &&
			XGetHardwareColor(&cdef))
			mouspix = cdef.pixel;
	}

	frame.bdrwidth = bwidth;
	screen_w = DisplayWidth() - 2*bwidth;
	screen_h = DisplayHeight() - 2*bwidth;
	frame.width = (page_w < screen_w ? page_w : screen_w);
	frame.height = (page_h < screen_h ? page_h : screen_h);
	frame.border = bdrmap;
	frame.background = backmap;
	frame.x = 0;
	frame.y = 0;
	Sprintf(def, "=%dx%d+0+0", frame.width, frame.height);
	mane.win = XCreate("DVI Previewer", prog, geometry, def,
		&frame, 50, 50);
	window_w = frame.width;
	window_h = frame.height;
	XSelectInput(mane.win,
		KeyPressed|ButtonPressed|ButtonReleased|
		ExposeWindow|ExposeRegion);
	XMapWindow(mane.win);
	XDefineCursor(mane.win,
	    XCreateCursor(xdvi_width, xdvi_height, xdvi_bits, xdvi_mask_bits,
			  xdvi_x_hot, xdvi_y_hot, mouspix, backpix, GXcopy));
	do_pages();
}
#endif X10
