/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>

#include "common.h"
#include "config_wrapper.h"
#include "kbindings.h"
#include "videowin.h"
#include "actions.h"
#include "event.h"
#include "errors.h"
#include "osd.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/button.h"
#include "xine-toolkit/browser.h"
#include "xine-toolkit/font.h"
#include "xine-toolkit/label.h"
#include "xine-toolkit/labelbutton.h"
#include "kbindings_common.h"

#undef TRACE_KBINDINGS

#define _DYN_STRING_STEP 4096

typedef struct {
  char *buf;
  unsigned int used, have;
} _dyn_string_t;

static void _dyn_string_init (_dyn_string_t *ds) {
  ds->buf = malloc (_DYN_STRING_STEP);
  if (ds->buf) {
    ds->buf[0] = 0;
    ds->have = _DYN_STRING_STEP;
  } else {
    ds->have = 0;
  }
  ds->used = 0;
}

static void _dyn_string_clear (_dyn_string_t *ds) {
  if (ds->buf)
    ds->buf[0] = 0;
  ds->used = 0;
}

static void _dyn_string_deinit (_dyn_string_t *ds) {
  SAFE_FREE (ds->buf);
  ds->used = 0;
  ds->have = 0;
}

static char *_dyn_string_get (_dyn_string_t *ds) {
  if (ds->used) {
    ds->used += 1;
    ds->buf[ds->used] = 0;
  }
  return (char *)(uintptr_t)ds->used;
}

static int _dyn_string_size (_dyn_string_t *ds, size_t s) {
  size_t ln = ds->used + s;
  char *ns;
  if (ln + 2 <= ds->have)
    return 1;
  ln += 2 + _DYN_STRING_STEP - 1;
  ln &= ~(_DYN_STRING_STEP - 1);
  ns = realloc (ds->buf, ln);
  if (!ns)
    return 0;
  ds->buf = ns;
  ds->have = ln;
  return 1;
}

static void _dyn_string_append (_dyn_string_t *ds, const char *s, size_t ls) {
  memcpy (ds->buf + ds->used, s, ls + 1);
  ds->used += ls;
}

static void _dyn_string_fix (_dyn_string_t *ds, const char **s, uint32_t n) {
  while (n) {
    if ((uintptr_t)*s < ds->used)
      *s = ds->buf + (uintptr_t)*s;
    s++;
    n--;
  }
}

typedef enum {
  _W_browser = 0,
  /* keep order */
  _W_edit,
  _W_alias,
  _W_delete,
  _W_grab,
  /* /keep order */
  _W_save,
  _W_reset,
  _W_done,
  _W_comment,
  _W_key,
  /* keep order */
  _W_ctrl,
  _W_meta,
  _W_mod3,
  _W_mod4,
  _W_mod5,
  /* /keep order */
  _W_impl_alias_box,
  _W_impl_alias_label,
  _W_enab_box,
  _W_enab_label,
  _W_LAST
} _W_t;

struct xui_keyedit_s {
  gui_new_window_t      nw;

  kbinding_t           *kbt;

  int                   nsel;
  const kbinding_entry_t *ksel;

  xitk_widget_t        *w[_W_LAST];

  int                    (*exit) (xui_keyedit_t *kbedit);

  int                   num_entries;
  _dyn_string_t         ebuf;

  int                   grabbing;
  enum {
    KBEDIT_NOOP = 0,
    KBEDIT_ALIASING,
    KBEDIT_EDITING
  }                     action_wanted;

  struct {
    xitk_register_key_t key;
    kbinding_entry_t    entry;
    xitk_window_t      *xwin;
  }                     kbr;
  char                  key_buf[128];
  const char          **shortcuts, *entries[2 * KBT_MAX_ENTRIES];
};

#define WINDOW_WIDTH        576
#define WINDOW_HEIGHT       458
#define MAX_DISP_ENTRIES    11

/*
  Remap file entries syntax are:
  ...
  WindowReduce {
      key = less
      modifier = none
  }

  Alias {
      entry = WindowReduce
      key = w
      modifier = control
  }
  ...
  WindowReduce action key is <less>, with no modifier usage.
  There is an alias: C-w (control w> keystroke will have same
                     action than WindowReduce.

  modifier entry is optional.
  modifier key can be multiple:
 ...
  WindowReduce {
      key = less
      modifier = control, meta
  }
  ...
  Here, WindowReduce action keystroke will be CM-<

  Key modifier name are:
    none, control, ctrl, meta, alt, mod3, mod4, mod5.

  shift,lock and numlock modifier are voluntary not handled.

*/

static void _kbindings_display_kbindings_to_stream (gGui_t *gui, kbinding_t *kbt, FILE *stream, kbindings_dump_mode_t mode) {
  int i, num_entries = kbindings_get_num_entries (kbt);

  if(kbt == NULL) {
    gui_msg (gui, XUI_MSG_ERROR, _("OOCH: key binding table is NULL.\n"));
    return;
  }

  switch(mode) {
  case KBT_DISPLAY_MODE_LIRC:
    fprintf(stream, "##\n# xine key bindings.\n"
		    "# Automatically generated by %s version %s.\n##\n\n", PACKAGE, VERSION);

    for (i = 0; i < num_entries; i++) {
      char buf[2048], *p = buf, *e = buf + sizeof (buf) - 2 - 7 - 16 - 16 - 15 - 12 - 10 - 7;
      const kbinding_entry_t *entry = kbindings_entry_from_index (kbt, i);
      if (!(entry->flags & KBE_FLAG_alias)) {
        memcpy (p, "# ", 2); p += 2; e += 2;
        p += strlcpy (p, entry->comment, e - p);
        if (p > e)
          p = e;
        memcpy (p, "\nbegin\n"
                   "\tremote = xxxxx\n"
                   "\tbutton = xxxxx\n"
                   "\tprog   = xine\n"
                   "\trepeat = 0\n"
                   "\tconfig = ", 7 + 16 + 16 + 15 + 12 + 10);
        p += 7 + 16 + 16 + 15 + 12 + 10;
        e += 7 + 16 + 16 + 15 + 12 + 10;
        p += strlcpy (p, entry->name, e - p);
        if (p > e)
          p = e;
        memcpy (p, "\nend\n\n", 7); p += 7;
        fputs (buf, stream);
      }
    }
    fprintf(stream, "##\n# End of xine key bindings.\n##\n");
    break;

  case KBT_DISPLAY_MODE_MAN:
    /* NOTE: this replaces kbinding_man, which would need to link in most stuff now anyway. */

    for (i = 0; i < num_entries; i++) {
      char buf[2048], *p = buf, *e = buf + sizeof (buf) - 5 - 2 * 9 - 3 * 10 - 3 - 2 - 7 - 2 - 5 - 1;
      const kbinding_entry_t *entry = kbindings_entry_from_index (kbt, i);

      if (entry->flags & (KBE_FLAG_alias | KBE_FLAG_void))
        continue;

      memcpy (p, ".IP \"", 5); p += 5; e += 5;
      if (entry->modifier != MODIFIER_NOMOD) {
        if (entry->modifier & MODIFIER_CTRL) {
          memcpy (p, "\\fBC\\fP\\-", 9); p += 9; e += 9;
        }
        if (entry->modifier & MODIFIER_META) {
          memcpy (p, "\\fBM\\fP\\-", 9); p += 9; e += 9;
        }
        if (entry->modifier & MODIFIER_MOD3) {
          memcpy (p, "\\fBM3\\fP\\-", 10); p += 10; e += 10;
        }
        if (entry->modifier & MODIFIER_MOD4) {
          memcpy (p, "\\fBM4\\fP\\-", 10); p += 10; e += 10;
        }
        if (entry->modifier & MODIFIER_MOD5) {
          memcpy (p, "\\fBM5\\fP\\-", 10); p += 10; e += 10;
        }
      }
      memcpy (p, "\\fB", 3); p += 3; e += 3;
      if (entry->key[0] && entry->key[1]) {
        memcpy (p, "\\<", 2); p += 2; e += 2;
      }
      if (!strncmp (entry->key, "KP_", 3)) {
        memcpy (p, "Keypad ", 7); p += 7; e += 7;
        p += strlcpy (p, entry->key + 3, e - p);
      } else {
        p += strlcpy (p, entry->key, e - p);
      }
      if (p > e)
        p = e;
      if (entry->key[0] && entry->key[1]) {
        memcpy (p, "\\>", 2); p += 2; e += 2;
      }
      memcpy (p, "\\fP\"\n", 5); p += 5; e += 5;
      p += strlcpy (p, entry->comment, e - p);
      if (p > e)
        p = e;
      *p++ = '\n';
      *p = 0;
      fputs (buf, stream);
    }
    break;

  default:
    fprintf(stream, "##\n# xine key bindings.\n"
		    "# Automatically generated by %s version %s.\n##\n\n", PACKAGE, VERSION);

    for (i = 0; i < num_entries; i++) {
      char buf[2048], *p = buf, *e = buf + sizeof (buf) - 2 - 18 - 1 - 7 - 13 - 9 - 4 * 6 - 5;
      const kbinding_entry_t *entry = kbindings_entry_from_index (kbt, i);

      memcpy (p, "# ", 2); p += 2; e += 2;
      p += strlcpy (p, entry->comment, e - p);
      if (p > e)
        p = e;
      if (entry->flags & KBE_FLAG_alias) {
        memcpy (p, "\nAlias {\n\tentry = ", 18); p += 18; e += 18;
        p += strlcpy (p, entry->name, e - p);
        if (p > e)
          p = e;
        *p++ = '\n'; e += 1;
      } else {
        *p++ = '\n';
        p += strlcpy (p, entry->name, e - p);
        memcpy (p, " {\n", 3); p += 3; e += 3;
      }
      memcpy (p, "\tkey = ", 7); p += 7; e += 7;
      p += strlcpy (p, entry->key, e - p);
      if (p > e)
        p = e;
      memcpy (p, "\n\tmodifier = ", 13); p += 13;
      if (entry->modifier == MODIFIER_NOMOD) {
        memcpy (p, "none, ", 6); p += 6;
      } else {
        if (entry->modifier & MODIFIER_CTRL) {
          memcpy (p, "control, ", 9); p += 9;
        }
        if (entry->modifier & MODIFIER_META) {
          memcpy (p, "meta, ", 6); p += 6;
        }
        if (entry->modifier & MODIFIER_MOD3) {
          memcpy (p, "mod3, ", 6); p += 6;
        }
        if (entry->modifier & MODIFIER_MOD4) {
          memcpy (p, "mod4, ", 6); p += 6;
        }
        if (entry->modifier & MODIFIER_MOD5) {
          memcpy (p, "mod5, ", 6); p += 6;
        }
      }
      p -= 2;
      memcpy (p, "\n}\n\n", 5); p += 5;
      fputs(buf, stream);
    }
    fprintf(stream, "##\n# End of xine key bindings.\n##\n");
    break;
  }

}

/*
 * Save current key binding table to keymap file.
 */
void kbindings_save (gGui_t *gui, kbinding_t *kbt, const char *filename) {
  FILE   *f;

  if (!kbt || !filename)
    return;
  f = fopen (filename, "w+");
  if (!f)
    return;
  _kbindings_display_kbindings_to_stream (gui, kbt, f, KBT_DISPLAY_MODE_DEFAULT);
  fclose (f);
}

/*
 * This could be used to create a default key binding file
 * with 'xine --keymap > $HOME/.xine_keymap'
 */
void kbindings_dump (gGui_t *gui, kbindings_dump_mode_t mode) {
  kbinding_t *kbt;
  int f = 1;

  switch (mode) {
    case KBT_DISPLAY_MODE_CURRENT:
    case KBT_DISPLAY_MODE_MAN:
      kbt = gui->kbindings;
      if (kbt) {
        f = 0;
        break;
      }
    /* fall through */
    case KBT_DISPLAY_MODE_DEFAULT:
    case KBT_DISPLAY_MODE_LIRC:
    default:
      kbt = kbindings_new (gui, NULL);
  }
  _kbindings_display_kbindings_to_stream (gui, kbt, stdout, mode);
  if (f)
    kbindings_delete (&kbt);
}

/*
 * Return action id from key binding kbt entry.
 */
action_id_t kbindings_aid_from_name (kbinding_t *kbt, const char *name_or_id) {
  const kbinding_entry_t *entry = kbindings_entry_from_name (kbt, name_or_id);

  return entry ? entry->id : ACTID_NOKEY;
}

static size_t _kbindings_shortcut_from_name_from_kbe (const char * const *mod_names,
  const kbinding_entry_t *kbe, char *shortcut, size_t shortcut_size, int style) {
  char *q = shortcut, *e = q + shortcut_size;

  do {
    if (kbe->flags & KBE_FLAG_void)
      break;
    if (q + 1 >= e)
      break;
    *q++ = '[';
    if (style == 0) {
      if (kbe->modifier & MODIFIER_CTRL) {
        q += strlcpy (q, mod_names[0], e - q);
        if (q + 1 >= e)
          break;
        *q++ = '+';
      }
      if (kbe->modifier & MODIFIER_META) {
        q += strlcpy (q, mod_names[1], e - q);
        if (q + 1 >= e)
          break;
        *q++ = '+';
      }
      if (kbe->modifier & MODIFIER_MOD3) {
        q += strlcpy (q, mod_names[2], e - q);
        if (q + 1 >= e)
          break;
        *q++ = '+';
      }
      if (kbe->modifier & MODIFIER_MOD4) {
        q += strlcpy (q, mod_names[3], e - q);
        if (q + 1 >= e)
          break;
        *q++ = '+';
      }
      if (kbe->modifier & MODIFIER_MOD5) {
        q += strlcpy (q, mod_names[4], e - q);
        if (q + 1 >= e)
          break;
        *q++ = '+';
      }
    } else {
      if (kbe->modifier & MODIFIER_CTRL) {
        if (q + 2 >= e)
          break;
        memcpy (q, "C-", 3);
        q += 2;
      }
      if (kbe->modifier & MODIFIER_META) {
        if (q + 2 >= e)
          break;
        memcpy (q, "M-", 3);
        q += 2;
      }
      if (kbe->modifier & MODIFIER_MOD3) {
        if (q + 3 >= e)
          break;
        memcpy (q, "M3-", 4);
        q += 3;
      }
      if (kbe->modifier & MODIFIER_MOD4) {
        if (q + 3 >= e)
          break;
        memcpy (q, "M4-", 4);
        q += 3;
      }
      if (kbe->modifier & MODIFIER_MOD5) {
        if (q + 3 >= e)
          break;
        memcpy (q, "M5-", 4);
        q += 3;
      }
    }
    q += strlcpy (q, kbe->key, e - q);
    if (q + 2 >= e)
      break;
    *q++ = ']';
  } while (0);
  if (q >= e)
    q = e - 1;
  *q = 0;
  return q - shortcut;
}

size_t kbindings_shortcut_from_name (kbinding_t *kbt, const char *action, char *buf, size_t buf_size, int style) {
  const kbinding_entry_t  *k;

  if (!kbt || !action || !buf || !buf_size)
    return 0;
  if (!(k = kbindings_entry_from_name (kbt, action)))
    return 0;
  return _kbindings_shortcut_from_name_from_kbe (kbindings_modifier_names (kbt), k, buf, buf_size, style);
}

/*
 * Handle key event from an XEvent.
 */
action_id_t kbindings_aid_from_be_event (kbinding_t *kbt, const xitk_be_event_t *event, gGui_t *gui) {
  char buf[256];
  const kbinding_entry_t *entry;
  uint32_t qual;

  if (!kbt || !event)
    return 0;

  if (xitk_be_event_name (event, buf, sizeof (buf)) < 1)
    return 0;

#ifdef TRACE_KBINDINGS
  printf ("Looking for: '%s' [", buf);
  if (event->qual == MODIFIER_NOMOD)
    printf ("none, ");
  if (event->qual & MODIFIER_CTRL)
    printf ("control, ");
  if (event->qual & MODIFIER_META)
    printf ("meta, ");
  if (event->qual & MODIFIER_MOD3)
    printf ("mod3, ");
  if (event->qual & MODIFIER_MOD4)
    printf ("mod4, ");
  if (event->qual & MODIFIER_MOD5)
    printf ("mod5, ");
  printf ("\b\b]\n");
#endif

  qual = event->qual & (MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD3 | MODIFIER_MOD4 | MODIFIER_MOD5);
  entry = kbindings_entry_from_key (kbt, buf, qual);

  if (entry && ((entry->modifier == qual) || (gui->kbindings_enabled & 2))) {
    int id = (!(entry->flags & KBE_FLAG_gui) || !(gui->flags & XUI_FLAG_no_gui)) ? entry->id : 0;
    /* keyboard lock: still allow revoke. */
    if ((gui->kbindings_enabled & 1) || (id == ACTID_KBENABLE))
      return id;
    /* avoid code duolication. */
    gui_action_args (gui, ACTID_KBENABLE, 0, NULL);
    return 0;
  }

  if ((event->type == XITK_EV_KEY_DOWN) && (gui->verbosity >= 1)) {
    static const uint8_t ign[XITK_KEY_LASTCODE] = {
      [XITK_KEY_CTRL]        = 1,
      [XITK_KEY_SHIFT]       = 1,
      [XITK_KEY_ALT]         = 1,
      [XITK_KEY_ALTGR]       = 1,
      [XITK_KEY_SUPER_L]     = 1,
      [XITK_KEY_SUPER_R]     = 1,
      [XITK_MOUSE_WHEEL_UP]  = 1,
      [XITK_MOUSE_WHEEL_DOWN]= 1
    };
    const uint8_t *s = (const uint8_t *)event->utf8;
    if (!((s[0] == XITK_CTRL_KEY_PREFIX) && ign[s[1]])) {
      kbinding_entry_t e2;
      char kstr[256];
      e2.key = buf;
      e2.modifier = event->qual;
      e2.flags = 0;
      _kbindings_shortcut_from_name_from_kbe (kbindings_modifier_names (kbt), &e2, kstr, sizeof (kstr), gui->shortcut_style);
      osd_message (gui, "%s ?", kstr);
    }
  }
  return 0;
}

/*
 * ***** Key Binding Editor ******
 */

static void kbedit_create_browser_entries (xui_keyedit_t *kbedit) {
  const char * const *mod_names = kbindings_modifier_names (kbedit->kbt);
  int i;

  _dyn_string_clear (&kbedit->ebuf);

  kbedit->num_entries = kbindings_get_num_entries (kbedit->kbt);
  kbedit->shortcuts = kbedit->entries + kbedit->num_entries;

  for (i = 0; i < kbedit->num_entries; i++) {
    /* beware of "Ctrl+Alt+Mod3+Mod4+Mod5+XF86AudioLowerVolume" ;-) */
    char  shortcut[256], *q = shortcut;
    const kbinding_entry_t *entry = kbindings_entry_from_index (kbedit->kbt, i);
    size_t lc = (entry->flags & KBE_FLAG_alias) ? xitk_find_byte (entry->comment, 0) + 3 : 0;

    if (!(entry->flags & KBE_FLAG_default))
      *q++ = '*';
    q += _kbindings_shortcut_from_name_from_kbe (mod_names, entry,
      q, sizeof (shortcut) - 1, kbedit->nw.gui->shortcut_style);
    if (!_dyn_string_size (&kbedit->ebuf, lc + (q - shortcut) + 1))
      break;

    if (entry->flags & KBE_FLAG_alias) {
      kbedit->entries[i] = _dyn_string_get (&kbedit->ebuf);
      _dyn_string_append (&kbedit->ebuf, "@{", 2);
      _dyn_string_append (&kbedit->ebuf, entry->comment, lc - 3);
      _dyn_string_append (&kbedit->ebuf, "}", 1);
    } else {
      kbedit->entries[i] = entry->comment;
    }
    kbedit->shortcuts[i] = _dyn_string_get (&kbedit->ebuf);
    _dyn_string_append (&kbedit->ebuf, shortcut, q - shortcut);
  }
  for (; i < kbedit->num_entries; i++)
    kbedit->entries[i] = kbedit->shortcuts[i] = NULL;
  _dyn_string_fix (&kbedit->ebuf, kbedit->entries, 2 * kbedit->num_entries);
  if (kbedit->nw.gui->verbosity >= 2)
    printf ("gui.kbindings.edit.list (%d, string buf %u/%u).\n",
      kbedit->num_entries, kbedit->ebuf.used, kbedit->ebuf.have);
}

static void kbedit_display_kbinding (xui_keyedit_t *kbedit, const char *action, const kbinding_entry_t *kbe) {
  if (kbe) {
    if (action) 
      xitk_label_change_label (kbedit->w[_W_comment], action);
    xitk_label_change_label (kbedit->w[_W_key], (kbe->flags & KBE_FLAG_void) ? _("None") : kbe->key);

    xitk_widgets_state (kbedit->w + _W_ctrl, 1, XITK_WIDGET_STATE_ON,
      xitk_bitmove (kbe->modifier, MODIFIER_CTRL, XITK_WIDGET_STATE_ON));
    xitk_widgets_state (kbedit->w + _W_meta, 1, XITK_WIDGET_STATE_ON,
      xitk_bitmove (kbe->modifier, MODIFIER_META, XITK_WIDGET_STATE_ON));
    xitk_widgets_state (kbedit->w + _W_mod3, 1, XITK_WIDGET_STATE_ON,
      xitk_bitmove (kbe->modifier, MODIFIER_MOD3, XITK_WIDGET_STATE_ON));
    xitk_widgets_state (kbedit->w + _W_mod4, 1, XITK_WIDGET_STATE_ON,
      xitk_bitmove (kbe->modifier, MODIFIER_MOD4, XITK_WIDGET_STATE_ON));
    xitk_widgets_state (kbedit->w + _W_mod5, 1, XITK_WIDGET_STATE_ON,
      xitk_bitmove (kbe->modifier, MODIFIER_MOD5, XITK_WIDGET_STATE_ON));
  }
}

/*
 *
 */
static void kbedit_unset (xui_keyedit_t *kbedit) {
  xitk_widgets_state (kbedit->w + _W_reset, 1, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, ~0u);
  /* edit, alias, delete, grab */
  xitk_widgets_state (kbedit->w + _W_edit, 4, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_ON, 0);
  kbedit->ksel = NULL;
  kbedit->nsel = -1;

  xitk_label_change_label (kbedit->w[_W_comment], _("Nothing selected"));
  xitk_label_change_label (kbedit->w[_W_key], _("None"));

  xitk_widgets_state (kbedit->w + _W_ctrl, 5, XITK_WIDGET_STATE_ON, 0);
}

static void _kbedit_set (xui_keyedit_t *kbedit) {
  xitk_widgets_state (kbedit->w + _W_edit, 1, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, ~0u);
  /* alias, delete */
  xitk_widgets_state (&kbedit->w[_W_alias], 2, XITK_WIDGET_STATE_ENABLE,
    xitk_bitmove (~kbedit->ksel->flags, KBE_FLAG_void, XITK_WIDGET_STATE_ENABLE));
  xitk_widgets_state (&kbedit->w[_W_reset], 1, XITK_WIDGET_STATE_ENABLE,
    xitk_bitmove (~kbedit->ksel->flags, KBE_FLAG_default, XITK_WIDGET_STATE_ENABLE));
  kbedit_display_kbinding (kbedit, kbedit->entries[kbedit->nsel], kbedit->ksel);
}

/*
 *
 */
static void kbedit_select (xui_keyedit_t *kbedit, int s) {
  if (kbedit->action_wanted != KBEDIT_NOOP) {
    /* edit, alias, delete, grab */
    xitk_widgets_state (kbedit->w + _W_edit, 4, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_ON, 0);
    kbedit->action_wanted = KBEDIT_NOOP;
  }

  kbedit->nsel = s;
  kbedit->ksel = kbindings_entry_from_index (kbedit->kbt, s);
  _kbedit_set (kbedit);
}

static void _kbr_close (xui_keyedit_t *kbe) {
  if (kbe->kbr.key)
    xitk_unregister_event_handler (kbe->nw.gui->xitk, &kbe->kbr.key);
  if (kbe->kbr.xwin) {
    xitk_window_destroy_window (kbe->kbr.xwin);
    kbe->kbr.xwin = NULL;
  }
  xitk_labelbutton_change_label (kbe->w[_W_grab], _("Grab"));
  xitk_labelbutton_set_state (kbe->w[_W_grab], 0);
  kbe->kbr.entry.key = NULL;
}

/*
 *
 */
static int _kbedit_exit (xui_keyedit_t *kbedit) {
  _kbr_close (kbedit);

  gui_window_delete (&kbedit->nw);

  _dyn_string_deinit (&kbedit->ebuf);

  kbindings_delete (&kbedit->kbt);

  kbedit->nw.gui->ssaver_enabled = 1;

  kbedit->nw.gui->keyedit = NULL;
  free (kbedit);
  return 1;
}

static void kbedit_exit (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_keyedit_t *kbedit = data;

  (void)w;
  (void)state;
  (void)modifier;
  kbedit->exit (kbedit);
}

/*
 *
 */
static void kbedit_sel (xitk_widget_t *w, void *data, int s, unsigned int modifier) {
  xui_keyedit_t *kbedit = data;

  (void)w;
  (void)modifier;
  _kbr_close (kbedit);
  if(s >= 0)
    kbedit_select (kbedit, s);
  else
    kbedit_unset (kbedit);
}

/*
 * Create an alias from the selected entry.
 */
static void kbedit_alias(xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_keyedit_t *kbedit = data;

  (void)w;
  (void)modifier;
  _kbr_close (kbedit);
  xitk_labelbutton_set_state(kbedit->w[_W_edit], 0);

  xitk_widgets_state (kbedit->w + _W_grab, 1, XITK_WIDGET_STATE_ENABLE, state ? ~0u : 0);
  kbedit->action_wanted = state ? KBEDIT_ALIASING : KBEDIT_NOOP;
}

/*
 * Change shortcut, should take care about reduncancy.
 */
static void kbedit_edit(xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_keyedit_t *kbedit = data;

  (void)w;
  (void)modifier;
  _kbr_close (kbedit);
  xitk_labelbutton_set_state(kbedit->w[_W_alias], 0);

  xitk_widgets_state (kbedit->w + _W_grab, 1, XITK_WIDGET_STATE_ENABLE, state ? ~0u : 0);
  kbedit->action_wanted = state ? KBEDIT_EDITING : KBEDIT_NOOP;
}

/*
 * Remove selected entry, but alias ones only.
 */
static void kbedit_delete (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_keyedit_t *kbedit = data;

  (void)w;
  (void)state;
  (void)modifier;
  _kbr_close (kbedit);
  xitk_labelbutton_set_state(kbedit->w[_W_alias], 0);
  xitk_labelbutton_set_state(kbedit->w[_W_edit], 0);
  xitk_widgets_state (kbedit->w + _W_grab, 1, XITK_WIDGET_STATE_ENABLE, 0);

  if (kbedit->nsel >= 0) {
    int r, flags;

    kbedit->ksel = kbindings_entry_from_index (kbedit->kbt, kbedit->nsel);
    flags = kbedit->ksel->flags;
    r = kbindings_entry_set (kbedit->kbt, kbedit->nsel, 0, "VOID");

    if (r == -1) {
      kbedit_create_browser_entries (kbedit);
      xitk_browser_update_list (kbedit->w[_W_browser], kbedit->entries,
        kbedit->shortcuts, kbedit->num_entries, XITK_INT_KEEP);
      if (flags & KBE_FLAG_alias) {
        xitk_browser_set_select (kbedit->w[_W_browser], -1);
        kbedit_unset (kbedit);
      } else {
        _kbedit_set (kbedit);
      }
    }
    /* gone ;-)
    gui_msg (kbedit->nw.gui, XUI_MSG_ERROR, _("You can only delete alias entries."));
    */
  }
}

/*
 * Reset to xine's default table.
 */
static void kbedit_reset (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_keyedit_t *kbedit = data;
  int n1, n2, r;

  (void)w;
  (void)state;
  (void)modifier;
  _kbr_close (kbedit);

  n1 = kbindings_get_num_entries (kbedit->kbt);
  r = kbindings_entry_set (kbedit->kbt, kbedit->nsel, 0, NULL);
  n2 = kbindings_get_num_entries (kbedit->kbt);

  kbedit_create_browser_entries (kbedit);
  xitk_browser_update_list (kbedit->w[_W_browser], (const char * const *) kbedit->entries,
    (const char * const *)kbedit->shortcuts, kbedit->num_entries, XITK_INT_KEEP);
  xitk_labelbutton_set_state (kbedit->w[_W_alias], 0);
  xitk_labelbutton_set_state (kbedit->w[_W_edit], 0);
  xitk_widgets_state (kbedit->w + _W_grab, 1, XITK_WIDGET_STATE_ENABLE, 0);

  if (kbedit->nsel < 0) {
    /* full list reset */
    ;
  } else if (n2 < n1) {
    /* alias deleted */
    xitk_browser_set_select (kbedit->w[_W_browser], -1);
    kbedit_unset (kbedit);
  } else if (r == -1) {
    /* base entry changed */
    kbedit->ksel = kbindings_entry_from_index (kbedit->kbt, kbedit->nsel);
    _kbedit_set (kbedit);
  }
}

/*
 * Save keymap file, then set global table to hacked one
 */
static void kbedit_save (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_keyedit_t *kbedit = data;

  (void)w;
  (void)state;
  (void)modifier;
  _kbr_close (kbedit);
  xitk_labelbutton_set_state(kbedit->w[_W_alias], 0);
  xitk_labelbutton_set_state(kbedit->w[_W_edit], 0);
  xitk_widgets_state (kbedit->w + _W_grab, 1, XITK_WIDGET_STATE_ENABLE, 0);

  kbindings_delete (&kbedit->nw.gui->kbindings);
  kbedit->nw.gui->kbindings = kbindings_dup (kbedit->kbt);
  kbindings_save (kbedit->nw.gui, kbedit->nw.gui->kbindings, kbedit->nw.gui->keymap_file);
}

static void _kbedit_store_1 (xui_keyedit_t *kbe) {
  int r;
  switch (kbe->action_wanted) {
    case KBEDIT_ALIASING:
      r = kbindings_get_num_entries (kbe->kbt);
      if (r >= KBT_MAX_ENTRIES) {
        gui_msg (kbe->nw.gui, XUI_MSG_ERROR, _("No more space for additional entries!"));
        return;
      }
      kbe->kbr.entry.flags |= KBE_FLAG_alias;
      r = kbindings_entry_set (kbe->kbt, ~kbe->nsel, kbe->kbr.entry.modifier, kbe->kbr.entry.key);
      kbe->kbr.entry.key = NULL;
      if (r == -1) {
        kbedit_create_browser_entries (kbe);
        xitk_browser_update_list (kbe->w[_W_browser], (const char * const *) kbe->entries,
          (const char * const *) kbe->shortcuts, kbe->num_entries, XITK_INT_KEEP);
      }
      break;
    case KBEDIT_EDITING:
      kbe->kbr.entry.flags &= ~KBE_FLAG_alias;
      kbe->kbr.entry.flags |= kbe->ksel->flags & KBE_FLAG_alias;
      r = kbindings_entry_set (kbe->kbt, kbe->nsel, kbe->kbr.entry.modifier, kbe->kbr.entry.key);
      kbe->kbr.entry.key = NULL;
      kbe->ksel = kbindings_entry_from_index (kbe->kbt, kbe->nsel);
      if (r == -1) {
        kbedit_create_browser_entries (kbe);
        xitk_browser_update_list (kbe->w[_W_browser], (const char * const *) kbe->entries,
          (const char * const *) kbe->shortcuts, kbe->num_entries, XITK_INT_KEEP);
        _kbedit_set (kbe);
      }
      break;
    default: ;
  }
  if (kbe->nsel < 0) {
    kbedit_unset (kbe);
    kbe->action_wanted = KBEDIT_NOOP;
  }
}

/*
 * Grab key binding.
 */

static int kbr_event (void *data, const xitk_be_event_t *e) {
  xui_keyedit_t *kbe = data;

  if (e->type == XITK_EV_DEL_WIN) {
    _kbr_close (kbe);
    kbe->grabbing = 0;
    return 1;
  }

  if (e->type == XITK_EV_KEY_DOWN) {
    kbe->grabbing = 2;
    return 1;
  }

  if (e->type == XITK_EV_KEY_UP) {
    const kbinding_entry_t *redundant;

    /* 1. user hits grab button with ENTER/SPACE,
     * 2. grab window opens,
     * 3. we get the ENTER/SPACE key up here, and
     * 4. ignore it :-)) */
    if (kbe->grabbing < 2)
      return 0;
    if (xitk_be_event_name (e, kbe->key_buf, sizeof (kbe->key_buf)) < 1)
      return 0;

    _kbr_close (kbe);
    xitk_labelbutton_set_state (kbe->w[_W_grab], 0);
    kbe->grabbing = 0;

    kbe->kbr.entry.comment   = kbe->ksel->comment;
    kbe->kbr.entry.name      = kbe->ksel->name;
    kbe->kbr.entry.id        = kbe->ksel->id;
    kbe->kbr.entry.flags     = kbe->ksel->flags & ~KBE_FLAG_void;
    kbe->kbr.entry.key       = kbe->key_buf;
    kbe->kbr.entry.modifier  = e->qual & (MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD3 | MODIFIER_MOD4 | MODIFIER_MOD5);

    redundant = kbindings_entry_from_key (kbe->kbt, kbe->kbr.entry.key, kbe->kbr.entry.modifier);
    if (redundant && (redundant->modifier == kbe->kbr.entry.modifier) && (redundant->key[0] == kbe->kbr.entry.key[0])) {
      /* error, redundant */
      gui_msg (kbe->nw.gui, XUI_MSG_ERROR, _("This key binding is redundant with action:\n\"%s\".\n"),
        redundant->comment);
      kbe->kbr.entry.key = NULL;
      return 1;
    }

    kbedit_display_kbinding (kbe, NULL, &kbe->kbr.entry);
    xitk_labelbutton_change_label (kbe->w[_W_grab], _("Store new key binding"));
    return 1;
  }

  return 0;
}

static void kbedit_grab (xitk_widget_t *w, void *data, int state, unsigned int qual) {
  xui_keyedit_t *kbe = data;

  (void)w;
  (void)qual;
  if (!kbe)
    return;

  if (kbe->kbr.key) {
    if (!state)
      _kbr_close (kbe);
    return;
  }

  if (kbe->kbr.entry.key) {
    if (state) {
      _kbedit_store_1 (kbe);
      _kbr_close (kbe);
    }
    return;
  }

  if (!state)
    return;

  kbe->grabbing = 1;
  /* xitk_labelbutton_change_label (kbe->w[_W_grab], _("Press Keyboard Keys...")); */
  {
    xitk_rect_t wr = {0, 0, 0, 0};
    xitk_window_get_window_position (kbe->nw.xwin, &wr);
    kbe->kbr.xwin = xitk_window_create_window_ext (kbe->nw.gui->xitk,
      wr.x, wr.y + wr.height, wr.width, 50, _("Press keyboard keys to bind..."),
      NULL, "xine", 0, gui_layer_above (kbe->nw.gui, NULL), kbe->nw.gui->icon, XITK_WINDOW_BG_FRAME);
  }
  set_window_type_start (kbe->nw.gui, kbe->kbr.xwin);
  xitk_window_flags (kbe->kbr.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
  xitk_window_raise_window (kbe->kbr.xwin);
  /* this works only in this order :-/ */
  kbe->kbr.key = xitk_be_register_event_handler ("kbrec", kbe->kbr.xwin, kbr_event, kbe, NULL, NULL);
  xitk_window_set_transient_for_win (kbe->kbr.xwin, kbe->nw.xwin);
}

/*
 *
 */

static int kbedit_event (void *data, const xitk_be_event_t *e) {
  xui_keyedit_t *kbedit = data;

  if (((e->type == XITK_EV_KEY_DOWN) && (e->utf8[0] == XITK_CTRL_KEY_PREFIX) && (e->utf8[1] == XITK_KEY_ESCAPE))
    || (e->type == XITK_EV_DEL_WIN))
    return kbedit->exit (kbedit);
  if (e->type == XITK_EV_KEY_DOWN) {
    char buf[256];
    const kbinding_entry_t *ke;
    uint32_t qual;

    if (xitk_be_event_name (e, buf, sizeof (buf)) < 1)
      return 0;
    qual = e->qual & (MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD3 | MODIFIER_MOD4 | MODIFIER_MOD5);
    ke = kbindings_entry_from_key (kbedit->kbt, buf, qual);
    if (!ke)
      return 0;
    if ((ke->modifier != qual) && !(kbedit->nw.gui->kbindings_enabled & 2))
      return 0;
    _kbr_close (kbedit);
    xitk_browser_set_select (kbedit->w[_W_browser], ke->index);
    kbedit_select (kbedit, ke->index);
    return 1;
  }
  return 0;
}

/* needed to turn button into checkbox */
static void kbedit_dummy (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_keyedit_t *kbedit = (xui_keyedit_t *)data;

  (void)modifier;
  if (w == kbedit->w[_W_impl_alias_box]) {
    config_update_num (kbedit->nw.gui->xine, "gui.kbindings.implicit_aliases", state);
  } else if (w == kbedit->w[_W_enab_box]) {
    /* rare event, avoid code duplication. */
    gui_action_args (kbedit->nw.gui, ACTID_KBENABLE, !!state, NULL);
  }
}

/*
 *
 */
void kbedit_main (xitk_widget_t *mode, void *data) {
  gGui_t *gui = data;
  int            x, y, y1;
  int            fontheight;
  xitk_font_t   *fs;
  xui_keyedit_t *kbedit;

  if (!gui)
    return;

  kbedit = gui->keyedit;
  if (mode == XUI_W_OFF) {
    if (!kbedit)
      return;
    kbedit->exit (kbedit);
    return;
  } else if (mode == XUI_W_ON) {
    if (kbedit) {
      gui_raise_window (gui, kbedit->nw.xwin);
      xitk_button_set_state (kbedit->w[_W_impl_alias_box], kbedit->nw.gui->kbindings_enabled & 2);
      xitk_button_set_state (kbedit->w[_W_enab_box], kbedit->nw.gui->kbindings_enabled & 1);
      return;
    }
  } else { /* toggle */
    if (kbedit) {
      kbedit->exit (kbedit);
      return;
    }
  }

  kbedit = (xui_keyedit_t *)calloc (1, sizeof (*kbedit));
  if (!kbedit)
    return;
  kbedit->nw.gui = gui;
  kbedit->nw.gui->keyedit = kbedit;

  kbedit->exit = _kbedit_exit;
  _dyn_string_init (&kbedit->ebuf);

  kbedit->kbt = kbindings_dup (gui->kbindings);
  kbedit->action_wanted = KBEDIT_NOOP;

  if (xitk_init_NULL ()) {
    kbedit->nw.skin = NULL;
    kbedit->nw.wfskin = NULL;
    kbedit->nw.adjust = NULL;
    kbedit->kbr.entry.key = NULL;
    kbedit->kbr.xwin = NULL;
    kbedit->ksel = NULL;
  }

  kbedit->nw.title = _("Key Binding Editor");
  kbedit->nw.id = "kbedit";
  kbedit->nw.wr.x = 80;
  kbedit->nw.wr.y = 80;
  kbedit->nw.wr.width = WINDOW_WIDTH;
  kbedit->nw.wr.height = WINDOW_HEIGHT;
  if (gui_window_new (&kbedit->nw) < 0) {
    free (kbedit);
    return;
  }

#define _XBRD 15 /* window border */
#define _XGAP 10
#define _FRAMEWIDTH 2
#define _CSIZE 12 /* checkbox */
#define _XMODTEXT 40
#define _XMODALL (2 * _FRAMEWIDTH + 5 * (_CSIZE + _XGAP + _XMODTEXT) + 6 * _XGAP)
#define _XBTN ((WINDOW_WIDTH - 2 * _XBRD - 5 * _XGAP / 2) / 6)

  y = 34;
  xitk_image_draw_rectangular_box (kbedit->nw.bg,
    _XBRD, y, WINDOW_WIDTH - 2 * _XBRD, MAX_DISP_ENTRIES * 20 + 10, XITK_DRAW_INNER);

  y += MAX_DISP_ENTRIES * 20 + 20 + 10 + 30;
  y1 = y; /* remember for later */
  xitk_image_draw_frame (kbedit->nw.bg,
    _("Binding Action"), hboldfontname, _XBRD, y, WINDOW_WIDTH - 2 * _XBRD, 45, XITK_DRAW_OUTTER);

  y += 45 + 3;
  xitk_image_draw_frame (kbedit->nw.bg,
    _("Key"), hboldfontname, _XBRD, y, WINDOW_WIDTH - 2 * _XBRD - _XGAP / 2 - _XMODALL, 45, XITK_DRAW_OUTTER);
  xitk_image_draw_frame (kbedit->nw.bg,
    _("Modifiers"), hboldfontname, WINDOW_WIDTH - _XBRD - _XMODALL, y, _XMODALL, 45, XITK_DRAW_OUTTER);

  xitk_window_set_background_image (kbedit->nw.xwin, kbedit->nw.bg);

  kbedit_create_browser_entries (kbedit);

  y = 34;

  {
    xitk_browser_widget_t br = {
      .nw = {
        .wl = kbedit->nw.wl,
        .userdata = kbedit,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      },
      .browser = {
        .max_displayed_entries = MAX_DISP_ENTRIES,
        .num_entries           = kbedit->num_entries,
        .entries               = (const char* const*)kbedit->entries,
        .shortcuts             = (const char* const*)kbedit->shortcuts
      },
      .callback = kbedit_sel
    };

    kbedit->w[_W_browser] = xitk_noskin_browser_create (&br,
      _XBRD + 5, y + 5, WINDOW_WIDTH - 2 * (_XBRD + 5), 20, -16, br_fontname);
  }
  xitk_browser_set_alignment (kbedit->w[_W_browser], ALIGN_LEFT);

  y = y1 - 30 + 4;

  {
    uint32_t style = XITK_DRAW_SAT (kbedit->nw.gui->gfx_saturation);
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = kbedit->nw.wl,
        .userdata = kbedit,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      },
      .align = ALIGN_CENTER,
      .button_type = RADIO_BUTTON
    };

    x = (WINDOW_WIDTH - 5 * _XGAP / 2 - 6 * _XBTN) / 2;

    lb.style    = XITK_DRAW_R | XITK_DRAW_B | style;
    lb.label    = _("Alias");
    lb.callback = kbedit_alias;
    kbedit->w[_W_alias] = xitk_noskin_labelbutton_create (&lb, x, y, _XBTN, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);

    x += _XBTN + _XGAP / 2;
    lb.label    = _("Edit");
    lb.callback = kbedit_edit;
    kbedit->w[_W_edit] = xitk_noskin_labelbutton_create (&lb, x, y, _XBTN, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);

    x += _XBTN + _XGAP / 2;
    lb.button_type = CLICK_BUTTON;
    lb.label       = _("Delete");
    lb.callback    = kbedit_delete;
    kbedit->w[_W_delete] = xitk_noskin_labelbutton_create (&lb, x, y, _XBTN, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);

    x += _XBTN + _XGAP / 2;
    lb.style          = XITK_DRAW_B | style;
    lb.label          = _("Save");
    lb.callback       = kbedit_save;
    kbedit->w[_W_save] = xitk_noskin_labelbutton_create (&lb, x, y, _XBTN, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);

    x += _XBTN + _XGAP / 2;
    lb.style          = XITK_DRAW_R | XITK_DRAW_B | style;
    lb.label          = _("Reset");
    lb.callback       = kbedit_reset;
    kbedit->w[_W_reset] =  xitk_noskin_labelbutton_create (&lb, x, y, _XBTN, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);

    x += _XBTN + _XGAP / 2;
    lb.style          = 0;
    lb.label          = _("Done");
    lb.callback       = kbedit_exit;
    kbedit->w[_W_done] = xitk_noskin_labelbutton_create (&lb, x, y, _XBTN, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);
  }

  fs = xitk_font_load_font (gui->xitk, hboldfontname);
  fontheight = xitk_font_text_height (fs, " Jg", 3) + 2;
  xitk_font_unload_font (fs);

  y = y1 + (45 / 2);                /* Checkbox                     */
  y1 = y - ((fontheight - _CSIZE - 2 + 1) / 2); /* Text, v-centered to ckeckbox */

  {
    uint32_t i;
    const char * const *mod_names = kbindings_modifier_names (kbedit->kbt);
    xitk_button_widget_t b = {
      .nw = {
        .wl = kbedit->nw.wl,
        .userdata = kbedit,
        .add_state = XITK_WIDGET_STATE_VISIBLE
      },
      .symbol = XITK_SYMBOL_CHECK,
      .state_callback = kbedit_dummy
    };
    xitk_label_widget_t l = {
      .nw = {
        .wl = kbedit->nw.wl,
        .userdata = kbedit,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      }
    };

    kbedit->w[_W_comment] = xitk_noskin_label_create (&l, _XBRD + _FRAMEWIDTH + _XGAP, y1 - 1,
      WINDOW_WIDTH - 2 * (_XBRD + _FRAMEWIDTH + _XGAP), fontheight, hboldfontname);

    y += 45 + 3;
    y1 += 45 + 3;

    kbedit->w[_W_key] = xitk_noskin_label_create (&l, _XBRD + _FRAMEWIDTH + _XGAP, y1 - 1,
      WINDOW_WIDTH - 2 * (_XBRD + _FRAMEWIDTH + _XGAP) - _XMODALL - _XGAP / 2, fontheight, hboldfontname);

    x = WINDOW_WIDTH - _XBRD - _XMODALL + _FRAMEWIDTH + _XGAP;
    for (i = 0; i < 5; i++) {
      xitk_widget_t *w;

      kbedit->w[_W_ctrl + i] = w = xitk_noskin_button_create (&b, x - 1, y - 1, _CSIZE + 2, _CSIZE + 2);
      x += _CSIZE + _XGAP;
      l.label = mod_names[i];
      w =  xitk_noskin_label_create (&l, x, y1 - 1, _XMODTEXT, fontheight, hboldfontname);
      x += _XMODTEXT + _XGAP;
    }

    y -= 45 + 3 + 65;
    y1 -= 45 + 3 + 65;

    b.nw.tips = _("If not assigned otherwise, \"Ctrl+Key\", \"Meta+Key\" etc. mean the same as \"Key\" alone.");
    l.label = _("Convenient key bindings"); /** << same as in src/xitk/event.c */
    b.nw.add_state = (kbedit->nw.gui->kbindings_enabled & 2)
                   ? (XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_ON)
                   : (XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE);
    x = _XBRD + _FRAMEWIDTH + _XGAP;
    kbedit->w[_W_impl_alias_box] = xitk_noskin_button_create (&b, x - 1, y - 1, _CSIZE + 2, _CSIZE + 2);
    x += _CSIZE + _XGAP;
    kbedit->w[_W_impl_alias_label] = xitk_noskin_label_create (&l, x, y1 - 1,
        (WINDOW_WIDTH - x - (_XBRD + _FRAMEWIDTH + _XGAP)) * 3 / 5, fontheight, hboldfontname);
    xitk_widget_set_focus_redirect (kbedit->w[_W_impl_alias_label], kbedit->w[_W_impl_alias_box]);

    x += (WINDOW_WIDTH - x - (_XBRD + _FRAMEWIDTH + _XGAP)) * 3 / 5 + _XGAP;
    b.nw.tips = NULL;
    l.label = _("enabled"); /** << same as in src/xitk/event.c */
    b.nw.add_state = (kbedit->nw.gui->kbindings_enabled & 1)
                   ? (XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_ON)
                   : (XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE);
    kbedit->w[_W_enab_box] = xitk_noskin_button_create (&b, x - 1, y - 1, _CSIZE + 2, _CSIZE + 2);
    x += _CSIZE + _XGAP;
    kbedit->w[_W_enab_label] = xitk_noskin_label_create (&l, x, y1 - 1,
      WINDOW_WIDTH - x - (_XBRD + _FRAMEWIDTH + _XGAP), fontheight, hboldfontname);
    xitk_widget_set_focus_redirect (kbedit->w[_W_enab_label], kbedit->w[_W_enab_box]);

  }

  y = WINDOW_HEIGHT - (23 + 15);

  {
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = kbedit->nw.wl,
        .userdata = kbedit,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      },
      .style = XITK_DRAW_R | XITK_DRAW_G | XITK_DRAW_SAT (kbedit->nw.gui->gfx_saturation),
      .label = _("Grab"),
      .callback = kbedit_grab,
      .align = ALIGN_CENTER,
      .button_type = RADIO_BUTTON
    };

    kbedit->w[_W_grab] = xitk_noskin_labelbutton_create (&lb, _XBRD, y, WINDOW_WIDTH - 2 * _XBRD, 23,
      XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, hboldfontname);
  }

  kbedit_unset (kbedit);

  kbedit->nw.key = xitk_be_register_event_handler ("kbedit", kbedit->nw.xwin, kbedit_event, kbedit, NULL, NULL);

  kbedit->kbr.key = 0;

  kbedit->nsel = -1;

  gui->ssaver_enabled = 0;
  gui_raise_window (gui, kbedit->nw.xwin);
}
