/* c_pde.c : Extention library for partial differential equation solvers
 * All in NArray
 */

#include<stdio.h>
#include<string.h>
#include<math.h>
#include "ruby.h"
#include "narray.h"

/* for compatibility for NArray and NArray with big memory patch */
#ifndef NARRAY_BIGMEM
typedef int na_shape_t;
#endif

#define ID(i,j) ((i)+nx*(j))

/* Eq: A z_xx + 2 B z_xy + C z_yy + D z_x + E z_y = F
 *              ^^^
 *
 * Assumption: dx and dy are constants (uniform).
 * Boundary values are not changed. --> change afterwords if not Dirichlet.
 *
 * Basic discretization: by central differences:
 *   A/dx^2 (z[i-1,j]-2z[i,j]+z[i+1,j])
 *   + B/(2dxdy) ((z[i+1,j+1]-z[i-1,j+1]-z[i+1,j-1]+z[i-1,j-1])
 *   + C/dy^2 (z[i,j-1]-2z[i,j]+z[i,j+1])
 *   + D/(2dx) (z[i+1,j]-z[i-1,j]) + E/(2dy) (z[i,j+1]-z[i,j-1]) - F = 0
 * 
 * * Z [2D Float(=DFLOAT) NArray] (inout) -- to be overwritten
 * * A, B, ... [2D Float(=DFLOAT) NArray] coeffitients
 * * OME [Float] constant, which should 1<= OME < 2 (Gauss-Seidel if OME=1)
 * * Return value: res = |H dz| (L2 norm),
 *   where H = 2(diag(A)/dx^2+diag(C)/dy^2).
 *   Convergence can be judged by res/|F| < eps
 */
static VALUE
SOR_2D_2ndorder_1step(obj, Z, A, B, C, D, E, F, DX, DY, OME, Ignore_non_eliptic)
     VALUE obj;
     VALUE Z;
     VALUE A, B, C, D, E, F;
     VALUE DX, DY, OME;
     VALUE Ignore_non_eliptic;
{
    double *z, *ap, *bp, *cp, *dp, *ep, *fp;
    double a, b, c, d, e, f, g, h, sum=0;
    double dx, dy, ome;
    struct NARRAY *na;
    na_shape_t nx, ny, i, j;
    VALUE res;

    /* check & read arguments */
    
    if (NA_TYPE(Z) != NA_DFLOAT) rb_raise(rb_eArgError, "Z nust be DFLOAT");
    if (NA_TYPE(A) != NA_DFLOAT) rb_raise(rb_eArgError, "A nust be DFLOAT");
    if (NA_TYPE(B) != NA_DFLOAT) rb_raise(rb_eArgError, "B nust be DFLOAT");
    if (NA_TYPE(C) != NA_DFLOAT) rb_raise(rb_eArgError, "C nust be DFLOAT");
    if (NA_TYPE(D) != NA_DFLOAT) rb_raise(rb_eArgError, "D nust be DFLOAT");
    if (NA_TYPE(E) != NA_DFLOAT) rb_raise(rb_eArgError, "E nust be DFLOAT");
    if (NA_TYPE(F) != NA_DFLOAT) rb_raise(rb_eArgError, "F nust be DFLOAT");

    if (NA_RANK(Z) != 2) rb_raise(rb_eArgError, "Z must be 2-dimensional");
    nx = NA_SHAPE0(Z);
    ny = NA_SHAPE1(Z);
    if (NA_SHAPE0(A)!=nx || NA_SHAPE1(A)!=ny) rb_raise(rb_eArgError,
                                                       "A: invalid shape");
    if (NA_SHAPE0(B)!=nx || NA_SHAPE1(B)!=ny) rb_raise(rb_eArgError,
                                                       "B: invalid shape");
    if (NA_SHAPE0(C)!=nx || NA_SHAPE1(C)!=ny) rb_raise(rb_eArgError,
                                                       "C: invalid shape");
    if (NA_SHAPE0(D)!=nx || NA_SHAPE1(D)!=ny) rb_raise(rb_eArgError,
                                                       "D: invalid shape");
    if (NA_SHAPE0(E)!=nx || NA_SHAPE1(E)!=ny) rb_raise(rb_eArgError,
                                                       "E: invalid shape");
    if (NA_SHAPE0(F)!=nx || NA_SHAPE1(F)!=ny) rb_raise(rb_eArgError,
                                                       "F: invalid shape");

    GetNArray(Z, na); z = (double *)NA_PTR(na, 0);
    GetNArray(A, na); ap = (double *)NA_PTR(na, 0);
    GetNArray(B, na); bp = (double *)NA_PTR(na, 0);
    GetNArray(C, na); cp = (double *)NA_PTR(na, 0);
    GetNArray(D, na); dp = (double *)NA_PTR(na, 0);
    GetNArray(E, na); ep = (double *)NA_PTR(na, 0);
    GetNArray(F, na); fp = (double *)NA_PTR(na, 0);

    dx = NUM2DBL(DX);
    dy = NUM2DBL(DY);
    if (dx==0.0) rb_raise(rb_eArgError, "dx == 0");
    if (dy==0.0) rb_raise(rb_eArgError, "dy == 0");
    ome = NUM2DBL(OME);

    /* solve (1 step) */
    for (j=1; j<ny-1; j++){
        for (i=1; i<nx-1; i++){
            a = ap[ID(i,j)] / (dx*dx);
            b = bp[ID(i,j)] / (dx*dy);
            c = cp[ID(i,j)] / (dy*dy);
            if (Ignore_non_eliptic == Qfalse || Ignore_non_eliptic == Qnil ) {
                if((a*c-b*b)<=0.0) 
                    rb_raise(rb_eStandardError,"non-eliptic@[%d,%d]",i,j);
            }
            d = dp[ID(i,j)] / dx;
            e = ep[ID(i,j)] / dy;
            f = fp[ID(i,j)];
            h = 2*(a+c);
            g = ome * ( a*( z[ID(i-1,j)]+z[ID(i+1,j)] )
                      + b/2*( z[ID(i+1,j+1)] - z[ID(i-1,j+1)] - z[ID(i+1,j-1)]
                             +z[ID(i-1,j-1)] )
                      + c*( z[ID(i,j-1)]+z[ID(i,j+1)] )
                      + d/2*( z[ID(i+1,j)]-z[ID(i-1,j)] )
                      + e/2*( z[ID(i,j+1)]-z[ID(i,j-1)] )
                      - f - h*z[ID(i,j)]);
            sum += g*g;
            z[ID(i,j)] += g/h;
        }
    }
    res = DBL2NUM( sqrt(sum) );
    return(res);
}

void
init_ganalysis_pde_ext()
{
    static VALUE mNumRu, mGA, mPDE;
    mNumRu = rb_define_module("NumRu");
    mGA = rb_define_module_under(mNumRu, "GAnalysis");
    mPDE = rb_define_module_under(mGA, "PDE");
    rb_define_module_function(mPDE, "SOR_2D_2ndorder_1step", 
                              SOR_2D_2ndorder_1step, 11);
}
