% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcmc-diagnostics.R
\name{MCMC-diagnostics}
\alias{MCMC-diagnostics}
\alias{mcmc_rhat}
\alias{mcmc_rhat_hist}
\alias{mcmc_rhat_data}
\alias{mcmc_neff}
\alias{mcmc_neff_hist}
\alias{mcmc_neff_data}
\alias{mcmc_acf}
\alias{mcmc_acf_bar}
\title{General MCMC diagnostics}
\usage{
mcmc_rhat(rhat, ..., size = NULL)

mcmc_rhat_hist(rhat, ..., binwidth = NULL, breaks = NULL)

mcmc_rhat_data(rhat, ...)

mcmc_neff(ratio, ..., size = NULL)

mcmc_neff_hist(ratio, ..., binwidth = NULL, breaks = NULL)

mcmc_neff_data(ratio, ...)

mcmc_acf(
  x,
  pars = character(),
  regex_pars = character(),
  ...,
  facet_args = list(),
  lags = 20,
  size = NULL
)

mcmc_acf_bar(
  x,
  pars = character(),
  regex_pars = character(),
  ...,
  facet_args = list(),
  lags = 20
)
}
\arguments{
\item{rhat}{A vector of R-hat estimates.}

\item{...}{Currently ignored.}

\item{size}{Optional values to override \code{\link[ggplot2:geom_point]{ggplot2::geom_point()}}'s
default size (for \code{mcmc_rhat()}, \code{mcmc_neff()}) or
\code{\link[ggplot2:geom_path]{ggplot2::geom_line()}}'s default line width (for \code{mcmc_acf()}).}

\item{binwidth}{Passed to \code{\link[ggplot2:geom_histogram]{ggplot2::geom_histogram()}} to override
the default binwidth.}

\item{breaks}{Passed to \code{\link[ggplot2:geom_histogram]{ggplot2::geom_histogram()}} as an
alternative to \code{binwidth}.}

\item{ratio}{A vector of \emph{ratios} of effective sample size estimates to
total sample size. See \code{\link[=neff_ratio]{neff_ratio()}}.}

\item{x}{An object containing MCMC draws:
\itemize{
\item A 3-D array, matrix, list of matrices, or data frame. The \link{MCMC-overview}
page provides details on how to specify each these.
\item A \code{draws} object from the \pkg{\link{posterior}} package (e.g.,
\code{draws_array}, \code{draws_rvars}, etc.).
\item An object with an \code{as.array()} method that returns the same kind of 3-D
array described on the \link{MCMC-overview} page.
}}

\item{pars}{An optional character vector of parameter names. If neither
\code{pars} nor \code{regex_pars} is specified then the default is to use \emph{all}
parameters. As of version \verb{1.7.0}, \strong{bayesplot} also supports 'tidy'
parameter selection by specifying \code{pars = vars(...)}, where \code{...} is
specified the same way as in \link[dplyr:select]{dplyr::select(...)} and
similar functions. Examples of using \code{pars} in this way can be found on the
\link[=tidy-params]{Tidy parameter selection} page.}

\item{regex_pars}{An optional \link[base:grep]{regular expression} to use for
parameter selection. Can be specified instead of \code{pars} or in addition to
\code{pars}. When using \code{pars} for tidy parameter selection, the \code{regex_pars}
argument is ignored since \link[tidyselect:language]{select helpers}
perform a similar function.}

\item{facet_args}{A named list of arguments (other than \code{facets}) passed
to \code{\link[ggplot2:facet_wrap]{ggplot2::facet_wrap()}} or \code{\link[ggplot2:facet_grid]{ggplot2::facet_grid()}}
to control faceting. Note: if \code{scales} is not included in \code{facet_args}
then \strong{bayesplot} may use \code{scales="free"} as the default (depending
on the plot) instead of the \strong{ggplot2} default of \code{scales="fixed"}.}

\item{lags}{The number of lags to show in the autocorrelation plot.}
}
\value{
The plotting functions return a ggplot object that can be further
customized using the \strong{ggplot2} package. The functions with suffix
\verb{_data()} return the data that would have been drawn by the plotting
function.
}
\description{
Plots of Rhat statistics, ratios of effective sample size to total sample
size, and autocorrelation of MCMC draws. See the \strong{Plot Descriptions}
section, below, for details. For models fit using the No-U-Turn-Sampler, see
also \link{MCMC-nuts} for additional MCMC diagnostic plots.
}
\section{Plot Descriptions}{

\describe{
\item{\code{mcmc_rhat()}, \code{mcmc_rhat_hist()}}{
Rhat values as either points or a histogram. Values are colored using
different shades (lighter is better). The chosen thresholds are somewhat
arbitrary, but can be useful guidelines in practice.
\itemize{
\item \emph{light}: below 1.05 (good)
\item \emph{mid}: between 1.05 and 1.1 (ok)
\item \emph{dark}: above 1.1 (too high)
}
}

\item{\code{mcmc_neff()}, \code{mcmc_neff_hist()}}{
Ratios of effective sample size to total sample size as either points or a
histogram. Values are colored using different shades (lighter is better).
The chosen thresholds are somewhat arbitrary, but can be useful guidelines
in practice.
\itemize{
\item \emph{light}: between 0.5 and 1 (high)
\item \emph{mid}: between 0.1 and 0.5 (good)
\item \emph{dark}: below 0.1 (low)
}
}

\item{\code{mcmc_acf()}, \code{mcmc_acf_bar()}}{
Grid of autocorrelation plots by chain and parameter. The \code{lags} argument
gives the maximum number of lags at which to calculate the autocorrelation
function. \code{mcmc_acf()} is a line plot whereas \code{mcmc_acf_bar()} is a
barplot.
}
}
}

\examples{
# autocorrelation
x <- example_mcmc_draws()
dim(x)
dimnames(x)

color_scheme_set("green")
mcmc_acf(x, pars = c("alpha", "beta[1]"))
\donttest{
color_scheme_set("pink")
(p <- mcmc_acf_bar(x, pars = c("alpha", "beta[1]")))

# add horiztonal dashed line at 0.5
p + hline_at(0.5, linetype = 2, size = 0.15, color = "gray")
}

# fake rhat values to use for demonstration
rhat <- c(runif(100, 1, 1.15))
mcmc_rhat_hist(rhat)
mcmc_rhat(rhat)

# lollipops
color_scheme_set("purple")
mcmc_rhat(rhat[1:10], size = 5)

color_scheme_set("blue")
mcmc_rhat(runif(1000, 1, 1.07))
mcmc_rhat(runif(1000, 1, 1.3)) + legend_move("top") # add legend above plot

# fake neff ratio values to use for demonstration
ratio <- c(runif(100, 0, 1))
mcmc_neff_hist(ratio)
mcmc_neff(ratio)

\dontrun{
# Example using rstanarm model (requires rstanarm package)
library(rstanarm)

# intentionally use small 'iter' so there are some
# problems with rhat and neff for demonstration
fit <- stan_glm(mpg ~ ., data = mtcars, iter = 50, refresh = 0)
rhats <- rhat(fit)
ratios <- neff_ratio(fit)
mcmc_rhat(rhats)
mcmc_neff(ratios, size = 3)

# there's a small enough number of parameters in the
# model that we can display their names on the y-axis
mcmc_neff(ratios) + yaxis_text(hjust = 1)

# can also look at autocorrelation
draws <- as.array(fit)
mcmc_acf(draws, pars = c("wt", "cyl"), lags = 10)

# increase number of iterations and plots look much better
fit2 <- update(fit, iter = 500)
mcmc_rhat(rhat(fit2))
mcmc_neff(neff_ratio(fit2))
mcmc_acf(as.array(fit2), pars = c("wt", "cyl"), lags = 10)
}

}
\references{
Stan Development Team.
\emph{Stan Modeling Language Users Guide and Reference Manual.}
\url{https://mc-stan.org/users/documentation/}

Gelman, A. and Rubin, D. B. (1992). Inference from iterative
simulation using multiple sequences. \emph{Statistical Science}. 7(4),
457--472.
}
\seealso{
\itemize{
\item The \href{https://mc-stan.org/bayesplot/articles/visual-mcmc-diagnostics.html}{Visual MCMC Diagnostics}
vignette.
\item \link{MCMC-nuts} for additional MCMC diagnostic plots for models fit
using the No-U-Turn-Sampler.
}

Other MCMC: 
\code{\link{MCMC-combos}},
\code{\link{MCMC-distributions}},
\code{\link{MCMC-intervals}},
\code{\link{MCMC-nuts}},
\code{\link{MCMC-overview}},
\code{\link{MCMC-parcoord}},
\code{\link{MCMC-recover}},
\code{\link{MCMC-scatterplots}},
\code{\link{MCMC-traces}}
}
\concept{MCMC}
