from typing import Tuple
from functools import lru_cache

ALPHABET = 1
LATIN = 2
CYRILLIC = 3
GREEK = 4
ARABIC = 5
CJK = 6
HANGUL = 7
ABJAD = 99
ABUGIDA = 100
SYLLABARY = 101

HISTORIC = 99999
FUNKY = 100000
UNKNOWN = 0

# Source: https://www.unicode.org/Public/UCD/latest/ucd/Blocks.txt
UNICODE_BLOCKS: Tuple[Tuple[int, int, str, Tuple[int, ...]]] = (  # type: ignore
    (
        0x0000,
        0x007F,
        "Basic Latin",
        (
            ALPHABET,
            LATIN,
        ),
    ),
    (
        0x0080,
        0x00FF,
        "Latin-1 Supplement",
        (
            ALPHABET,
            LATIN,
        ),
    ),
    (
        0x0100,
        0x017F,
        "Latin Extended-A",
        (
            ALPHABET,
            LATIN,
        ),
    ),
    (
        0x0180,
        0x024F,
        "Latin Extended-B",
        (
            ALPHABET,
            LATIN,
        ),
    ),
    (0x0250, 0x02AF, "IPA Extensions", ()),
    (0x02B0, 0x02FF, "Spacing Modifier Letters", ()),
    (0x0300, 0x036F, "Combining Diacritical Marks", ()),
    (
        0x0370,
        0x03FF,
        "Greek and Coptic",
        (
            ALPHABET,
            GREEK,
        ),
    ),
    (
        0x0400,
        0x04FF,
        "Cyrillic",
        (
            ALPHABET,
            CYRILLIC,
        ),
    ),
    (
        0x0500,
        0x052F,
        "Cyrillic Supplement",
        (
            ALPHABET,
            CYRILLIC,
        ),
    ),
    (0x0530, 0x058F, "Armenian", (ALPHABET,)),
    (0x0590, 0x05FF, "Hebrew", (ABJAD,)),
    (
        0x0600,
        0x06FF,
        "Arabic",
        (
            ARABIC,
            ABJAD,
        ),
    ),
    (0x0700, 0x074F, "Syriac", (ABJAD,)),
    (
        0x0750,
        0x077F,
        "Arabic Supplement",
        (
            ARABIC,
            ABJAD,
        ),
    ),
    (0x0780, 0x07BF, "Thaana", (ABUGIDA,)),
    (0x07C0, 0x07FF, "NKo", (FUNKY,)),
    (0x0800, 0x083F, "Samaritan", (ABJAD,)),
    (
        0x0840,
        0x085F,
        "Mandaic",
        (
            ALPHABET,
            HISTORIC,
        ),
    ),
    (0x0860, 0x086F, "Syriac Supplement", (ABJAD,)),
    (
        0x0870,
        0x089F,
        "Arabic Extended-B",
        (
            ARABIC,
            ABJAD,
        ),
    ),
    (
        0x08A0,
        0x08FF,
        "Arabic Extended-A",
        (
            ARABIC,
            ABJAD,
        ),
    ),
    (0x0900, 0x097F, "Devanagari", (ABUGIDA,)),
    (0x0980, 0x09FF, "Bengali", (ABUGIDA,)),
    (0x0A00, 0x0A7F, "Gurmukhi", (ABUGIDA,)),
    (0x0A80, 0x0AFF, "Gujarati", (ABUGIDA,)),
    (
        0x0B00,
        0x0B7F,
        "Oriya",
        (
            ABUGIDA,
            HISTORIC,
        ),
    ),
    (0x0B80, 0x0BFF, "Tamil", (ABUGIDA,)),
    (0x0C00, 0x0C7F, "Telugu", (ABUGIDA,)),
    (0x0C80, 0x0CFF, "Kannada", (ABUGIDA,)),
    (0x0D00, 0x0D7F, "Malayalam", (ABUGIDA,)),
    (0x0D80, 0x0DFF, "Sinhala", (ABUGIDA,)),
    (0x0E00, 0x0E7F, "Thai", (ABUGIDA,)),
    (0x0E80, 0x0EFF, "Lao", (ABUGIDA,)),
    (0x0F00, 0x0FFF, "Tibetan", (ABUGIDA,)),
    (0x1000, 0x109F, "Myanmar", (ABUGIDA,)),
    (0x10A0, 0x10FF, "Georgian", (ALPHABET,)),
    (
        0x1100,
        0x11FF,
        "Hangul Jamo",
        (
            ALPHABET,
            HANGUL,
        ),
    ),
    (0x1200, 0x137F, "Ethiopic", (ABUGIDA,)),
    (0x1380, 0x139F, "Ethiopic Supplement", (ABUGIDA,)),
    (0x13A0, 0x13FF, "Cherokee", (SYLLABARY,)),
    (0x1400, 0x167F, "Unified Canadian Aboriginal Syllabics", (SYLLABARY,)),
    (
        0x1680,
        0x169F,
        "Ogham",
        (
            ALPHABET,
            HISTORIC,
        ),
    ),
    (
        0x16A0,
        0x16FF,
        "Runic",
        (
            ALPHABET,
            HISTORIC,
        ),
    ),
    (0x1700, 0x171F, "Tagalog", (ABUGIDA,)),
    (0x1720, 0x173F, "Hanunoo", (ABUGIDA,)),
    (0x1740, 0x175F, "Buhid", (ABUGIDA,)),
    (0x1760, 0x177F, "Tagbanwa", (ABUGIDA,)),
    (0x1780, 0x17FF, "Khmer", (ABUGIDA,)),
    (
        0x1800,
        0x18AF,
        "Mongolian",
        (
            ALPHABET,
            FUNKY,
        ),
    ),
    (0x18B0, 0x18FF, "Unified Canadian Aboriginal Syllabics Extended", (SYLLABARY,)),
    (0x1900, 0x194F, "Limbu", (ABUGIDA,)),
    (0x1950, 0x197F, "Tai Le", (ABUGIDA,)),
    (0x1980, 0x19DF, "New Tai Lue", (ABUGIDA,)),
    (0x19E0, 0x19FF, "Khmer Symbols", (ABUGIDA,)),
    (0x1A00, 0x1A1F, "Buginese", (ABUGIDA,)),
    (0x1A20, 0x1AAF, "Tai Tham", (ABUGIDA,)),
    (0x1AB0, 0x1AFF, "Combining Diacritical Marks Extended", ()),
    (0x1B00, 0x1B7F, "Balinese", (ABUGIDA,)),
    (0x1B80, 0x1BBF, "Sundanese", (ABUGIDA,)),
    (0x1BC0, 0x1BFF, "Batak", (ABUGIDA,)),
    (0x1C00, 0x1C4F, "Lepcha", (ABUGIDA,)),
    (
        0x1C50,
        0x1C7F,
        "Ol Chiki",
        (
            ALPHABET,
            FUNKY,
        ),
    ),
    (
        0x1C80,
        0x1C8F,
        "Cyrillic Extended-C",
        (
            ALPHABET,
            CYRILLIC,
        ),
    ),
    (0x1C90, 0x1CBF, "Georgian Extended", (ALPHABET,)),
    (0x1CC0, 0x1CCF, "Sundanese Supplement", (ABUGIDA,)),
    (0x1CD0, 0x1CFF, "Vedic Extensions", ()),
    (0x1D00, 0x1D7F, "Phonetic Extensions", ()),
    (0x1D80, 0x1DBF, "Phonetic Extensions Supplement", ()),
    (0x1DC0, 0x1DFF, "Combining Diacritical Marks Supplement", ()),
    (
        0x1E00,
        0x1EFF,
        "Latin Extended Additional",
        (
            ALPHABET,
            LATIN,
        ),
    ),
    (
        0x1F00,
        0x1FFF,
        "Greek Extended",
        (
            ALPHABET,
            GREEK,
        ),
    ),
    (0x2000, 0x206F, "General Punctuation", ()),
    (0x2070, 0x209F, "Superscripts and Subscripts", ()),
    (0x20A0, 0x20CF, "Currency Symbols", ()),
    (0x20D0, 0x20FF, "Combining Diacritical Marks for Symbols", ()),
    (0x2100, 0x214F, "Letterlike Symbols", ()),
    (0x2150, 0x218F, "Number Forms", ()),
    (0x2190, 0x21FF, "Arrows", ()),
    (0x2200, 0x22FF, "Mathematical Operators", ()),
    (0x2300, 0x23FF, "Miscellaneous Technical", ()),
    (0x2400, 0x243F, "Control Pictures", ()),
    (0x2440, 0x245F, "Optical Character Recognition", ()),
    (0x2460, 0x24FF, "Enclosed Alphanumerics", ()),
    (0x2500, 0x257F, "Box Drawing", ()),
    (0x2580, 0x259F, "Block Elements", ()),
    (0x25A0, 0x25FF, "Geometric Shapes", ()),
    (0x2600, 0x26FF, "Miscellaneous Symbols", ()),
    (0x2700, 0x27BF, "Dingbats", ()),
    (0x27C0, 0x27EF, "Miscellaneous Mathematical Symbols-A", ()),
    (0x27F0, 0x27FF, "Supplemental Arrows-A", ()),
    (0x2800, 0x28FF, "Braille Patterns", ()),
    (0x2900, 0x297F, "Supplemental Arrows-B", ()),
    (0x2980, 0x29FF, "Miscellaneous Mathematical Symbols-B", ()),
    (0x2A00, 0x2AFF, "Supplemental Mathematical Operators", ()),
    (0x2B00, 0x2BFF, "Miscellaneous Symbols and Arrows", ()),
    (
        0x2C00,
        0x2C5F,
        "Glagolitic",
        (
            ALPHABET,
            HISTORIC,
        ),
    ),
    (
        0x2C60,
        0x2C7F,
        "Latin Extended-C",
        (
            ALPHABET,
            LATIN,
        ),
    ),
    (
        0x2C80,
        0x2CFF,
        "Coptic",
        (
            ALPHABET,
            HISTORIC,
        ),
    ),
    (0x2D00, 0x2D2F, "Georgian Supplement", (ALPHABET,)),
    (0x2D30, 0x2D7F, "Tifinagh", (ABJAD,)),
    (0x2D80, 0x2DDF, "Ethiopic Extended", (ABUGIDA,)),
    (
        0x2DE0,
        0x2DFF,
        "Cyrillic Extended-A",
        (
            ALPHABET,
            CYRILLIC,
        ),
    ),
    (0x2E00, 0x2E7F, "Supplemental Punctuation", ()),
    (0x2E80, 0x2EFF, "CJK Radicals Supplement", (CJK,)),
    (0x2F00, 0x2FDF, "Kangxi Radicals", (CJK,)),
    (0x2FF0, 0x2FFF, "Ideographic Description Characters", ()),
    (0x3000, 0x303F, "CJK Symbols and Punctuation", (CJK,)),
    (0x3040, 0x309F, "Hiragana", (CJK,)),
    (0x30A0, 0x30FF, "Katakana", (CJK,)),
    (0x3100, 0x312F, "Bopomofo", (CJK,)),
    (
        0x3130,
        0x318F,
        "Hangul Compatibility Jamo",
        (HANGUL, CJK),
    ),
    (0x3190, 0x319F, "Kanbun", (CJK,)),
    (0x31A0, 0x31BF, "Bopomofo Extended", (CJK,)),
    (0x31C0, 0x31EF, "CJK Strokes", (CJK,)),
    (0x31F0, 0x31FF, "Katakana Phonetic Extensions", (CJK,)),
    (0x3200, 0x32FF, "Enclosed CJK Letters and Months", (CJK,)),
    (0x3300, 0x33FF, "CJK Compatibility", (CJK,)),
    (0x3400, 0x4DBF, "CJK Unified Ideographs Extension A", (CJK,)),
    (0x4DC0, 0x4DFF, "Yijing Hexagram Symbols", (CJK,)),
    (0x4E00, 0x9FFF, "CJK Unified Ideographs", (CJK,)),
    (0xA000, 0xA48F, "Yi Syllables", (SYLLABARY,)),
    (0xA490, 0xA4CF, "Yi Radicals", ()),
    (0xA4D0, 0xA4FF, "Lisu", (ABUGIDA,)),
    (0xA500, 0xA63F, "Vai", (SYLLABARY,)),
    (
        0xA640,
        0xA69F,
        "Cyrillic Extended-B",
        (
            ALPHABET,
            CYRILLIC,
        ),
    ),
    (0xA6A0, 0xA6FF, "Bamum", (SYLLABARY,)),
    (0xA700, 0xA71F, "Modifier Tone Letters", ()),
    (
        0xA720,
        0xA7FF,
        "Latin Extended-D",
        (
            ALPHABET,
            LATIN,
        ),
    ),
    (0xA800, 0xA82F, "Syloti Nagri", ()),
    (0xA830, 0xA83F, "Common Indic Number Forms", ()),
    (0xA840, 0xA87F, "Phags-pa", ()),
    (0xA880, 0xA8DF, "Saurashtra", ()),
    (0xA8E0, 0xA8FF, "Devanagari Extended", (ABUGIDA,)),
    (0xA900, 0xA92F, "Kayah Li", ()),
    (0xA930, 0xA95F, "Rejang", ()),
    (0xA960, 0xA97F, "Hangul Jamo Extended-A", (HANGUL,)),
    (0xA980, 0xA9DF, "Javanese", (ABUGIDA,)),
    (0xA9E0, 0xA9FF, "Myanmar Extended-B", (ABUGIDA,)),
    (0xAA00, 0xAA5F, "Cham", (ABUGIDA,)),
    (0xAA60, 0xAA7F, "Myanmar Extended-A", (ABUGIDA,)),
    (0xAA80, 0xAADF, "Tai Viet", ()),
    (0xAAE0, 0xAAFF, "Meetei Mayek Extensions", ()),
    (0xAB00, 0xAB2F, "Ethiopic Extended-A", (ABUGIDA,)),
    (
        0xAB30,
        0xAB6F,
        "Latin Extended-E",
        (
            ALPHABET,
            LATIN,
        ),
    ),
    (0xAB70, 0xABBF, "Cherokee Supplement", ()),
    (0xABC0, 0xABFF, "Meetei Mayek", ()),
    (
        0xAC00,
        0xD7AF,
        "Hangul Syllables",
        (HANGUL, SYLLABARY),
    ),
    (0xD7B0, 0xD7FF, "Hangul Jamo Extended-B", (HANGUL,)),
    (0xD800, 0xDB7F, "High Surrogates", ()),
    (0xDB80, 0xDBFF, "High Private Use Surrogates", ()),
    (0xDC00, 0xDFFF, "Low Surrogates", ()),
    (0xE000, 0xF8FF, "Private Use Area", ()),
    (0xF900, 0xFAFF, "CJK Compatibility Ideographs", (CJK,)),
    (0xFB00, 0xFB4F, "Alphabetic Presentation Forms", ()),
    (
        0xFB50,
        0xFDFF,
        "Arabic Presentation Forms-A",
        (
            ARABIC,
            ABJAD,
        ),
    ),
    (0xFE00, 0xFE0F, "Variation Selectors", ()),
    (0xFE10, 0xFE1F, "Vertical Forms", ()),
    (0xFE20, 0xFE2F, "Combining Half Marks", (CJK,)),
    (0xFE30, 0xFE4F, "CJK Compatibility Forms", (CJK,)),
    (0xFE50, 0xFE6F, "Small Form Variants", ()),
    (
        0xFE70,
        0xFEFF,
        "Arabic Presentation Forms-B",
        (
            ARABIC,
            ABJAD,
        ),
    ),
    (0xFF00, 0xFFEF, "Halfwidth and Fullwidth Forms", (CJK,)),
    (0xFFF0, 0xFFFF, "Specials", ()),
    (0x10000, 0x1007F, "Linear B Syllabary", (SYLLABARY,)),
    (0x10080, 0x100FF, "Linear B Ideograms", ()),
    (0x10100, 0x1013F, "Aegean Numbers", ()),
    (
        0x10140,
        0x1018F,
        "Ancient Greek Numbers",
        (
            ALPHABET,
            GREEK,
        ),
    ),
    (0x10190, 0x101CF, "Ancient Symbols", ()),
    (0x101D0, 0x101FF, "Phaistos Disc", ()),
    (0x10280, 0x1029F, "Lycian", ()),
    (0x102A0, 0x102DF, "Carian", ()),
    (0x102E0, 0x102FF, "Coptic Epact Numbers", ()),
    (0x10300, 0x1032F, "Old Italic", ()),
    (0x10330, 0x1034F, "Gothic", ()),
    (0x10350, 0x1037F, "Old Permic", ()),
    (0x10380, 0x1039F, "Ugaritic", (ABJAD,)),
    (0x103A0, 0x103DF, "Old Persian", (ABJAD,)),
    (0x10400, 0x1044F, "Deseret", ()),
    (0x10450, 0x1047F, "Shavian", ()),
    (0x10480, 0x104AF, "Osmanya", ()),
    (0x104B0, 0x104FF, "Osage", ()),
    (0x10500, 0x1052F, "Elbasan", ()),
    (0x10530, 0x1056F, "Caucasian Albanian", ()),
    (0x10570, 0x105BF, "Vithkuqi", ()),
    (0x10600, 0x1077F, "Linear A", ()),
    (
        0x10780,
        0x107BF,
        "Latin Extended-F",
        (
            ALPHABET,
            LATIN,
        ),
    ),
    (0x10800, 0x1083F, "Cypriot Syllabary", (SYLLABARY,)),
    (0x10840, 0x1085F, "Imperial Aramaic", (ABJAD,)),
    (0x10860, 0x1087F, "Palmyrene", ()),
    (0x10880, 0x108AF, "Nabataean", (ABJAD,)),
    (0x108E0, 0x108FF, "Hatran", ()),
    (0x10900, 0x1091F, "Phoenician", ()),
    (0x10920, 0x1093F, "Lydian", ()),
    (0x10980, 0x1099F, "Meroitic Hieroglyphs", ()),
    (0x109A0, 0x109FF, "Meroitic Cursive", ()),
    (0x10A00, 0x10A5F, "Kharoshthi", ()),
    (0x10A60, 0x10A7F, "Old South Arabian", (ABJAD,)),
    (0x10A80, 0x10A9F, "Old North Arabian", (ABJAD,)),
    (0x10AC0, 0x10AFF, "Manichaean", ()),
    (0x10B00, 0x10B3F, "Avestan", ()),
    (0x10B40, 0x10B5F, "Inscriptional Parthian", ()),
    (0x10B60, 0x10B7F, "Inscriptional Pahlavi", ()),
    (0x10B80, 0x10BAF, "Psalter Pahlavi", (ABJAD,)),
    (0x10C00, 0x10C4F, "Old Turkic", ()),
    (0x10C80, 0x10CFF, "Old Hungarian", ()),
    (0x10D00, 0x10D3F, "Hanifi Rohingya", ()),
    (0x10E60, 0x10E7F, "Rumi Numeral Symbols", ()),
    (
        0x10E80,
        0x10EBF,
        "Yezidi",
        (
            ALPHABET,
            FUNKY,
        ),
    ),
    (
        0x10EC0,
        0x10EFF,
        "Arabic Extended-C",
        (
            ARABIC,
            ABJAD,
        ),
    ),
    (0x10F00, 0x10F2F, "Old Sogdian", (ABJAD,)),
    (0x10F30, 0x10F6F, "Sogdian", (ABJAD,)),
    (0x10F70, 0x10FAF, "Old Uyghur", ()),
    (0x10FB0, 0x10FDF, "Chorasmian", ()),
    (0x10FE0, 0x10FFF, "Elymaic", ()),
    (0x11000, 0x1107F, "Brahmi", (ABUGIDA,)),
    (0x11080, 0x110CF, "Kaithi", ()),
    (0x110D0, 0x110FF, "Sora Sompeng", ()),
    (0x11100, 0x1114F, "Chakma", ()),
    (0x11150, 0x1117F, "Mahajani", ()),
    (0x11180, 0x111DF, "Sharada", ()),
    (0x111E0, 0x111FF, "Sinhala Archaic Numbers", ()),
    (0x11200, 0x1124F, "Khojki", ()),
    (0x11280, 0x112AF, "Multani", ()),
    (0x112B0, 0x112FF, "Khudawadi", ()),
    (0x11300, 0x1137F, "Grantha", ()),
    (0x11400, 0x1147F, "Newa", ()),
    (0x11480, 0x114DF, "Tirhuta", ()),
    (0x11580, 0x115FF, "Siddham", ()),
    (0x11600, 0x1165F, "Modi", ()),
    (0x11660, 0x1167F, "Mongolian Supplement", ()),
    (0x11680, 0x116CF, "Takri", ()),
    (0x11700, 0x1174F, "Ahom", ()),
    (0x11800, 0x1184F, "Dogra", ()),
    (0x118A0, 0x118FF, "Warang Citi", ()),
    (0x11900, 0x1195F, "Dives Akuru", ()),
    (0x119A0, 0x119FF, "Nandinagari", ()),
    (0x11A00, 0x11A4F, "Zanabazar Square", ()),
    (0x11A50, 0x11AAF, "Soyombo", ()),
    (
        0x11AB0,
        0x11ABF,
        "Unified Canadian Aboriginal Syllabics Extended-A",
        (SYLLABARY,),
    ),
    (0x11AC0, 0x11AFF, "Pau Cin Hau", ()),
    (0x11B00, 0x11B5F, "Devanagari Extended-A", ()),
    (0x11C00, 0x11C6F, "Bhaiksuki", ()),
    (0x11C70, 0x11CBF, "Marchen", ()),
    (0x11D00, 0x11D5F, "Masaram Gondi", ()),
    (0x11D60, 0x11DAF, "Gunjala Gondi", ()),
    (0x11EE0, 0x11EFF, "Makasar", ()),
    (0x11F00, 0x11F5F, "Kawi", ()),
    (0x11FB0, 0x11FBF, "Lisu Supplement", ()),
    (0x11FC0, 0x11FFF, "Tamil Supplement", ()),
    (0x12000, 0x123FF, "Cuneiform", ()),
    (0x12400, 0x1247F, "Cuneiform Numbers and Punctuation", ()),
    (0x12480, 0x1254F, "Early Dynastic Cuneiform", ()),
    (0x12F90, 0x12FFF, "Cypro-Minoan", ()),
    (0x13000, 0x1342F, "Egyptian Hieroglyphs", ()),
    (0x13430, 0x1345F, "Egyptian Hieroglyph Format Controls", ()),
    (0x14400, 0x1467F, "Anatolian Hieroglyphs", ()),
    (0x16800, 0x16A3F, "Bamum Supplement", ()),
    (0x16A40, 0x16A6F, "Mro", ()),
    (0x16A70, 0x16ACF, "Tangsa", ()),
    (0x16AD0, 0x16AFF, "Bassa Vah", ()),
    (0x16B00, 0x16B8F, "Pahawh Hmong", ()),
    (0x16E40, 0x16E9F, "Medefaidrin", ()),
    (0x16F00, 0x16F9F, "Miao", ()),
    (0x16FE0, 0x16FFF, "Ideographic Symbols and Punctuation", ()),
    (0x17000, 0x187FF, "Tangut", ()),
    (0x18800, 0x18AFF, "Tangut Components", ()),
    (0x18B00, 0x18CFF, "Khitan Small Script", ()),
    (0x18D00, 0x18D7F, "Tangut Supplement", ()),
    (0x1AFF0, 0x1AFFF, "Kana Extended-B", ()),
    (0x1B000, 0x1B0FF, "Kana Supplement", ()),
    (0x1B100, 0x1B12F, "Kana Extended-A", ()),
    (0x1B130, 0x1B16F, "Small Kana Extension", ()),
    (0x1B170, 0x1B2FF, "Nushu", ()),
    (0x1BC00, 0x1BC9F, "Duployan", ()),
    (0x1BCA0, 0x1BCAF, "Shorthand Format Controls", ()),
    (0x1CF00, 0x1CFCF, "Znamenny Musical Notation", ()),
    (0x1D000, 0x1D0FF, "Byzantine Musical Symbols", ()),
    (0x1D100, 0x1D1FF, "Musical Symbols", ()),
    (0x1D200, 0x1D24F, "Ancient Greek Musical Notation", ()),
    (0x1D2C0, 0x1D2DF, "Kaktovik Numerals", ()),
    (0x1D2E0, 0x1D2FF, "Mayan Numerals", ()),
    (0x1D300, 0x1D35F, "Tai Xuan Jing Symbols", ()),
    (0x1D360, 0x1D37F, "Counting Rod Numerals", ()),
    (0x1D400, 0x1D7FF, "Mathematical Alphanumeric Symbols", ()),
    (0x1D800, 0x1DAAF, "Sutton SignWriting", ()),
    (
        0x1DF00,
        0x1DFFF,
        "Latin Extended-G",
        (
            ALPHABET,
            LATIN,
        ),
    ),
    (0x1E000, 0x1E02F, "Glagolitic Supplement", ()),
    (
        0x1E030,
        0x1E08F,
        "Cyrillic Extended-D",
        (
            ALPHABET,
            CYRILLIC,
        ),
    ),
    (0x1E100, 0x1E14F, "Nyiakeng Puachue Hmong", ()),
    (0x1E290, 0x1E2BF, "Toto", ()),
    (0x1E2C0, 0x1E2FF, "Wancho", ()),
    (0x1E4D0, 0x1E4FF, "Nag Mundari", ()),
    (0x1E7E0, 0x1E7FF, "Ethiopic Extended-B", (ABUGIDA,)),
    (0x1E800, 0x1E8DF, "Mende Kikakui", ()),
    (0x1E900, 0x1E95F, "Adlam", ()),
    (0x1EC70, 0x1ECBF, "Indic Siyaq Numbers", ()),
    (0x1ED00, 0x1ED4F, "Ottoman Siyaq Numbers", ()),
    (
        0x1EE00,
        0x1EEFF,
        "Arabic Mathematical Alphabetic Symbols",
        (
            ARABIC,
            ABJAD,
        ),
    ),
    (0x1F000, 0x1F02F, "Mahjong Tiles", ()),
    (0x1F030, 0x1F09F, "Domino Tiles", ()),
    (0x1F0A0, 0x1F0FF, "Playing Cards", ()),
    (0x1F100, 0x1F1FF, "Enclosed Alphanumeric Supplement", ()),
    (0x1F200, 0x1F2FF, "Enclosed Ideographic Supplement", ()),
    (0x1F300, 0x1F5FF, "Miscellaneous Symbols and Pictographs", ()),
    (0x1F600, 0x1F64F, "Emoticons", ()),
    (0x1F650, 0x1F67F, "Ornamental Dingbats", ()),
    (0x1F680, 0x1F6FF, "Transport and Map Symbols", ()),
    (0x1F700, 0x1F77F, "Alchemical Symbols", ()),
    (0x1F780, 0x1F7FF, "Geometric Shapes Extended", ()),
    (0x1F800, 0x1F8FF, "Supplemental Arrows-C", ()),
    (0x1F900, 0x1F9FF, "Supplemental Symbols and Pictographs", ()),
    (0x1FA00, 0x1FA6F, "Chess Symbols", ()),
    (0x1FA70, 0x1FAFF, "Symbols and Pictographs Extended-A", ()),
    (0x1FB00, 0x1FBFF, "Symbols for Legacy Computing", ()),
    (0x20000, 0x2A6DF, "CJK Unified Ideographs Extension B", (CJK,)),
    (0x2A700, 0x2B73F, "CJK Unified Ideographs Extension C", (CJK,)),
    (0x2B740, 0x2B81F, "CJK Unified Ideographs Extension D", (CJK,)),
    (0x2B820, 0x2CEAF, "CJK Unified Ideographs Extension E", (CJK,)),
    (0x2CEB0, 0x2EBEF, "CJK Unified Ideographs Extension F", (CJK,)),
    (0x2EBF0, 0x2EE5F, "CJK Unified Ideographs Extension I", (CJK,)),
    (0x2F800, 0x2FA1F, "CJK Compatibility Ideographs Supplement", (CJK,)),
    (0x30000, 0x3134F, "CJK Unified Ideographs Extension G", (CJK,)),
    (0x31350, 0x323AF, "CJK Unified Ideographs Extension H", (CJK,)),
    (0xE0000, 0xE007F, "Tags", ()),
    (0xE0100, 0xE01EF, "Variation Selectors Supplement", ()),
    (0xF0000, 0xFFFFF, "Supplementary Private Use Area-A", ()),
    (0x100000, 0x10FFFF, "Supplementary Private Use Area-B", ()),
)
BLOCK_TAGS = [(s, e, t) for s, e, _, t in UNICODE_BLOCKS if len(t)]


@lru_cache(maxsize=5000)
def char_tags(char: str) -> Tuple[int, ...]:
    """Get the tags applicable to a particular character."""
    codepoint = ord(char)
    for start, end, tags in BLOCK_TAGS:
        if start <= codepoint <= end:
            return tags
    return ()


def is_modern_alphabet(word: str) -> bool:
    """Check if a word is written in a modern alphabet. The term alphabet is
    used in a narrow sense here: it includes only alphabets that have vowels and
    are safely transliterated to latin. Basically: Cyrillic, Greek, Armenian,
    and Latin."""
    for char in word:
        tags = char_tags(char)
        if not len(tags):
            continue
        if ALPHABET not in tags:
            return False
        if HISTORIC in tags or FUNKY in tags:
            return False
    return True


def is_latin(word: str) -> bool:
    """Check if a word is written in the latin alphabet."""
    for char in word:
        tags = char_tags(char)
        if not len(tags):
            continue
        if LATIN not in tags:
            return False
        if HISTORIC in tags or FUNKY in tags:
            return False
    return True
