// license:CC0-1.0
// copyright-holders:Colin Douglas Howell

#include "netlist/devices/net_lib.h"


#define USE_FRONTIERS   1
#define FAST_SWITCHES   1

// This is a netlist description for the sound circuits of Midway's Gun Fight,
// based on Midway's schematic "Gun Fight Sound Generator Detail P.C.
// 597-907E".
//
// (This sound circuitry seems to have evolved from that for an older Midway
// game, an electromechanical rifle game from 1974 called Twin Pirate Gun. The
// schematics for Twin Pirate Gun's sound circuitry (P.C. 569-907-88),
// although not completely identical to Gun Fight's, are similar both in
// general structure and in many details.)
//
// Gun Fight's sound effects are simple "bang" sounds, both for the shooting
// of bullets and for bullets hitting targets. The effects are directed to the
// left or right speaker, depending on whether the left or right player is
// shooting or being hit. (Hits to obstacles get the same sound as hits to the
// right player and thus play from the right speaker.) Each sound effect gets
// a different pitch, with shots being higher pitched than hits. Shot sounds
// also decay faster than hit sounds and have slightly separated initial and
// secondary attacks, resulting in a "ka-pow" effect.
//
// The sounds are generated in stages by different sections of the circuitry.
// A noise generator, based on a zener diode, continuously produces white
// noise that forms the basis for all the sounds, and this noise is fed to
// four separate sound generators. Each generator filters the noise to produce
// the distinct pitch of its sound. Each generator's sound is triggered by a
// switch, activated digitally by the CPU. When this switch is turned on
// momentarily, a storage capacitor is charged up, and that sends power to the
// sound generator's transistor amplifier. This amplifies the filtered noise
// with a sharp attack as the switch is turned on and gradual decay as the
// capacitor drains. The generated sound is filtered further and then
// attenuated by a potentiometer which controls that sound's relative volume.
// Each side's two sound effect signals, after being generated and then
// attenuated by their respective volume pots, are mixed together and then
// further amplified by a second-stage transistor amplifier for that side.
// This mixed and amplified signal is itself filtered and then attenuated by
// the side's master volume potentiometer. Finally it gets sent to the power
// amplifier IC for that side's speaker. (This emulation does not handle the
// final power amplification.)
//
// The different sound effect generators have a common structure which is
// repeated four times. The mixers and second-stage amplifiers likewise have a
// common structure which is repeated twice, and their amplifiers are also
// rather similar to the effect amplifiers.
//
// To make the netlist easier to follow and verify, I've grouped its
// components and connections by function, with the groups ordered roughly
// according to the flow of signals:
//
// * the activating switches,
// * the shared noise generator,
// * the sound-effect noise filters,
// * the sound-effect initial amplifiers, output coupling capacitors and
//   filters, and effect-volume potentiometers,
// * the mixing circuits,
// * the second-stage amplifiers, output coupling capacitors and filters, and
//   master-volume potentiometers.
//
// Within each group, I've placed the sets of four similar components or
// connections together and listed them in the order: left-shot, right-shot,
// left-hit, right-hit. (For mixing and second-stage amplification components,
// the order is simply left, right.) Individual components are labeled the
// same as on the schematic.


// There are some uncertainties in this emulation about sound levels. The
// average level of the initially generated noise is uncertain, which poses a
// problem because all of the sound effects are based on amplifying that
// noise. The noise is generated by passing a small amount of current through
// a zener diode on the verge of its breakdown voltage, which results in very
// noisy current flow. Even though both the type of zener and the average
// current are known, the average strength of the noise is still uncertain,
// both because zener manufacturers do not specify performance under these
// conditions (they recommended zeners for voltage control under more stable,
// relatively noise-free operating conditions) and because the amount of noise
// may vary greatly from one zener to the next, even within the same
// production batch--let alone from one manufacturer to another. This is an
// inherent problem with zener diodes when used for noise generation.
//
// I have chosen a round figure of 2 mV RMS for the zener's average (RMS)
// noise level. Although this is only a guess, it seems to be in the ballpark
// for similar zeners operated with similar amounts of average current. It
// also keeps the noise component of the sound effects strong enough so as not
// to be overwhelmed by the pulse of the sound effects' initial attacks; this
// pulse is created by switching on their generating amplifiers and is
// independent of noise level. Meanwhile, this noise level is still low enough
// that, with all the volume potentiometers set to their midpoints, the noise
// won't be clipped by any subsequent amplification stage in the netlist, up
// to the power amp inputs. (Some such clipping may occur if a sound effect's
// volume is turned up well beyond its midpoint. That may also be true on real
// hardware.)
//
// The other big uncertainty is the audible effect of the power amp ICs. These
// amplifiers add both power and voltage gain, which may be enough to distort
// or clip the output. They may also have an audible filtering effect. In the
// Gun Fight schematic, and apparently in real machines, they are configured
// for a voltage gain of 15. Furthermore, any input voltages beyond a few
// hundred mV (positive or negative) should produce at least some clipping
// distortion, with the distortion getting worse for stronger signals. With
// all potentiometers set to the midpoint, the power amp signal inputs should
// see extreme pulses of +/- 3 V from the initial attack for sound effects,
// and even past those initial attack pulses, the sound effect noise should
// kick in at levels above +/- 1 V on the power amp inputs. If these levels
// are correct and the power amp ICs work as described, the noise of the sound
// effects should initially be heavily distorted, but since the original
// signal is already random noise, it's not clear whether that distortion
// would be apparent. Anyhow, the power amp ICs are completely unemulated
// here, and any distortion effects they might produce would be quite
// different from the hard clipping produced when limiting output levels
// digitally.
//
// I have compromized by setting the volume multipliers for the netlist stream
// outputs so that output levels of +/- 3 volts will produce the maximum
// allowed stream output magnitudes of +/- 32767. Voltages beyond that will be
// clipped. This at least produces some distortion if the volume
// potentiometers are adjusted above their midpoints.
//
// Further improving accuracy would require testing signal levels on actual
// hardware, with allowances made for variations in components, as well as a
// better understanding of the electrical and sonic behavior of the power
// amplifiers, speakers, and cabinet. It's questionable whether doing so is
// worth the effort.


// As I've said, this netlist does not include the final stage of sound
// generation, the twin audio power amplifier ICs. Although it is the norm for
// MAME analog sound emulation to omit this stage, some discussion of the
// power amplifiers is worthwhile in this case. Each is an SGS-ATES
// TAA-621-A11 single-channel amplifier, driving one 8-ohm speaker of about 4
// or 5 inches.
//
// (On the Gun Fight schematic, the TAA-621-A11 is also labeled "LM354". The
// TAA-621-A11 was introduced in 1970 by the Italian firm SGS. The very
// similar LM354 was introduced in 1972 by European Electronic Products.
// However, this company seems to have been a mere U.S. importer and
// distributor of European components, making the LM354 just a rebadged
// TAA-621-A11. The LM354 occasionally comes up in discussions of Gun Fight
// audio but seems to have been otherwise short-lived. One problem with its
// name is that it can easily be mistaken for a National Semiconductor LMxxx
// linear IC, even though it's unrelated to that company. To further confuse
// matters, National Semiconductor had already introduced an LM354 of their
// own in 1970 which wasn't an audio amplifier at all, but a second-source
// version of Texas Instruments' SN7525 sense amplifier for minicomputer
// magnetic-core memory systems.)
//
// As is normal for TAA-621-A11 amps, those in Gun Fight are configured with
// some negative feedback to control their voltage gain. The gain is
// determined by the ratio of the chip's internal 15-kohm feedback resistor to
// the external resistor connected to the feedback terminal, pin 10. In Gun
// Fight this external resistor is 1 kohm, so the ratio, and thus the voltage
// gain, is 15.
//
// The TAA-621-A11 is rated at 4 watts at 10% total harmonic distortion (THD)
// when supplied with 24-volt power and driving a 16-ohm load. (Although not
// explicitly stated, this appears to be an RMS rating rather than a peak
// rating.) In Gun Fight's case, the speaker load is only 8 ohms, and the
// power supply voltage is a bit lower, about 22 volts (it comes from
// rectified 16.5-volt-RMS AC, buffered by a large smoothing capacitor). Both
// of these factors lower the power rating somewhat. Also, a power rating at
// 10% THD implies clipping is already heavy. The unclipped, clean power
// rating in Gun Fight would be lower, probably no more than 2 watts RMS,
// giving output around 4 volts RMS or about 5.5 volts in extreme amplitude.
// With a power amp voltage gain of 15, this would mean that any input signals
// with extreme amplitudes greater than about a third of a volt would be
// subject to power amp clipping.
//
// The power amps also have a few other connections, including ripple bypass
// capacitors to suppress ripple from the unregulated power supply, frequency
// compensation capacitors to prevent inducing unstable amplifier oscillations
// at high frequencies, and some filtering on the output to the speakers.


// All of the amplifying transistors in this circuit are NPN transistors of a
// particular type, marked as "A-138" on the Gun Fight schematics. This seems
// to be the Amperex A-138, a low-power silicon NPN transistor made by Amperex
// Electronic Corporation, which was a U.S. division of Philips. This
// transistor, being a vendor-specific type long out of production, seems to
// have fallen into obscurity. The only source I could find with any data on
// it (given in greatly abbreviated form) is _Transistor Discontinued Devices
// D.A.T.A.Book, Edition 16_ (1980), by Derivation and Tabulation Associates,
// which lists the A-138 on p. 76, line 86.
//
// This source shows the A-138 to be an ordinary amplifier transistor of
// fairly high gain, with a maximum small-signal forward current transfer
// ratio (h_fe, AC current gain) of 650. The minimum h_fe is not given, but it
// can be estimated using the fact that such transistors are often graded and
// designated by gain. The A-137, another Amperex transistor listed on line 85
// of the same D.A.T.A.Book page, has the same limits on power, collector
// current, and junction voltages, and thus appears to be in the same series,
// but it has a maximum h_fe of 415. If the A-137 was the next lower grade of
// gain from the A-138, the latter should have a minimum h_fe around 400.
// Values for h_FE, the DC forward current transfer ratio (DC current gain),
// aren't given for the A-138, but the range would be about the same or a bit
// lower, perhaps 350-600, with an average around 450-500.
//
// The high gain of Gun Fight's A-138 transistors causes a "ka-pow" effect in
// its shot sounds; I explain how later. (Andy Welburn, in a Discord chat on
// 2020-06-05, confirmed this effect can be heard in some actual machines. He
// measured an h_FE of 238 for one A-138 from a Gun Fight sound board, but he
// warned that for old transistors like this, h_FE measurements can vary a
// lot, and I'm not surprised that the transistors' gain might be reduced more
// than 40 years later. But even a gain of 238 turns out to be high enough to
// give the "ka-pow" shot effect.)
//
// Considering the A-138's high gain and other limits, a decent match for it
// here seems to be the BC548C, which is modeled in the netlist library. The
// BC548C is the high-gain version of the BC548, a widely used Pro Electron
// standard type of general-purpose small-signal transistor in a through-hole
// TO-92 package. All A-138 transistors in this netlist are modeled as
// BC548Cs.


static NETLIST_START(gunfight_schematics)
{

	// **** Sound effect activation switches.

	// These switches are triggered by digital logic signals activated by
	// the CPU. A high TTL logic level turns the switch on, allowing
	// 16-volt power to flow through the switch into the sound effect
	// generator's amplifier and storage capacitor, and a low logic level
	// turns the switch off, cutting off the power flow. In practice, each
	// sound effect is triggered by turning its switch on for about 50 ms
	// and then switching it off again.
	//
	// Each switch is built from two transistors: a "low-side" NPN
	// transistor which is switched on when a high TTL output level drives
	// the NPN's base high, and a "high-side" PNP transistor which is
	// switched on when the now conducting NPN pulls the PNP's base low.
	// It is the high-side PNP transistor that actually switches the
	// 16-volt power, hence the term "high-side".

#if FAST_SWITCHES

	// Use abstracted activation switches instead of a detailed circuit
	// model of the dual-transistor switches. This gives faster emulation
	// while not making any audible difference in the sound produced.

	SYS_DSW(SW_LEFT_SHOT,  IN_LS, I_V16.Q, R130.1)
	SYS_DSW(SW_RIGHT_SHOT, IN_RS, I_V16.Q, R230.1)
	SYS_DSW(SW_LEFT_HIT,   IN_LH, I_V16.Q, R117.1)
	SYS_DSW(SW_RIGHT_HIT,  IN_RH, I_V16.Q, R217.1)

	// Lower the on-resistance to a more accurate value. The charging
	// resistor which follows is only 15 ohms, so the default
	// on-resistance of 1 ohm might noticeably affect the result.
	PARAM(SW_LEFT_SHOT.RON, 0.1)
	PARAM(SW_RIGHT_SHOT.RON, 0.1)
	PARAM(SW_LEFT_HIT.RON, 0.1)
	PARAM(SW_RIGHT_HIT.RON, 0.1)

#else

	// Detailed circuit model of the dual-transistor switches.

	// "Low-side" NPN transistor switches, driven by TTL logic inputs.

	RES(R134, RES_R(470))
	RES(R234, RES_R(470))
	RES(R121, RES_R(470))
	RES(R221, RES_R(470))

	NET_C(IN_LS, R134.1)
	NET_C(IN_RS, R234.1)
	NET_C(IN_LH, R121.1)
	NET_C(IN_RH, R221.1)

	RES(R133, RES_K(5.1))
	RES(R233, RES_K(5.1))
	RES(R120, RES_K(5.1))
	RES(R220, RES_K(5.1))

	QBJT_SW(Q108, "BC548C")
	QBJT_SW(Q208, "BC548C")
	QBJT_SW(Q105, "BC548C")
	QBJT_SW(Q205, "BC548C")

	// These all go to TTL ground at pin 7 of 7404 IC H6, rather than the
	// ground used for the other sound circuits.
	NET_C(GND,
		  R133.2, R233.2, R120.2, R220.2,
		  Q108.E, Q208.E, Q105.E, Q205.E)

	NET_C(R134.2, R133.1, Q108.B)
	NET_C(R234.2, R233.1, Q208.B)
	NET_C(R121.2, R120.1, Q105.B)
	NET_C(R221.2, R220.1, Q205.B)

	RES(R131, RES_K(1))
	RES(R231, RES_K(1))
	RES(R118, RES_K(1))
	RES(R218, RES_K(1))

	NET_C(Q108.C, R131.1)
	NET_C(Q208.C, R231.1)
	NET_C(Q105.C, R118.1)
	NET_C(Q205.C, R218.1)

	// "High-side" PNP transistor switches, driven by "low-side" NPN
	// switch outputs. The PNP switch outputs charge the storage
	// capacitors that supply power to the sound-effect amplifiers; in
	// addition, while they are on they power the amplifiers directly.

	RES(R132, RES_K(5.1))
	RES(R232, RES_K(5.1))
	RES(R119, RES_K(5.1))
	RES(R219, RES_K(5.1))

	// The actual transistors used here are 2N4125s:
	QBJT_SW(Q107, "PNP")
	QBJT_SW(Q207, "PNP")
	QBJT_SW(Q104, "PNP")
	QBJT_SW(Q204, "PNP")

	// All connected to 16-volt power.
	NET_C(I_V16.Q,
		  R132.1, R232.1, R119.1, R219.1,
		  Q107.E, Q207.E, Q104.E, Q204.E)

	NET_C(R131.2, R132.2, Q107.B)
	NET_C(R231.2, R232.2, Q207.B)
	NET_C(R118.2, R119.2, Q104.B)
	NET_C(R218.2, R219.2, Q204.B)

	NET_C(Q107.C, R130.1)
	NET_C(Q207.C, R230.1)
	NET_C(Q104.C, R117.1)
	NET_C(Q204.C, R217.1)

	// End of switch description.

#endif


	// **** Current supply and storage capacitors for sound-effect
	// **** amplifiers.

	RES(R130, RES_R(15))
	RES(R230, RES_R(15))
	RES(R117, RES_R(15))
	RES(R217, RES_R(15))

	CAP(C122, CAP_U(10))
	CAP(C222, CAP_U(10))
	CAP(C116, CAP_U(20))
	CAP(C216, CAP_U(20))
	NET_C(GND, C122.2, C222.2, C116.2, C216.2)

	NET_C(R130.2, C122.1, R126.1, R124.1)
	NET_C(R230.2, C222.1, R226.1, R224.1)
	NET_C(R117.2, C116.1, R113.1, R111.1)
	NET_C(R217.2, C216.1, R213.1, R211.1)


	// **** Shared white-noise generator circuit. This is the basic noise
	// **** source which is filtered and amplified by the sound-effect
	// **** circuits.

	// Gun Fight's noise generator circuit is based on a reverse-biased
	// 9.1-volt zener diode (D304, a 1N5239) whose noise current is then
	// amplified by an A-138 NPN transistor, producing white noise at
	// audio frequencies.
	//
	// (Strictly speaking, this is not a *pure* white noise generator,
	// because the generator's bypass capacitor and biasing resistor, in
	// combination with negative feedback from the amplified signal, act
	// as a high-pass RC filter, filtering out the lowest noise
	// frequencies.)
	//
	// Figuring out how strong the noise signal should be for this circuit
	// is difficult. The noise generator's biasing resistors and
	// transistor gain limit the average current through its zener to
	// around 2 microamps, but the 1N5239 is normally expected to be used
	// with much larger currents, in the range of 250 microamps to 50
	// milliamps.
	//
	// Zeners are most often used for smoothly controlling and limiting
	// voltage with minimal fluctuation, like in a power regulator. But a
	// zener can only do this smoothly if it passes a large enough
	// current. This is especially true for "zeners" of voltages of 9.1
	// volts or more, which properly speaking are "avalanche diodes" that
	// don't user the actual Zener effect but rather a different effect,
	// avalanche breakdown, which is capable of generating far more noise.
	// Standard zener specifications include "knee" figures which in
	// effect give the minimum expected current, along with the maximum
	// "dynamic impedance" at that current: how much the voltage may vary
	// if the current changes, or vice versa. For the 1N5239, the knee is
	// at 250 microamps, and the maximum dynamic impedance at the knee is
	// 600 ohms, so a 1 microamp increase in current at that point would
	// raise the voltage drop by up to 0.6 millivolts. Noise values for
	// zeners are often not given at all, or if they are (as with
	// Motorola's zeners), they are given at the knee current. In short,
	// the manufacturer expects the current to be kept above the knee
	// value, and if your current is lower, you're on your own.
	//
	// At very low currents, still within the breakdown region but well
	// below the knee current, the zener's dynamic impedance is much
	// greater, and more importantly, so is its noise. The conduction of
	// current is no longer smooth and regular, but pulsing and episodic,
	// as small conducting "microplasmas" within the diode's junction are
	// repeatedly triggered and extinguished, like microscopic versions of
	// lightning bolts in a thunderstorm.
	//
	// The netlist library includes a Zener diode model, but this model
	// does not simulate the Zener's noise behavior. Instead I generate
	// the noise from a noise voltage source in series with the Zener.

	// Simple model of a 1N5239 9.1-volt Zener diode. The 1N5239 is
	// specified to conduct 20 mA of current at its nominal breakdown
	// voltage of 9.1 V. The model produces an exponential I-V curve,
	// passing through this point, which has the same general shape as
	// that of a normal forward-biased diode. NBV is an exponent scale
	// factor; its value here of 1 gives the curve a steep rise and a
	// relatively sharp knee. Actual breakdown I-V curves have an even
	// steeper rise and sharper knee, too steep and sharp to be
	// represented by an exponential, but this model is good enough for
	// this emulation, since the diode operates very close to a single
	// point on the curve.
	ZDIODE(D304, "D(BV=9.1 IBV=0.020 NBV=1)")

	// 24 kHz noise clock for the noise source, chosen to retain noise
	// frequencies as high as possible for 48 kHz sample rate.
	CLOCK(NCLK, 24000)
	NET_C(I_V5.Q, NCLK.VCC)
	NET_C(GND, NCLK.GND)

	// Normally-distributed noise of 2 millivolts RMS voltage.
	// This level was chosen to have a strong amplified noise signal that
	// won't be clipped by any subsequent stages of amplification before
	// the power amps, if the volume potentiometers are not raised beyond
	// their approximate midpoints.
	SYS_NOISE_MT_N(NOISE, 0.002)

	NET_C(NCLK.Q, NOISE.I)

	RES(R302, RES_K(6.8))
	RES(R303, RES_K(6.8))
	CAP(C307, CAP_U(10))
	NET_C(C307.2, GND)
	QBJT_EB(Q302, "BC548C")
	NET_C(Q302.E, GND)

	NET_C(I_V16.Q, R302.1)
	NET_C(Q302.B, NOISE.1)
	NET_C(D304.A, NOISE.2)
	NET_C(R303.2, C307.1, D304.K)

	// Coupling capacitors from noise generator to sound effect frequency
	// filters. (These coupling capacitors, together with the resistances
	// beyond them, act as high-pass filters with very low cutoff
	// frequencies.)

	CAP(C303, CAP_U(0.1))
	CAP(C306, CAP_U(0.1))
	CAP(C304, CAP_U(0.1))
	CAP(C305, CAP_U(0.1))

	NET_C(R302.2, Q302.C, R303.1, C303.1, C306.1, C304.1, C305.1)


	// **** Sound effect frequency filters.

	// Each sound effect has a pair of passive low-pass RC filters with
	// cutoff frequencies determined by their component values. The
	// different capacitor values produce each sound effect's distinct
	// pitch.

	RES(R129, RES_K(20))
	RES(R229, RES_K(20))
	RES(R116, RES_K(20))
	RES(R216, RES_K(20))

	NET_C(C303.2, R129.1)
	NET_C(C306.2, R229.1)
	NET_C(C304.2, R116.1)
	NET_C(C305.2, R216.1)

	CAP(C121, CAP_U(0.015))
	CAP(C221, CAP_U(0.015))
	CAP(C115, CAP_U(0.033))
	CAP(C215, CAP_U(0.033))
	NET_C(C121.2, C221.2, C115.2, C215.2, GND)

	RES(R128, RES_K(30))
	RES(R228, RES_K(30))
	RES(R115, RES_K(30))
	RES(R215, RES_K(30))

	NET_C(R129.2, C121.1, R128.1)
	NET_C(R229.2, C221.1, R228.1)
	NET_C(R116.2, C115.1, R115.1)
	NET_C(R216.2, C215.1, R215.1)

	CAP(C120, CAP_U(0.01))
	CAP(C220, CAP_U(0.015))
	CAP(C114, CAP_U(0.1))
	CAP(C214, CAP_U(0.047))
	NET_C(C120.2, C220.2, C114.2, C214.2, GND)


	// **** Sound effect amplifier circuits.

	// Each sound-effect amplifier is a single NPN transistor wired as a
	// common-emitter amplifier. The amplifiers for "hit" sounds also have
	// a bypass capacitor at the emitter, while those for "shot" sounds
	// have no bypass capacitor and a much lower emitter resistance. The
	// attack and decay of the sound effects is handled by controlling the
	// current supply to each amplifier, which is done by the switching
	// circuits and supply capacitors described above.

	// More explanation is needed for the "shot" sounds. Apart from their
	// higher frequency and faster decay, the "ka-pow" effect in their
	// initial attack further distinguishes them from the "hit" sounds.
	// This effect comes from the high current gain (around 450-500) of
	// the amplifier's A-138 transistor together with the low emitter
	// resistance. When the current supply for the sound is switched on,
	// the collector voltage at first spikes upward as the supply
	// capacitor is charged. But the transistor's base voltage and base
	// current also rise, which "turns on" the transistor, and as its
	// collector current increases through its biasing resistor, the
	// collector voltage plummets. For the "shot" sound transistors,
	// because of their high current gain and low emitter resistance, the
	// collector current grows so much that the collector voltage is
	// pulled below the base voltage, pushing the transistor into
	// saturation. This persists for as long as the current supply switch
	// remains on; the collector voltage stays low with little variation.
	// In this state the amplifier's input noise signal is being clipped
	// rather than amplified.

	// The result is that the sound effect's initial voltage spike is
	// followed by a relatively prolonged low with almost no noise. As
	// this signal passes through the output coupling capacitor, the
	// intervening filter and potentiometer, and then the second
	// amplification stage, it becomes a series of strong oscillations,
	// followed by a momentary silence which lasts as long as the sound's
	// switch is held on: around 50 milliseconds.

	// Finally the sound switch is turned off, and the amplifier's supply
	// voltage and current begin to drop as the supply capacitor
	// discharges. The base current and collector current drop also, and
	// the collector voltage begins to rise, eventually rising above the
	// base voltage again. The transistor leaves saturation and returns to
	// forward active mode. Now the noise signal is not being clipped but
	// gets properly amplified on the collector output. So the momentary
	// silence is followed by a sudden burst of noise, which then dies
	// away as the supply capacitor is drained.

	// The result is the shot's distinct "ka-pow" sound: an initial
	// punctuating crack, a very brief silence, and a sudden noise burst
	// that quickly fades.

	// The "hit" sounds don't have this effect, despite using the same
	// high-gain transistors, because their transistor amplifiers have a
	// bypass capacitor and a much larger emitter resistance, 1 Kohm
	// versus 100 ohms. That higher resistance keeps the collector current
	// low enough that the collector voltage never drops below the base
	// voltage, so the transistor never saturates, while the bypass
	// capacitor allows the amplifier's AC gain to remain very high.

	RES(R126, RES_K(330))
	RES(R226, RES_K(330))
	RES(R113, RES_K(330))
	RES(R213, RES_K(330))

	RES(R127, RES_K(30))
	RES(R227, RES_K(30))
	RES(R114, RES_K(30))
	RES(R214, RES_K(30))
	NET_C(R127.2, R227.2, R114.2, R214.2, GND)

	RES(R124, RES_K(5.1))
	RES(R224, RES_K(5.1))
	RES(R111, RES_K(5.1))
	RES(R211, RES_K(5.1))

	RES(R125, RES_R(100))
	RES(R225, RES_R(100))
	RES(R112, RES_K(1))
	RES(R212, RES_K(1))
	NET_C(R125.2, R225.2, R112.2, R212.2, GND)

	CAP(C113, CAP_U(50))
	CAP(C213, CAP_U(50))
	NET_C(C113.2, C213.2, GND)

	QBJT_EB(Q106, "BC548C")
	QBJT_EB(Q206, "BC548C")
	QBJT_EB(Q103, "BC548C")
	QBJT_EB(Q203, "BC548C")

	NET_C(R128.2, C120.1, R126.2, R127.1, Q106.B)
	NET_C(R228.2, C220.1, R226.2, R227.1, Q206.B)
	NET_C(R115.2, C114.1, R113.2, R114.1, Q103.B)
	NET_C(R215.2, C214.1, R213.2, R214.1, Q203.B)

	NET_C(R125.1, Q106.E)
	NET_C(R225.1, Q206.E)
	NET_C(R112.1, C113.1, Q103.E)
	NET_C(R212.1, C213.1, Q203.E)


	// **** Coupling capacitors, high-pass (pulse-differentiator) filters,
	// **** and volume potentiometers for sound effect amplifier outputs.

	// These circuits act as high-pass filters on the sound effect
	// generator outputs, with very low cutoff frequencies. Because the
	// cutoff frequency is so low, one of the main effects of the filter
	// is to remove any flat areas from the initial turn-on pulse of the
	// sound effect generator amplifier. The filter effectively
	// differentiates the pulse, producing output voltage proportional to
	// the steepness of its slope. This replaces the single wide pulse of
	// the initial attack with a sequence of sharp spike pulses.

	CAP(C119, CAP_U(0.047))
	CAP(C219, CAP_U(0.047))
	CAP(C112, CAP_U(0.1))
	CAP(C212, CAP_U(0.1))

	NET_C(R124.2, Q106.C, C119.1)
	NET_C(R224.2, Q206.C, C219.1)
	NET_C(R111.2, Q103.C, C112.1)
	NET_C(R211.2, Q203.C, C212.1)

	CAP(C118, CAP_U(0.022))
	CAP(C218, CAP_U(0.022))
	CAP(C111, CAP_U(0.033))
	CAP(C211, CAP_U(0.033))
	NET_C(C118.2, C218.2, C111.2, C211.2, GND)

	// There are four sound-effect volume pots, for shot and hit sounds on
	// left and right.

	POT(R123, RES_K(50))
	POT(R223, RES_K(50))
	POT(R110, RES_K(50))
	POT(R210, RES_K(50))
	NET_C(R123.3, R223.3, R110.3, R210.3, GND)

	// Reverse the sense of pot adjustments so that larger values result
	// in greater volume.
	PARAM(R123.REVERSE, 1)
	PARAM(R223.REVERSE, 1)
	PARAM(R110.REVERSE, 1)
	PARAM(R210.REVERSE, 1)

	NET_C(C119.2, C118.1, R123.1)
	NET_C(C219.2, C218.1, R223.1)
	NET_C(C112.2, C111.1, R110.1)
	NET_C(C212.2, C211.1, R210.1)


	// **** Mixing of shot and hit sounds for each side.

	RES(R122, RES_K(30))
	RES(R222, RES_K(30))
	RES(R109, RES_K(30))
	RES(R209, RES_K(30))

	NET_C(R123.2, R122.2)
	NET_C(R223.2, R222.2)
	NET_C(R110.2, R109.2)
	NET_C(R210.2, R209.2)

	CAP(C117, CAP_U(0.047))
	CAP(C217, CAP_U(0.047))
	CAP(C110, CAP_U(0.1))
	CAP(C210, CAP_U(0.1))

	NET_C(R122.1, C117.2)
	NET_C(R222.1, C217.2)
	NET_C(R109.1, C110.2)
	NET_C(R209.1, C210.2)


	// **** Second-stage amplifier circuits, which amplify each side's
	// **** mixed shot and hit sounds.

	// These amplifiers are similar to those for the "hit" sound effects,
	// each being a single A-138 NPN transistor wired in common-emitter
	// configuration, with a 1-Kohm resistance and a bypass capacitor at
	// the emitter. They have no need for an attack-decay envelope,
	// however, and so get their current directly from the 16-volt power
	// supply.

	RES(R107, RES_K(150))
	RES(R207, RES_K(150))

	RES(R105, RES_K(30))
	RES(R205, RES_K(30))
	NET_C(R105.2, R205.2, GND)

	RES(R108, RES_K(5.1))
	RES(R208, RES_K(5.1))

	RES(R106, RES_K(1))
	RES(R206, RES_K(1))
	NET_C(R106.2, R206.2, GND)

	CAP(C109, CAP_U(50))
	CAP(C209, CAP_U(50))
	NET_C(C109.2, C209.2, GND)

	NET_C(R107.1, R207.1, R108.1, R208.1, I_V16.Q)

	QBJT_EB(Q102, "BC548C")
	QBJT_EB(Q202, "BC548C")

	NET_C(C110.1, C117.1, R107.2, R105.1, Q102.B)
	NET_C(C210.1, C217.1, R207.2, R205.1, Q202.B)

	NET_C(R106.1, C109.1, Q102.E)
	NET_C(R206.1, C209.1, Q202.E)


	// **** Coupling capacitors, bandpass filters, and volume
	// **** potentiometers for second-stage amplifier outputs.

	CAP(C108, CAP_U(0.047))
	CAP(C208, CAP_U(0.047))

	NET_C(R108.2, Q102.C, C108.1)
	NET_C(R208.2, Q202.C, C208.1)

	RES(R104, RES_K(30))
	RES(R204, RES_K(30))

	NET_C(C108.2, R104.1)
	NET_C(C208.2, R204.1)

	CAP(C107, CAP_U(0.001))
	CAP(C207, CAP_U(0.001))
	NET_C(C107.2, C207.2, GND)

	// There are two master volume pots, for left and right.

	POT(R103, RES_K(47))
	POT(R203, RES_K(47))
	NET_C(R103.3, R203.3, GND)

	// Reverse the sense of pot adjustments so that larger values result
	// in greater volume.
	PARAM(R103.REVERSE, 1)
	PARAM(R203.REVERSE, 1)

	NET_C(R104.2, C107.1, R103.1)
	NET_C(R204.2, C207.1, R203.1)

	// The potentiometer outputs are used here as the left and right audio
	// outputs. In the real circuit they drive the signal inputs of the
	// audio power amplifier ICs for the left and right speakers.

	ALIAS(OUT_L, R103.2)
	ALIAS(OUT_R, R203.2)

	// The real outputs are somewhat constrained in that they drive the
	// bases of the input transistors within the power amplifiers. If they
	// go too low in voltage, there seems to be a peculiar effect on the
	// speaker output waveforms, although I'm not sure whether this is a
	// real effect or an artifact of the LTspice simulation I constructed.
	// Nor am I sure whether it matters in practice. In any case, it's not
	// modeled here.

}


NETLIST_START(gunfight)
{

	SOLVER(Solver, 48000)
	PARAM(Solver.SORT_TYPE, "ASCENDING")
	// For this netlist, ASCENDING turns out to be slightly faster than
	// the default sort type of PREFER_IDENTITY_TOP_LEFT, but the
	// difference when using static solvers is very small.

	LOCAL_SOURCE(gunfight_schematics)

	INCLUDE(gunfight_schematics)

	// The amplifying transistors all get 16-volt power. The raw AC power
	// input from the main power supply to the game logic board is 16.5
	// volts, but this is rectified and regulated to about 16 volts via
	// TIP-31 power transistor Q301 and BZX61-C16 16-volt Zener diode
	// D304.
	ANALOG_INPUT(I_V16, 16)  // 16-volt power for sound amplifiers
	ANALOG_INPUT(I_V5, 5)  // 5-volt power for logic input devices

	LOGIC_INPUT(I_LEFT_SHOT,  0, "74XX")
	LOGIC_INPUT(I_RIGHT_SHOT, 0, "74XX")
	LOGIC_INPUT(I_LEFT_HIT,   0, "74XX")
	LOGIC_INPUT(I_RIGHT_HIT,  0, "74XX")

	// Power and ground connections for logic input devices:
	NET_C(I_V5.Q,
		  I_LEFT_SHOT.VCC, I_RIGHT_SHOT.VCC,
		  I_LEFT_HIT.VCC, I_RIGHT_HIT.VCC)
	NET_C(GND,
		  I_LEFT_SHOT.GND, I_RIGHT_SHOT.GND,
		  I_LEFT_HIT.GND, I_RIGHT_HIT.GND)

	ALIAS(IN_LS, I_LEFT_SHOT.Q)
	ALIAS(IN_RS, I_RIGHT_SHOT.Q)
	ALIAS(IN_LH, I_LEFT_HIT.Q)
	ALIAS(IN_RH, I_RIGHT_HIT.Q)

#if USE_FRONTIERS
	// These frontiers keep the mostly independent halves of the circuit
	// (left channel and right channel) from affecting each other and the
	// noise generator, which speeds up processing substantially while
	// making no audible change in the output. These seem to be the only
	// frontiers which improve performance; I haven't been able to find
	// any additional beneficial ones from examining the circuit and
	// experimenting.
	OPTIMIZE_FRONTIER(C303.1, RES_M(1), 50)
	OPTIMIZE_FRONTIER(C306.1, RES_M(1), 50)
	OPTIMIZE_FRONTIER(C304.1, RES_M(1), 50)
	OPTIMIZE_FRONTIER(C305.1, RES_M(1), 50)
#endif

}
