/* Copyright (C) 2006 to 2010 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

However, it is not intended that the object code of a program whose
source code instantiates a template from this file or uses macros or
inline functions (of any length) should by reason only of that
instantiation or use be subject to the restrictions of use in the GNU
Lesser General Public License.  With that in mind, the words "and
macros, inline functions and instantiations of templates (of any
length)" shall be treated as substituted for the words "and small
macros and small inline functions (ten lines or less in length)" in
the fourth paragraph of section 5 of that licence.  This does not
affect any other reason why object code may be subject to the
restrictions in that licence (nor for the avoidance of doubt does it
affect the application of section 2 of that licence to modifications
of the source code in this file).

*/

/**
 * @file async_queue.h
 * @brief This file provides thread-safe asynchronous queue classes.
 *
 * AsyncQueue is a class which provides some of the functionality of a
 * std::queue object (but note that the AsyncQueue::pop(value_type&
 * obj) method provides the pop()ed element by reference - see the
 * comments on that method for the reason), except that it has mutex
 * locking of the data container so as to permit push()ing and
 * pop()ing from different threads.  It is therefore useful for
 * passing data between threads, perhaps in response to a signal being
 * emitted from a Notifier object.  Passing the data by means of a
 * SharedLockPtr object, or an IntrusivePtr object referencing data
 * derived from IntrusiveLockCounter, would be ideal.
 *
 * AsyncQueueDispatch is a class which has a blocking pop() method,
 * which allows it to be waited on by a dedicated event/message
 * dispatching thread for incoming work (represented by the data
 * pushed onto the queue).  In the same way, it can be used to
 * implement thread pools, by having threads in the pool waiting on
 * the queue.
 *
 * By default the queues use a std::list object as their container
 * because in the kind of use mentioned above they are unlikely to
 * hold many objects but they can be changed to, say, a std::deque
 * object by specifying it as the second template parameter.
 */

#ifndef CGU_ASYNC_QUEUE_H
#define CGU_ASYNC_QUEUE_H

#include <queue>
#include <list>
#include <exception>
#include <time.h>
#include <c++-gtk-utils/mutex.h>
#include <c++-gtk-utils/thread.h>
#include <c++-gtk-utils/cgu_config.h>

namespace Cgu {

/**
 * @class AsyncQueuePopError async_queue.h c++-gtk-utils/async_queue.h
 * @brief An exception thrown if calling pop() on a AsyncQueue or
 * AsyncQueueDispatch object fails because the queue is empty.
 * @sa AsyncQueue AsyncQueueDispatch
 */

struct AsyncQueuePopError: public std::exception {
  virtual const char* what() const throw() {return "AsyncQueuePopError: popping from empty AsyncQueue object\n";}
};


/**
 * @class AsyncQueue async_queue.h c++-gtk-utils/async_queue.h
 * @brief A thread-safe asynchronous queue.
 * @sa AsyncQueueDispatch
 *
 * AsyncQueue is a class which provides some of the functionality of a
 * std::queue object (but note that the AsyncQueue::pop(value_type&
 * obj) method provides the pop()ed element by reference - see the
 * comments on that method for the reason), except that it has mutex
 * locking of the data container so as to permit push()ing and
 * pop()ing from different threads.  It is therefore useful for
 * passing data between threads, perhaps in response to a signal being
 * emitted from a Notifier object.  Passing the data by means of a
 * SharedLockPtr object, or an IntrusivePtr object referencing data
 * derived from IntrusiveLockCounter, would be ideal.
 *
 * By default the queue uses a std::list object as its container
 * because in the kind of use mentioned above it is unlikely to hold
 * many objects but it can be changed to, say, a std::deque object by
 * specifying it as the second template parameter.
 *
 * If the library is installed using the
 * --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat configuration options, any
 * AsyncQueue objects constructed on free store will be constructed in
 * glib memory slices.  This does not affect the queue container
 * itself: to change the allocator of the C++ container, a custom
 * allocator type can be provided when the AsyncQueue object is
 * instantiated offering the standard allocator interface.
 */

template <class T, class Container = std::list<T> > class AsyncQueue {
public:
  typedef typename Container::value_type value_type;
  typedef typename Container::size_type size_type;
  typedef Container container_type;
private:
  std::queue<T, Container> q;
  mutable Thread::Mutex mutex;

  // AsyncQueue objects cannot be copied - they are mainly
  // intended to pass data between two known threads
  AsyncQueue(const AsyncQueue&);
  AsyncQueue& operator=(const AsyncQueue&);
public:
/**
 * Pushes an item onto the queue.  This method has strong exception
 * safety if the container is a std::deque, std::list or std::vector
 * container (the default is std::list).
 * @param obj The item to be pushed onto the queue.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  It might
 * also throw if the assignment operator of the queue item might
 * throw.
 */
  void push(const value_type& obj) {
    Thread::Mutex::Lock lock(mutex);
    q.push(obj);
  }

/**
 * Pops an item from the queue.  This method has strong exception
 * safety if the container is a std::deque or std::list container (the
 * default is std::list), provided the destructor of a contained item
 * does not throw.
 * @param obj A value type reference to which the item at the front of
 * the queue will be assigned.
 * @exception AsyncQueuePopError If the queue is empty when a pop is
 * attempted, this method will throw AsyncQueuePopError.  It might
 * also throw if the assignment operator of the queue item might
 * throw.  In order to complete pop() operations atomically under a
 * single lock and to retain strong exception safety, the object into
 * which the pop()ed data is to be placed is passed as an argument by
 * reference (this avoids a copy from a temporary object after the
 * data has been extracted from the queue, which would occur if the
 * item extracted were returned by value).  It might also throw if the
 * destructor of the queue item might throw (but that should never
 * happen), or if the empty() method of the container type throws
 * (which would not happen on any sane implementation).
 */
  void pop(value_type& obj) {
    Thread::Mutex::Lock lock(mutex);
    if (q.empty()) throw AsyncQueuePopError();
    obj = q.front();
    q.pop();
  }

/**
 * Discards the item at the front of the queue.  This method has
 * strong exception safety if the container is a std::deque or
 * std::list container (the default is std::list), provided the
 * destructor of a contained item does not throw.
 * @exception AsyncQueuePopError If the queue is empty when a pop is
 * attempted, this method will throw AsyncQueuePopError.  It might
 * also throw if the destructor of the queue item might throw (but
 * that should never happen), or if the empty() method of the
 * container type throws (which would not happen on any sane
 * implementation).
 */
  void pop() {
    Thread::Mutex::Lock lock(mutex);
    if (q.empty()) throw AsyncQueuePopError();
    q.pop();
  }

/**
 * @return Whether the queue is empty.  It will not throw assuming
 * that the empty() method of the container type does not throw, as it
 * will not on any sane implementation.
 * @note The return value may not be valid if another thread has
 * pushed to or popped from the queue before the value returned by the
 * method is acted on.  It is provided as a utility, but may not be
 * meaningful, depending on the intended usage.
 */
  bool empty() const {
    Thread::Mutex::Lock lock(mutex);
    return q.empty();
  }

/**
 * @exception std::bad_alloc The constructor might throw
 * std::bad_alloc if memory is exhausted and the system throws in that
 * case.
 * @exception Thread::MutexError The constructor might throw
 * Thread::MutexError if initialisation of the contained mutex fails.
 * (It is often not worth checking for this, as it means either memory
 * is exhausted or pthread has run out of other resources to create
 * new mutexes.)
*/
  AsyncQueue() {}

/**
 * The destructor does not throw unless the destructor of a contained
 * item throws.  It is thread-safe (any thread may delete the
 * AsyncQueue object).
 */
  ~AsyncQueue() {
    // lock and unlock the mutex in the destructor so that we have an
    // acquire operation to ensure that when the std::queue object is
    // destroyed memory is synchronised, so any thread may destroy the
    // AsyncQueue object
    Thread::Mutex::Lock lock(mutex);
  }

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

/**
 * @class AsyncQueueDispatch async_queue.h c++-gtk-utils/async_queue.h
 * @brief A thread-safe asynchronous queue with a blocking pop()
 * method.
 * @sa AsyncQueue
 *
 * AsyncQueueDispatch is similar to the AsyncQueue class, except that
 * it has a blocking pop_dispatch() method, which allows it to be
 * waited on by a dedicated event/message dispatching thread for
 * incoming work (represented by the data pushed onto the queue).  In
 * the same way, it can be used to implement thread pools, by having
 * threads in the pool waiting on the queue.
 *
 * By default the queue uses a std::list object as its container
 * because in the kind of use mentioned above it is unlikely to hold
 * many objects but it can be changed to, say, a std::deque object by
 * specifying it as the second template parameter.
 *
 * If the library is installed using the
 * --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat configuration options, any
 * AsyncQueueDispatch objects constructed on free store will be
 * constructed in glib memory slices.  This does not affect the queue
 * container itself: to change the allocator of the C++ container, a
 * custom allocator type can be provided when the AsyncQueueDispatch
 * object is instantiated offering the standard allocator interface.
 */

template <class T, class Container = std::list<T> > class AsyncQueueDispatch {
public:
  typedef typename Container::value_type value_type;
  typedef typename Container::size_type size_type;
  typedef Container container_type;
private:
  std::queue<T, Container> q;
  mutable Thread::Mutex mutex;
  Thread::Cond cond;

  // AsyncQueueDispatch objects cannot be copied - they are mainly
  // intended to pass data between two known threads
  AsyncQueueDispatch(const AsyncQueueDispatch&);
  AsyncQueueDispatch& operator=(const AsyncQueueDispatch&);
public:
/**
 * Pushes an item onto the queue.  This method has strong exception
 * safety if the container is a std::deque, std::list or std::vector
 * container (the default is std::list).
 * @param obj The item to be pushed onto the queue.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  It might
 * also throw if the assignment operator of the queue item might
 * throw.
 */
  void push(const value_type& obj) {
    Thread::Mutex::Lock lock(mutex);
    q.push(obj);
    cond.signal();
  }

/**
 * Pops an item from the queue.  This method has strong exception
 * safety if the container is a std::deque or std::list container (the
 * default is std::list), provided the destructor of a contained item
 * does not throw.
 * @param obj A value type reference to which the item at the front of
 * the queue will be assigned.
 * @exception AsyncQueuePopError If the queue is empty when a pop is
 * attempted, this method will throw AsyncQueuePopError.  It might
 * also throw if the assignment operator of the queue item might
 * throw.  In order to complete pop() operations atomically under a
 * single lock and to retain strong exception safety, the object into
 * which the pop()ed data is to be placed is passed as an argument by
 * reference (this avoids a copy from a temporary object after the
 * data has been extracted from the queue, which would occur if the
 * item extracted were returned by value).  It might also throw if the
 * destructor of the queue item might throw (but that should never
 * happen), or if the empty() method of the container type throws
 * (which would not happen on any sane implementation).
 */
  void pop(value_type& obj) {
    Thread::Mutex::Lock lock(mutex);
    if (q.empty()) throw AsyncQueuePopError();
    obj = q.front();
    q.pop();
  }

/**
 * Pops an item from the queue.  If the queue is empty, it will block
 * until an item becomes available.  If it blocks, the wait comprises
 * a cancellation point.  This method is cancellation safe if the
 * stack unwinds on cancellation, as cancellation is blocked while the
 * queue is being operated on after coming out of a wait.  This method
 * has strong exception safety if the container is a std::deque or
 * std::list container (the default is std::list), provided the
 * destructor of a contained item does not throw.
 * @param obj A value type reference to which the item at the front of
 * the queue will be assigned.  This method might throw if the
 * assignment operator of the queue item might throw.  In order to
 * complete pop() operations atomically under a single lock and to
 * retain strong exception safety, the object into which the pop()ed
 * data is to be placed is passed as an argument by reference (this
 * avoids a copy from a temporary object after the data has been
 * extracted from the queue, which would occur if the item extracted
 * were returned by value).  It might also throw if the destructor of
 * the queue item might throw (but that should never happen), or if
 * the empty() method of the container type throws (which would not
 * happen on any sane implementation).
 */
  void pop_dispatch(value_type& obj) {
    Thread::Mutex::Lock lock(mutex);
    while (q.empty()) cond.wait(mutex);
    Thread::CancelBlock b;
    obj = q.front();
    q.pop();
  }    

/**
 * Pops an item from the queue.  If the queue is empty, it will block
 * until an item becomes available or until the timeout expires.  If
 * it blocks, the wait comprises a cancellation point.  This method is
 * cancellation safe if the stack unwinds on cancellation, as
 * cancellation is blocked while the queue is being operated on after
 * coming out of a wait.  This method has strong exception safety if
 * the container is a std::deque or std::list container (the default
 * is std::list), provided the destructor of a contained item does not
 * throw.
 * @param obj A value type reference to which the item at the front of
 * the queue will be assigned.  This method might throw if the
 * assignment operator of the queue item might throw.  In order to
 * complete pop() operations atomically under a single lock and to
 * retain strong exception safety, the object into which the pop()ed
 * data is to be placed is passed as an argument by reference (this
 * avoids a copy from a temporary object after the data has been
 * extracted from the queue, which would occur if the item extracted
 * were returned by value).  It might also throw if the destructor of
 * the queue item might throw (but that should never happen), or if
 * the empty() method of the container type throws (which would not
 * happen on any sane implementation).
 * @param millisec The timeout interval, in milliseconds.
 * @return If the timeout expires without an item becoming available,
 * the method will return true.  If an item from the queue is
 * extracted, it returns false.
 */
  bool pop_timed_dispatch(value_type& obj, unsigned int millisec) {
    timespec ts;
    Thread::Cond::get_abs_time(ts, millisec);
    Thread::Mutex::Lock lock(mutex);
    while (q.empty()) {
      if (cond.timed_wait(mutex, ts)) return true;
    }
    Thread::CancelBlock b;
    obj = q.front();
    q.pop();
    return false;
  }

/**
 * Discards the item at the front of the queue.  This method has
 * strong exception safety if the container is a std::deque or
 * std::list container (the default is std::list), provided the
 * destructor of a contained item does not throw.
 * @exception AsyncQueuePopError If the queue is empty when a pop is
 * attempted, this method will throw AsyncQueuePopError.  It might
 * also throw if the destructor of the queue item might throw (but
 * that should never happen), or if the empty() method of the
 * container type throws (which would not happen on any sane
 * implementation).
 */
  void pop() {
    Thread::Mutex::Lock lock(mutex);
    if (q.empty()) throw AsyncQueuePopError();
    q.pop();
  }

/**
 * @return Whether the queue is empty.  It will not throw assuming
 * that the empty() method of the container type does not throw, as it
 * will not on any sane implementation.
 * @note The return value may not be valid if another thread has
 * pushed to or popped from the queue before the value returned by the
 * method is acted on.  It is provided as a utility, but may not be
 * meaningful, depending on the intended usage.
*/
  bool empty() const {
    Thread::Mutex::Lock lock(mutex);
    return q.empty();
  }

/**
 * @exception std::bad_alloc The constructor might throw this
 * exception if memory is exhausted and the system throws in that
 * case.
 * @exception Thread::MutexError The constructor might throw this
 * exception if initialisation of the contained mutex fails.  (It is
 * often not worth checking for this, as it means either memory is
 * exhausted or pthread has run out of other resources to create new
 * mutexes.)
 * @exception Thread::CondError The constructor might throw this
 * exception if initialisation of the contained condition variable
 * fails.  (It is often not worth checking for this, as it means
 * either memory is exhausted or pthread has run out of other
 * resources to create new condition variables.)
 */
  AsyncQueueDispatch() {}

/**
 * The destructor does not throw unless the destructor of a contained
 * item throws.  It is thread-safe (any thread may delete the
 * AsyncQueueDispatch object).
 */
  ~AsyncQueueDispatch() {
    // lock and unlock the mutex in the destructor so that we have an
    // acquire operation to ensure that when the std::queue object is
    // destroyed memory is synchronised, so any thread may destroy the
    // AsyncQueueDispatch object
    Thread::Mutex::Lock lock(mutex);
  }

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

} // namespace Cgu

#endif
