/*
 * Copyright (C) 2009-2011 Mathias Doenitz
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.parboiled.scala.rules

import org.parboiled.matchers._
import Rule._

/**
 * The base class of all reduction rules, which take a certain number of input values and produce one output value.
 */
abstract class ReductionRule extends Rule

/**
 * A rule taking one value off the value stack and replacing it with another value.
 */
class ReductionRule1[-Z, +A](val matcher: Matcher) extends ReductionRule {
  def ~[X, Y, AA >: A](other: PopRule3[X, Y, AA]) = new PopRule3[X, Y, Z](append(other))
  def ~[Y, AA >: A](other: PopRule2[Y, AA]) = new PopRule2[Y, Z](append(other))
  def ~[AA >: A](other: PopRule1[AA]) = new PopRule1[Z](append(other))
  def ~[X, Y, AA >: A, RA](other: ReductionRule3[X, Y, AA, RA]) = new ReductionRule3[X, Y, Z, RA](append(other))
  def ~[X, Y, AA >: A, RA, B](other: ReductionRule3_2[X, Y, AA, RA, B]) = new ReductionRule3_2[X, Y, Z, RA, B](append(other))
  def ~[X, Y, AA >: A, RA, B, C](other: ReductionRule3_3[X, Y, AA, RA, B, C]) = new ReductionRule3_3[X, Y, Z, RA, B, C](append(other))
  def ~[X, Y, AA >: A, RA, B, C, D](other: ReductionRule3_4[X, Y, AA, RA, B, C, D]) = new ReductionRule3_4[X, Y, Z, RA, B, C, D](append(other))
  def ~[X, Y, AA >: A, RA, B, C, D, E](other: ReductionRule3_5[X, Y, AA, RA, B, C, D, E]) = new ReductionRule3_5[X, Y, Z, RA, B, C, D, E](append(other))
  def ~[X, Y, AA >: A, RA, B, C, D, E, F](other: ReductionRule3_6[X, Y, AA, RA, B, C, D, E, F]) = new ReductionRule3_6[X, Y, Z, RA, B, C, D, E, F](append(other))
  def ~[X, Y, AA >: A, RA, B, C, D, E, F, G](other: ReductionRule3_7[X, Y, AA, RA, B, C, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, RA, B, C, D, E, F, G](append(other))
  def ~[Y, AA >: A, RA](other: ReductionRule2[Y, AA, RA]) = new ReductionRule2[Y, Z, RA](append(other))
  def ~[Y, AA >: A, RA, B](other: ReductionRule2_2[Y, AA, RA, B]) = new ReductionRule2_2[Y, Z, RA, B](append(other))
  def ~[Y, AA >: A, RA, B, C](other: ReductionRule2_3[Y, AA, RA, B, C]) = new ReductionRule2_3[Y, Z, RA, B, C](append(other))
  def ~[Y, AA >: A, RA, B, C, D](other: ReductionRule2_4[Y, AA, RA, B, C, D]) = new ReductionRule2_4[Y, Z, RA, B, C, D](append(other))
  def ~[Y, AA >: A, RA, B, C, D, E](other: ReductionRule2_5[Y, AA, RA, B, C, D, E]) = new ReductionRule2_5[Y, Z, RA, B, C, D, E](append(other))
  def ~[Y, AA >: A, RA, B, C, D, E, F](other: ReductionRule2_6[Y, AA, RA, B, C, D, E, F]) = new ReductionRule2_6[Y, Z, RA, B, C, D, E, F](append(other))
  def ~[Y, AA >: A, RA, B, C, D, E, F, G](other: ReductionRule2_7[Y, AA, RA, B, C, D, E, F, G]) = new ReductionRule2_7[Y, Z, RA, B, C, D, E, F, G](append(other))
  def ~[AA >: A, RA](other: ReductionRule1[AA, RA]) = new ReductionRule1[Z, RA](append(other))
  def ~[AA >: A, RA, B](other: ReductionRule1_2[AA, RA, B]) = new ReductionRule1_2[Z, RA, B](append(other))
  def ~[AA >: A, RA, B, C](other: ReductionRule1_3[AA, RA, B, C]) = new ReductionRule1_3[Z, RA, B, C](append(other))
  def ~[AA >: A, RA, B, C, D](other: ReductionRule1_4[AA, RA, B, C, D]) = new ReductionRule1_4[Z, RA, B, C, D](append(other))
  def ~[AA >: A, RA, B, C, D, E](other: ReductionRule1_5[AA, RA, B, C, D, E]) = new ReductionRule1_5[Z, RA, B, C, D, E](append(other))
  def ~[AA >: A, RA, B, C, D, E, F](other: ReductionRule1_6[AA, RA, B, C, D, E, F]) = new ReductionRule1_6[Z, RA, B, C, D, E, F](append(other))
  def ~[AA >: A, RA, B, C, D, E, F, G](other: ReductionRule1_7[AA, RA, B, C, D, E, F, G]) = new ReductionRule1_7[Z, RA, B, C, D, E, F, G](append(other))
  def ~[B](other: Rule1[B]) = new ReductionRule1_2[Z, A, B](append(other))
  def ~[B, C](other: Rule2[B, C]) = new ReductionRule1_3[Z, A, B, C](append(other))
  def ~[B, C, D](other: Rule3[B, C, D]) = new ReductionRule1_4[Z, A, B, C, D](append(other))
  def ~[B, C, D, E](other: Rule4[B, C, D, E]) = new ReductionRule1_5[Z, A, B, C, D, E](append(other))
  def ~[B, C, D, E, F](other: Rule5[B, C, D, E, F]) = new ReductionRule1_6[Z, A, B, C, D, E, F](append(other))
  def ~[B, C, D, E, F, G](other: Rule6[B, C, D, E, F, G]) = new ReductionRule1_7[Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, Y, RA](f: (X, Y, A) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack3(Pop), f))))
  def ~~>[Y, RA](f: (Y, A) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack2(Pop), f))))
  def ~~>[RA](f: A => RA) = new ReductionRule1[Z, RA](append(push(exec(stack1(Pop), f))))
  def ~~?[X, Y](f: (X, Y, A) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack3(Pop), f)))
  def ~~?[Y](f: (Y, A) => Boolean) = new PopRule2[Y, Z](append(exec(stack2(Pop), f)))
  def ~~?(f: A => Boolean) = new PopRule1[Z](append(exec(stack1(Pop), f)))
  def ~~%[X, Y](f: (X, Y, A) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack3(Pop), f))))
  def ~~%[Y](f: (Y, A) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: A => Unit) = new PopRule1[Z](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: A => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: A => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[ZZ <: Z, AA >: A](other: ReductionRule1[ZZ, AA]) = new ReductionRule1[ZZ, AA](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule1[Z, A](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking one value off the value stack and replacing it with two values.
  */
class ReductionRule1_2[-Z, +A, +B](val matcher: Matcher) extends ReductionRule {
  def ~[Y, AA >: A, BB >: B](other: PopRule3[Y, AA, BB]) = new PopRule2[Y, Z](append(other))
  def ~[AA >: A, BB >: B](other: PopRule2[AA, BB]) = new PopRule1[Z](append(other))
  def ~[BB >: B](other: PopRule1[BB]) = new ReductionRule1[Z, A](append(other))
  def ~[Y, AA >: A, BB >: B, RA](other: ReductionRule3[Y, AA, BB, RA]) = new ReductionRule2[Y, Z, RA](append(other))
  def ~[Y, AA >: A, BB >: B, RA, RB](other: ReductionRule3_2[Y, AA, BB, RA, RB]) = new ReductionRule2_2[Y, Z, RA, RB](append(other))
  def ~[Y, AA >: A, BB >: B, RA, RB, C](other: ReductionRule3_3[Y, AA, BB, RA, RB, C]) = new ReductionRule2_3[Y, Z, RA, RB, C](append(other))
  def ~[Y, AA >: A, BB >: B, RA, RB, C, D](other: ReductionRule3_4[Y, AA, BB, RA, RB, C, D]) = new ReductionRule2_4[Y, Z, RA, RB, C, D](append(other))
  def ~[Y, AA >: A, BB >: B, RA, RB, C, D, E](other: ReductionRule3_5[Y, AA, BB, RA, RB, C, D, E]) = new ReductionRule2_5[Y, Z, RA, RB, C, D, E](append(other))
  def ~[Y, AA >: A, BB >: B, RA, RB, C, D, E, F](other: ReductionRule3_6[Y, AA, BB, RA, RB, C, D, E, F]) = new ReductionRule2_6[Y, Z, RA, RB, C, D, E, F](append(other))
  def ~[Y, AA >: A, BB >: B, RA, RB, C, D, E, F, G](other: ReductionRule3_7[Y, AA, BB, RA, RB, C, D, E, F, G]) = new ReductionRule2_7[Y, Z, RA, RB, C, D, E, F, G](append(other))
  def ~[AA >: A, BB >: B, RA](other: ReductionRule2[AA, BB, RA]) = new ReductionRule1[Z, RA](append(other))
  def ~[AA >: A, BB >: B, RA, RB](other: ReductionRule2_2[AA, BB, RA, RB]) = new ReductionRule1_2[Z, RA, RB](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C](other: ReductionRule2_3[AA, BB, RA, RB, C]) = new ReductionRule1_3[Z, RA, RB, C](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D](other: ReductionRule2_4[AA, BB, RA, RB, C, D]) = new ReductionRule1_4[Z, RA, RB, C, D](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D, E](other: ReductionRule2_5[AA, BB, RA, RB, C, D, E]) = new ReductionRule1_5[Z, RA, RB, C, D, E](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D, E, F](other: ReductionRule2_6[AA, BB, RA, RB, C, D, E, F]) = new ReductionRule1_6[Z, RA, RB, C, D, E, F](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D, E, F, G](other: ReductionRule2_7[AA, BB, RA, RB, C, D, E, F, G]) = new ReductionRule1_7[Z, RA, RB, C, D, E, F, G](append(other))
  def ~[BB >: B, RB](other: ReductionRule1[BB, RB]) = new ReductionRule1_2[Z, A, RB](append(other))
  def ~[BB >: B, RB, C](other: ReductionRule1_2[BB, RB, C]) = new ReductionRule1_3[Z, A, RB, C](append(other))
  def ~[BB >: B, RB, C, D](other: ReductionRule1_3[BB, RB, C, D]) = new ReductionRule1_4[Z, A, RB, C, D](append(other))
  def ~[BB >: B, RB, C, D, E](other: ReductionRule1_4[BB, RB, C, D, E]) = new ReductionRule1_5[Z, A, RB, C, D, E](append(other))
  def ~[BB >: B, RB, C, D, E, F](other: ReductionRule1_5[BB, RB, C, D, E, F]) = new ReductionRule1_6[Z, A, RB, C, D, E, F](append(other))
  def ~[BB >: B, RB, C, D, E, F, G](other: ReductionRule1_6[BB, RB, C, D, E, F, G]) = new ReductionRule1_7[Z, A, RB, C, D, E, F, G](append(other))
  def ~[C](other: Rule1[C]) = new ReductionRule1_3[Z, A, B, C](append(other))
  def ~[C, D](other: Rule2[C, D]) = new ReductionRule1_4[Z, A, B, C, D](append(other))
  def ~[C, D, E](other: Rule3[C, D, E]) = new ReductionRule1_5[Z, A, B, C, D, E](append(other))
  def ~[C, D, E, F](other: Rule4[C, D, E, F]) = new ReductionRule1_6[Z, A, B, C, D, E, F](append(other))
  def ~[C, D, E, F, G](other: Rule5[C, D, E, F, G]) = new ReductionRule1_7[Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, Y, RA](f: (X, Y, A, B) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack4(Pop), f))))
  def ~~>[Y, RA](f: (Y, A, B) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack3(Pop), f))))
  def ~~>[RA](f: (A, B) => RA) = new ReductionRule1[Z, RA](append(push(exec(stack2(Pop), f))))
  def ~~>[RB](f: B => RB) = new ReductionRule1_2[Z, A, RB](append(push(exec(stack1(Pop), f))))
  def ~~?[X, Y](f: (X, Y, A, B) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack4(Pop), f)))
  def ~~?[Y](f: (Y, A, B) => Boolean) = new PopRule2[Y, Z](append(exec(stack3(Pop), f)))
  def ~~?(f: (A, B) => Boolean) = new PopRule1[Z](append(exec(stack2(Pop), f)))
  def ~~?(f: B => Boolean) = new ReductionRule1[Z, A](append(exec(stack1(Pop), f)))
  def ~~%[X, Y](f: (X, Y, A, B) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack4(Pop), f))))
  def ~~%[Y](f: (Y, A, B) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack3(Pop), f))))
  def ~~%[Y](f: (A, B) => Unit) = new PopRule1[Z](append(ok(exec(stack2(Pop), f))))
  def ~~%[Y](f: B => Unit) = new ReductionRule1[Z, A](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: B => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: B => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[ZZ <: Z, AA >: A, BB >: B](other: ReductionRule1_2[ZZ, AA, BB]) = new ReductionRule1_2[ZZ, AA, BB](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule1_2[Z, A, B](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking one value off the value stack and replacing it with three values.
  */
class ReductionRule1_3[-Z, +A, +B, +C](val matcher: Matcher) extends ReductionRule {
  def ~[AA >: A, BB >: B, CC >: C](other: PopRule3[AA, BB, CC]) = new PopRule1[Z](append(other))
  def ~[BB >: B, CC >: C](other: PopRule2[BB, CC]) = new ReductionRule1[Z, A](append(other))
  def ~[CC >: C](other: PopRule1[CC]) = new ReductionRule1_2[Z, A, B](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA](other: ReductionRule3[AA, BB, CC, RA]) = new ReductionRule1[Z, RA](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB](other: ReductionRule3_2[AA, BB, CC, RA, RB]) = new ReductionRule1_2[Z, RA, RB](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC](other: ReductionRule3_3[AA, BB, CC, RA, RB, RC]) = new ReductionRule1_3[Z, RA, RB, RC](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D](other: ReductionRule3_4[AA, BB, CC, RA, RB, RC, D]) = new ReductionRule1_4[Z, RA, RB, RC, D](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D, E](other: ReductionRule3_5[AA, BB, CC, RA, RB, RC, D, E]) = new ReductionRule1_5[Z, RA, RB, RC, D, E](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D, E, F](other: ReductionRule3_6[AA, BB, CC, RA, RB, RC, D, E, F]) = new ReductionRule1_6[Z, RA, RB, RC, D, E, F](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D, E, F, G](other: ReductionRule3_7[AA, BB, CC, RA, RB, RC, D, E, F, G]) = new ReductionRule1_7[Z, RA, RB, RC, D, E, F, G](append(other))
  def ~[BB >: B, CC >: C, RB](other: ReductionRule2[BB, CC, RB]) = new ReductionRule1_2[Z, A, RB](append(other))
  def ~[BB >: B, CC >: C, RB, RC](other: ReductionRule2_2[BB, CC, RB, RC]) = new ReductionRule1_3[Z, A, RB, RC](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D](other: ReductionRule2_3[BB, CC, RB, RC, D]) = new ReductionRule1_4[Z, A, RB, RC, D](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D, E](other: ReductionRule2_4[BB, CC, RB, RC, D, E]) = new ReductionRule1_5[Z, A, RB, RC, D, E](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D, E, F](other: ReductionRule2_5[BB, CC, RB, RC, D, E, F]) = new ReductionRule1_6[Z, A, RB, RC, D, E, F](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D, E, F, G](other: ReductionRule2_6[BB, CC, RB, RC, D, E, F, G]) = new ReductionRule1_7[Z, A, RB, RC, D, E, F, G](append(other))
  def ~[CC >: C, RC](other: ReductionRule1[CC, RC]) = new ReductionRule1_3[Z, A, B, RC](append(other))
  def ~[CC >: C, RC, D](other: ReductionRule1_2[CC, RC, D]) = new ReductionRule1_4[Z, A, B, RC, D](append(other))
  def ~[CC >: C, RC, D, E](other: ReductionRule1_3[CC, RC, D, E]) = new ReductionRule1_5[Z, A, B, RC, D, E](append(other))
  def ~[CC >: C, RC, D, E, F](other: ReductionRule1_4[CC, RC, D, E, F]) = new ReductionRule1_6[Z, A, B, RC, D, E, F](append(other))
  def ~[CC >: C, RC, D, E, F, G](other: ReductionRule1_5[CC, RC, D, E, F, G]) = new ReductionRule1_7[Z, A, B, RC, D, E, F, G](append(other))
  def ~[D](other: Rule1[D]) = new ReductionRule1_4[Z, A, B, C, D](append(other))
  def ~[D, E](other: Rule2[D, E]) = new ReductionRule1_5[Z, A, B, C, D, E](append(other))
  def ~[D, E, F](other: Rule3[D, E, F]) = new ReductionRule1_6[Z, A, B, C, D, E, F](append(other))
  def ~[D, E, F, G](other: Rule4[D, E, F, G]) = new ReductionRule1_7[Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, Y, RA](f: (X, Y, A, B, C) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack5(Pop), f))))
  def ~~>[Y, RA](f: (Y, A, B, C) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack4(Pop), f))))
  def ~~>[RA](f: (A, B, C) => RA) = new ReductionRule1[Z, RA](append(push(exec(stack3(Pop), f))))
  def ~~>[RB](f: (B, C) => RB) = new ReductionRule1_2[Z, A, RB](append(push(exec(stack2(Pop), f))))
  def ~~>[RC](f: C => RC) = new ReductionRule1_3[Z, A, B, RC](append(push(exec(stack1(Pop), f))))
  def ~~?[X, Y](f: (X, Y, A, B, C) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack5(Pop), f)))
  def ~~?[Y](f: (Y, A, B, C) => Boolean) = new PopRule2[Y, Z](append(exec(stack4(Pop), f)))
  def ~~?(f: (A, B, C) => Boolean) = new PopRule1[Z](append(exec(stack3(Pop), f)))
  def ~~?(f: (B, C) => Boolean) = new ReductionRule1[Z, A](append(exec(stack2(Pop), f)))
  def ~~?(f: C => Boolean) = new ReductionRule1_2[Z, A, B](append(exec(stack1(Pop), f)))
  def ~~%[X, Y](f: (X, Y, A, B, C) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack5(Pop), f))))
  def ~~%[Y](f: (Y, A, B, C) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (A, B, C) => Unit) = new PopRule1[Z](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (B, C) => Unit) = new ReductionRule1[Z, A](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: C => Unit) = new ReductionRule1_2[Z, A, B](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (B, C) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: C => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (B, C) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: C => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[ZZ <: Z, AA >: A, BB >: B, CC >: C](other: ReductionRule1_3[ZZ, AA, BB, CC]) = new ReductionRule1_3[ZZ, AA, BB, CC](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule1_3[Z, A, B, C](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking one value off the value stack and replacing it with four values.
  */
class ReductionRule1_4[-Z, +A, +B, +C, +D](val matcher: Matcher) extends ReductionRule {
  def ~[BB >: B, CC >: C, DD >: D](other: PopRule3[BB, CC, DD]) = new ReductionRule1[Z, A](append(other))
  def ~[CC >: C, DD >: D](other: PopRule2[CC, DD]) = new ReductionRule1_2[Z, A, B](append(other))
  def ~[DD >: D](other: PopRule1[DD]) = new ReductionRule1_3[Z, A, B, C](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB](other: ReductionRule3[BB, CC, DD, RB]) = new ReductionRule1_2[Z, A, RB](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC](other: ReductionRule3_2[BB, CC, DD, RB, RC]) = new ReductionRule1_3[Z, A, RB, RC](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD](other: ReductionRule3_3[BB, CC, DD, RB, RC, RD]) = new ReductionRule1_4[Z, A, RB, RC, RD](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD, E](other: ReductionRule3_4[BB, CC, DD, RB, RC, RD, E]) = new ReductionRule1_5[Z, A, RB, RC, RD, E](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD, E, F](other: ReductionRule3_5[BB, CC, DD, RB, RC, RD, E, F]) = new ReductionRule1_6[Z, A, RB, RC, RD, E, F](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD, E, F, G](other: ReductionRule3_6[BB, CC, DD, RB, RC, RD, E, F, G]) = new ReductionRule1_7[Z, A, RB, RC, RD, E, F, G](append(other))
  def ~[CC >: C, DD >: D, RC](other: ReductionRule2[CC, DD, RC]) = new ReductionRule1_3[Z, A, B, RC](append(other))
  def ~[CC >: C, DD >: D, RC, RD](other: ReductionRule2_2[CC, DD, RC, RD]) = new ReductionRule1_4[Z, A, B, RC, RD](append(other))
  def ~[CC >: C, DD >: D, RC, RD, E](other: ReductionRule2_3[CC, DD, RC, RD, E]) = new ReductionRule1_5[Z, A, B, RC, RD, E](append(other))
  def ~[CC >: C, DD >: D, RC, RD, E, F](other: ReductionRule2_4[CC, DD, RC, RD, E, F]) = new ReductionRule1_6[Z, A, B, RC, RD, E, F](append(other))
  def ~[CC >: C, DD >: D, RC, RD, E, F, G](other: ReductionRule2_5[CC, DD, RC, RD, E, F, G]) = new ReductionRule1_7[Z, A, B, RC, RD, E, F, G](append(other))
  def ~[DD >: D, RD](other: ReductionRule1[DD, RD]) = new ReductionRule1_4[Z, A, B, C, RD](append(other))
  def ~[DD >: D, RD, E](other: ReductionRule1_2[DD, RD, E]) = new ReductionRule1_5[Z, A, B, C, RD, E](append(other))
  def ~[DD >: D, RD, E, F](other: ReductionRule1_3[DD, RD, E, F]) = new ReductionRule1_6[Z, A, B, C, RD, E, F](append(other))
  def ~[DD >: D, RD, E, F, G](other: ReductionRule1_4[DD, RD, E, F, G]) = new ReductionRule1_7[Z, A, B, C, RD, E, F, G](append(other))
  def ~[E](other: Rule1[E]) = new ReductionRule1_5[Z, A, B, C, D, E](append(other))
  def ~[E, F](other: Rule2[E, F]) = new ReductionRule1_6[Z, A, B, C, D, E, F](append(other))
  def ~[E, F, G](other: Rule3[E, F, G]) = new ReductionRule1_7[Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, Y, RA](f: (X, Y, A, B, C, D) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack6(Pop), f))))
  def ~~>[Y, RA](f: (Y, A, B, C, D) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack5(Pop), f))))
  def ~~>[RA](f: (A, B, C, D) => RA) = new ReductionRule1[Z, RA](append(push(exec(stack4(Pop), f))))
  def ~~>[RB](f: (B, C, D) => RB) = new ReductionRule1_2[Z, A, RB](append(push(exec(stack3(Pop), f))))
  def ~~>[RC](f: (C, D) => RC) = new ReductionRule1_3[Z, A, B, RC](append(push(exec(stack2(Pop), f))))
  def ~~>[RD](f: D => RD) = new ReductionRule1_4[Z, A, B, C, RD](append(push(exec(stack1(Pop), f))))
  def ~~?[X, Y](f: (X, Y, A, B, C, D) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack6(Pop), f)))
  def ~~?[Y](f: (Y, A, B, C, D) => Boolean) = new PopRule2[Y, Z](append(exec(stack5(Pop), f)))
  def ~~?(f: (A, B, C, D) => Boolean) = new PopRule1[Z](append(exec(stack4(Pop), f)))
  def ~~?(f: (B, C, D) => Boolean) = new ReductionRule1[Z, A](append(exec(stack3(Pop), f)))
  def ~~?(f: (C, D) => Boolean) = new ReductionRule1_2[Z, A, B](append(exec(stack2(Pop), f)))
  def ~~?(f: D => Boolean) = new ReductionRule1_3[Z, A, B, C](append(exec(stack1(Pop), f)))
  def ~~%[X, Y](f: (X, Y, A, B, C, D) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack6(Pop), f))))
  def ~~%[Y](f: (Y, A, B, C, D) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (A, B, C, D) => Unit) = new PopRule1[Z](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (B, C, D) => Unit) = new ReductionRule1[Z, A](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (C, D) => Unit) = new ReductionRule1_2[Z, A, B](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: D => Unit) = new ReductionRule1_3[Z, A, B, C](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (B, C, D) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (C, D) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: D => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (B, C, D) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (C, D) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: D => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D](other: ReductionRule1_4[ZZ, AA, BB, CC, DD]) = new ReductionRule1_4[ZZ, AA, BB, CC, DD](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule1_4[Z, A, B, C, D](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking one value off the value stack and replacing it with five values.
  */
class ReductionRule1_5[-Z, +A, +B, +C, +D, +E](val matcher: Matcher) extends ReductionRule {
  def ~[CC >: C, DD >: D, EE >: E](other: PopRule3[CC, DD, EE]) = new ReductionRule1_2[Z, A, B](append(other))
  def ~[DD >: D, EE >: E](other: PopRule2[DD, EE]) = new ReductionRule1_3[Z, A, B, C](append(other))
  def ~[EE >: E](other: PopRule1[EE]) = new ReductionRule1_4[Z, A, B, C, D](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC](other: ReductionRule3[CC, DD, EE, RC]) = new ReductionRule1_3[Z, A, B, RC](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD](other: ReductionRule3_2[CC, DD, EE, RC, RD]) = new ReductionRule1_4[Z, A, B, RC, RD](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD, RE](other: ReductionRule3_3[CC, DD, EE, RC, RD, RE]) = new ReductionRule1_5[Z, A, B, RC, RD, RE](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD, RE, RF](other: ReductionRule3_4[CC, DD, EE, RC, RD, RE, RF]) = new ReductionRule1_6[Z, A, B, RC, RD, RE, RF](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD, RE, RF, G](other: ReductionRule3_5[CC, DD, EE, RC, RD, RE, RF, G]) = new ReductionRule1_7[Z, A, B, RC, RD, RE, RF, G](append(other))
  def ~[DD >: D, EE >: E, RD](other: ReductionRule2[DD, EE, RD]) = new ReductionRule1_4[Z, A, B, C, RD](append(other))
  def ~[DD >: D, EE >: E, RD, RE](other: ReductionRule2_2[DD, EE, RD, RE]) = new ReductionRule1_5[Z, A, B, C, RD, RE](append(other))
  def ~[DD >: D, EE >: E, RD, RE, F](other: ReductionRule2_3[DD, EE, RD, RE, F]) = new ReductionRule1_6[Z, A, B, C, RD, RE, F](append(other))
  def ~[DD >: D, EE >: E, RD, RE, F, G](other: ReductionRule2_4[DD, EE, RD, RE, F, G]) = new ReductionRule1_7[Z, A, B, C, RD, RE, F, G](append(other))
  def ~[EE >: E, RE](other: ReductionRule1[EE, RE]) = new ReductionRule1_5[Z, A, B, C, D, RE](append(other))
  def ~[EE >: E, RE, F](other: ReductionRule1_2[EE, RE, F]) = new ReductionRule1_6[Z, A, B, C, D, RE, F](append(other))
  def ~[EE >: E, RE, F, G](other: ReductionRule1_3[EE, RE, F, G]) = new ReductionRule1_7[Z, A, B, C, D, RE, F, G](append(other))
  def ~[F](other: Rule1[F]) = new ReductionRule1_6[Z, A, B, C, D, E, F](append(other))
  def ~[F, G](other: Rule2[F, G]) = new ReductionRule1_7[Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, Y, RA](f: (X, Y, A, B, C, D, E) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack7(Pop), f))))
  def ~~>[Y, RA](f: (Y, A, B, C, D, E) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack6(Pop), f))))
  def ~~>[RA](f: (A, B, C, D, E) => RA) = new ReductionRule1[Z, RA](append(push(exec(stack5(Pop), f))))
  def ~~>[RB](f: (B, C, D, E) => RB) = new ReductionRule1_2[Z, A, RB](append(push(exec(stack4(Pop), f))))
  def ~~>[RC](f: (C, D, E) => RC) = new ReductionRule1_3[Z, A, B, RC](append(push(exec(stack3(Pop), f))))
  def ~~>[RD](f: (D, E) => RD) = new ReductionRule1_4[Z, A, B, C, RD](append(push(exec(stack2(Pop), f))))
  def ~~>[RE](f: E => RE) = new ReductionRule1_5[Z, A, B, C, D, RE](append(push(exec(stack1(Pop), f))))
  def ~~?[X, Y](f: (X, Y, A, B, C, D, E) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack7(Pop), f)))
  def ~~?[Y](f: (Y, A, B, C, D, E) => Boolean) = new PopRule2[Y, Z](append(exec(stack6(Pop), f)))
  def ~~?(f: (A, B, C, D, E) => Boolean) = new PopRule1[Z](append(exec(stack5(Pop), f)))
  def ~~?(f: (B, C, D, E) => Boolean) = new ReductionRule1[Z, A](append(exec(stack4(Pop), f)))
  def ~~?(f: (C, D, E) => Boolean) = new ReductionRule1_2[Z, A, B](append(exec(stack3(Pop), f)))
  def ~~?(f: (D, E) => Boolean) = new ReductionRule1_3[Z, A, B, C](append(exec(stack2(Pop), f)))
  def ~~?(f: E => Boolean) = new ReductionRule1_4[Z, A, B, C, D](append(exec(stack1(Pop), f)))
  def ~~%[X, Y](f: (X, Y, A, B, C, D, E) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack7(Pop), f))))
  def ~~%[Y](f: (Y, A, B, C, D, E) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack6(Pop), f))))
  def ~~%(f: (A, B, C, D, E) => Unit) = new PopRule1[Z](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (B, C, D, E) => Unit) = new ReductionRule1[Z, A](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (C, D, E) => Unit) = new ReductionRule1_2[Z, A, B](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (D, E) => Unit) = new ReductionRule1_3[Z, A, B, C](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: E => Unit) = new ReductionRule1_4[Z, A, B, C, D](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D, E) => Boolean) = withMatcher(append(exec(stack5(Peek), f)))
  def ~~~?(f: (B, C, D, E) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (C, D, E) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (D, E) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: E => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D, E) => Unit) = withMatcher(append(ok(exec(stack5(Peek), f))))
  def ~~~%(f: (B, C, D, E) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (C, D, E) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (D, E) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: E => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D, EE >: E](other: ReductionRule1_5[ZZ, AA, BB, CC, DD, EE]) = new ReductionRule1_5[ZZ, AA, BB, CC, DD, EE](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule1_5[Z, A, B, C, D, E](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking one value off the value stack and replacing it with six values.
  */
class ReductionRule1_6[-Z, +A, +B, +C, +D, +E, +F](val matcher: Matcher) extends ReductionRule {
  def ~[DD >: D, EE >: E, FF >: F](other: PopRule3[DD, EE, FF]) = new ReductionRule1_3[Z, A, B, C](append(other))
  def ~[EE >: E, FF >: F](other: PopRule2[EE, FF]) = new ReductionRule1_4[Z, A, B, C, D](append(other))
  def ~[FF >: F](other: PopRule1[FF]) = new ReductionRule1_5[Z, A, B, C, D, E](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD](other: ReductionRule3[DD, EE, FF, RD]) = new ReductionRule1_4[Z, A, B, C, RD](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD, RE](other: ReductionRule3_2[DD, EE, FF, RD, RE]) = new ReductionRule1_5[Z, A, B, C, RD, RE](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD, RE, RF](other: ReductionRule3_3[DD, EE, FF, RD, RE, RF]) = new ReductionRule1_6[Z, A, B, C, RD, RE, RF](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD, RE, RF, G](other: ReductionRule3_4[DD, EE, FF, RD, RE, RF, G]) = new ReductionRule1_7[Z, A, B, C, RD, RE, RF, G](append(other))
  def ~[EE >: E, FF >: F, RE](other: ReductionRule2[EE, FF, RE]) = new ReductionRule1_5[Z, A, B, C, D, RE](append(other))
  def ~[EE >: E, FF >: F, RE, RF](other: ReductionRule2_2[EE, FF, RE, RF]) = new ReductionRule1_6[Z, A, B, C, D, RE, RF](append(other))
  def ~[EE >: E, FF >: F, RE, RF, G](other: ReductionRule2_3[EE, FF, RE, RF, G]) = new ReductionRule1_7[Z, A, B, C, D, RE, RF, G](append(other))
  def ~[FF >: F, RF](other: ReductionRule1[FF, RF]) = new ReductionRule1_6[Z, A, B, C, D, E, RF](append(other))
  def ~[FF >: F, RF, G](other: ReductionRule1_2[FF, RF, G]) = new ReductionRule1_7[Z, A, B, C, D, E, RF, G](append(other))
  def ~[G](other: Rule1[G]) = new ReductionRule1_7[Z, A, B, C, D, E, F, G](append(other))
  def ~~>[Y, RA](f: (Y, A, B, C, D, E, F) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack7(Pop), f))))
  def ~~>[RA](f: (A, B, C, D, E, F) => RA) = new ReductionRule1[Z, RA](append(push(exec(stack6(Pop), f))))
  def ~~>[RB](f: (B, C, D, E, F) => RB) = new ReductionRule1_2[Z, A, RB](append(push(exec(stack5(Pop), f))))
  def ~~>[RC](f: (C, D, E, F) => RC) = new ReductionRule1_3[Z, A, B, RC](append(push(exec(stack4(Pop), f))))
  def ~~>[RD](f: (D, E, F) => RD) = new ReductionRule1_4[Z, A, B, C, RD](append(push(exec(stack3(Pop), f))))
  def ~~>[RE](f: (E, F) => RE) = new ReductionRule1_5[Z, A, B, C, D, RE](append(push(exec(stack2(Pop), f))))
  def ~~>[RF](f: F => RF) = new ReductionRule1_6[Z, A, B, C, D, E, RF](append(push(exec(stack1(Pop), f))))
  def ~~?[Y](f: (Y, A, B, C, D, E, F) => Boolean) = new PopRule2[Y, Z](append(exec(stack7(Pop), f)))
  def ~~?(f: (A, B, C, D, E, F) => Boolean) = new PopRule1[Z](append(exec(stack6(Pop), f)))
  def ~~?(f: (B, C, D, E, F) => Boolean) = new ReductionRule1[Z, A](append(exec(stack5(Pop), f)))
  def ~~?(f: (C, D, E, F) => Boolean) = new ReductionRule1_2[Z, A, B](append(exec(stack4(Pop), f)))
  def ~~?(f: (D, E, F) => Boolean) = new ReductionRule1_3[Z, A, B, C](append(exec(stack3(Pop), f)))
  def ~~?(f: (E, F) => Boolean) = new ReductionRule1_4[Z, A, B, C, D](append(exec(stack2(Pop), f)))
  def ~~?(f: F => Boolean) = new ReductionRule1_5[Z, A, B, C, D, E](append(exec(stack1(Pop), f)))
  def ~~%[Y](f: (Y, A, B, C, D, E, F) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack7(Pop), f))))
  def ~~%(f: (A, B, C, D, E, F) => Unit) = new PopRule1[Z](append(ok(exec(stack6(Pop), f))))
  def ~~%(f: (B, C, D, E, F) => Unit) = new ReductionRule1[Z, A](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (C, D, E, F) => Unit) = new ReductionRule1_2[Z, A, B](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (D, E, F) => Unit) = new ReductionRule1_3[Z, A, B, C](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (E, F) => Unit) = new ReductionRule1_4[Z, A, B, C, D](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: F => Unit) = new ReductionRule1_5[Z, A, B, C, D, E](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D, E, F) => Boolean) = withMatcher(append(exec(stack6(Peek), f)))
  def ~~~?(f: (B, C, D, E, F) => Boolean) = withMatcher(append(exec(stack5(Peek), f)))
  def ~~~?(f: (C, D, E, F) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (D, E, F) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (E, F) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: F => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D, E, F) => Unit) = withMatcher(append(ok(exec(stack6(Peek), f))))
  def ~~~%(f: (B, C, D, E, F) => Unit) = withMatcher(append(ok(exec(stack5(Peek), f))))
  def ~~~%(f: (C, D, E, F) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (D, E, F) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (E, F) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: F => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D, EE >: E, FF >: F](other: ReductionRule1_6[ZZ, AA, BB, CC, DD, EE, FF]) = new ReductionRule1_6[ZZ, AA, BB, CC, DD, EE, FF](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule1_6[Z, A, B, C, D, E, F](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking one value off the value stack and replacing it with seven values.
  */
class ReductionRule1_7[-Z, +A, +B, +C, +D, +E, +F, +G](val matcher: Matcher) extends ReductionRule {
  def ~[EE >: E, FF >: F, GG >: G](other: PopRule3[EE, FF, GG]) = new ReductionRule1_4[Z, A, B, C, D](append(other))
  def ~[FF >: F, GG >: G](other: PopRule2[FF, GG]) = new ReductionRule1_5[Z, A, B, C, D, E](append(other))
  def ~[GG >: G](other: PopRule1[GG]) = new ReductionRule1_6[Z, A, B, C, D, E, F](append(other))
  def ~[EE >: E, FF >: F, GG >: G, RE](other: ReductionRule3[EE, FF, GG, RE]) = new ReductionRule1_5[Z, A, B, C, D, RE](append(other))
  def ~[EE >: E, FF >: F, GG >: G, RE, RF](other: ReductionRule3_2[EE, FF, GG, RE, RF]) = new ReductionRule1_6[Z, A, B, C, D, RE, RF](append(other))
  def ~[EE >: E, FF >: F, GG >: G, RE, RF, RG](other: ReductionRule3_3[EE, FF, GG, RE, RF, RG]) = new ReductionRule1_7[Z, A, B, C, D, RE, RF, RG](append(other))
  def ~[FF >: F, GG >: G, RF](other: ReductionRule2[FF, GG, RF]) = new ReductionRule1_6[Z, A, B, C, D, E, RF](append(other))
  def ~[FF >: F, GG >: G, RF, RG](other: ReductionRule2_2[FF, GG, RF, RG]) = new ReductionRule1_7[Z, A, B, C, D, E, RF, RG](append(other))
  def ~[GG >: G, RG](other: ReductionRule1[GG, RG]) = new ReductionRule1_7[Z, A, B, C, D, E, F, RG](append(other))
  def ~~>[RA](f: (A, B, C, D, E, F, G) => RA) = new ReductionRule1[Z, RA](append(push(exec(stack7(Pop), f))))
  def ~~>[RB](f: (B, C, D, E, F, G) => RB) = new ReductionRule1_2[Z, A, RB](append(push(exec(stack6(Pop), f))))
  def ~~>[RC](f: (C, D, E, F, G) => RC) = new ReductionRule1_3[Z, A, B, RC](append(push(exec(stack5(Pop), f))))
  def ~~>[RD](f: (D, E, F, G) => RD) = new ReductionRule1_4[Z, A, B, C, RD](append(push(exec(stack4(Pop), f))))
  def ~~>[RE](f: (E, F, G) => RE) = new ReductionRule1_5[Z, A, B, C, D, RE](append(push(exec(stack3(Pop), f))))
  def ~~>[RF](f: (F, G) => RF) = new ReductionRule1_6[Z, A, B, C, D, E, RF](append(push(exec(stack2(Pop), f))))
  def ~~>[RG](f: G => RG) = new ReductionRule1_7[Z, A, B, C, D, E, F, RG](append(push(exec(stack1(Pop), f))))
  def ~~?(f: (A, B, C, D, E, F, G) => Boolean) = new PopRule1[Z](append(exec(stack7(Pop), f)))
  def ~~?(f: (B, C, D, E, F, G) => Boolean) = new ReductionRule1[Z, A](append(exec(stack6(Pop), f)))
  def ~~?(f: (C, D, E, F, G) => Boolean) = new ReductionRule1_2[Z, A, B](append(exec(stack5(Pop), f)))
  def ~~?(f: (D, E, F, G) => Boolean) = new ReductionRule1_3[Z, A, B, C](append(exec(stack4(Pop), f)))
  def ~~?(f: (E, F, G) => Boolean) = new ReductionRule1_4[Z, A, B, C, D](append(exec(stack3(Pop), f)))
  def ~~?(f: (F, G) => Boolean) = new ReductionRule1_5[Z, A, B, C, D, E](append(exec(stack2(Pop), f)))
  def ~~?(f: G => Boolean) = new ReductionRule1_6[Z, A, B, C, D, E, F](append(exec(stack1(Pop), f)))
  def ~~%(f: (A, B, C, D, E, F, G) => Unit) = new PopRule1[Z](append(ok(exec(stack7(Pop), f))))
  def ~~%(f: (B, C, D, E, F, G) => Unit) = new ReductionRule1[Z, A](append(ok(exec(stack6(Pop), f))))
  def ~~%(f: (C, D, E, F, G) => Unit) = new ReductionRule1_2[Z, A, B](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (D, E, F, G) => Unit) = new ReductionRule1_3[Z, A, B, C](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (E, F, G) => Unit) = new ReductionRule1_4[Z, A, B, C, D](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (F, G) => Unit) = new ReductionRule1_5[Z, A, B, C, D, E](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: G => Unit) = new ReductionRule1_6[Z, A, B, C, D, E, F](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D, E, F, G) => Boolean) = withMatcher(append(exec(stack7(Peek), f)))
  def ~~~?(f: (B, C, D, E, F, G) => Boolean) = withMatcher(append(exec(stack6(Peek), f)))
  def ~~~?(f: (C, D, E, F, G) => Boolean) = withMatcher(append(exec(stack5(Peek), f)))
  def ~~~?(f: (D, E, F, G) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (E, F, G) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (F, G) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: G => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack7(Peek), f))))
  def ~~~%(f: (B, C, D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack6(Peek), f))))
  def ~~~%(f: (C, D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack5(Peek), f))))
  def ~~~%(f: (D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (E, F, G) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (F, G) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: G => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D, EE >: E, FF >: F, GG >: G](other: ReductionRule1_7[ZZ, AA, BB, CC, DD, EE, FF, GG]) = new ReductionRule1_7[ZZ, AA, BB, CC, DD, EE, FF, GG](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule1_7[Z, A, B, C, D, E, F, G](matcher).asInstanceOf[this.type]
}

/**
 * A rule taking two values off the value stack and replacing them with one other value.
 */
class ReductionRule2[-Y, -Z, +A](val matcher: Matcher) extends ReductionRule {
  def ~[X, AA >: A](other: PopRule2[X, AA]) = new PopRule3[X, Y, Z](append(other))
  def ~[AA >: A](other: PopRule1[AA]) = new PopRule2[Y, Z](append(other))
  def ~[X, AA >: A, RA](other: ReductionRule2[X, AA, RA]) = new ReductionRule3[X, Y, Z, RA](append(other))
  def ~[X, AA >: A, RA, B](other: ReductionRule2_2[X, AA, RA, B]) = new ReductionRule3_2[X, Y, Z, RA, B](append(other))
  def ~[X, AA >: A, RA, B, C](other: ReductionRule2_3[X, AA, RA, B, C]) = new ReductionRule3_3[X, Y, Z, RA, B, C](append(other))
  def ~[X, AA >: A, RA, B, C, D](other: ReductionRule2_4[X, AA, RA, B, C, D]) = new ReductionRule3_4[X, Y, Z, RA, B, C, D](append(other))
  def ~[X, AA >: A, RA, B, C, D, E](other: ReductionRule2_5[X, AA, RA, B, C, D, E]) = new ReductionRule3_5[X, Y, Z, RA, B, C, D, E](append(other))
  def ~[X, AA >: A, RA, B, C, D, E, F](other: ReductionRule2_6[X, AA, RA, B, C, D, E, F]) = new ReductionRule3_6[X, Y, Z, RA, B, C, D, E, F](append(other))
  def ~[X, AA >: A, RA, B, C, D, E, F, G](other: ReductionRule2_7[X, AA, RA, B, C, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, RA, B, C, D, E, F, G](append(other))
  def ~[AA >: A, RA](other: ReductionRule1[AA, RA]) = new ReductionRule2[Y, Z, RA](append(other))
  def ~[AA >: A, RA, B](other: ReductionRule1_2[AA, RA, B]) = new ReductionRule2_2[Y, Z, RA, B](append(other))
  def ~[AA >: A, RA, B, C](other: ReductionRule1_3[AA, RA, B, C]) = new ReductionRule2_3[Y, Z, RA, B, C](append(other))
  def ~[AA >: A, RA, B, C, D](other: ReductionRule1_4[AA, RA, B, C, D]) = new ReductionRule2_4[Y, Z, RA, B, C, D](append(other))
  def ~[AA >: A, RA, B, C, D, E](other: ReductionRule1_5[AA, RA, B, C, D, E]) = new ReductionRule2_5[Y, Z, RA, B, C, D, E](append(other))
  def ~[AA >: A, RA, B, C, D, E, F](other: ReductionRule1_6[AA, RA, B, C, D, E, F]) = new ReductionRule2_6[Y, Z, RA, B, C, D, E, F](append(other))
  def ~[AA >: A, RA, B, C, D, E, F, G](other: ReductionRule1_7[AA, RA, B, C, D, E, F, G]) = new ReductionRule2_7[Y, Z, RA, B, C, D, E, F, G](append(other))
  def ~[B](other: Rule1[B]) = new ReductionRule2_2[Y, Z, A, B](append(other))
  def ~[B, C](other: Rule2[B, C]) = new ReductionRule2_3[Y, Z, A, B, C](append(other))
  def ~[B, C, D](other: Rule3[B, C, D]) = new ReductionRule2_4[Y, Z, A, B, C, D](append(other))
  def ~[B, C, D, E](other: Rule4[B, C, D, E]) = new ReductionRule2_5[Y, Z, A, B, C, D, E](append(other))
  def ~[B, C, D, E, F](other: Rule5[B, C, D, E, F]) = new ReductionRule2_6[Y, Z, A, B, C, D, E, F](append(other))
  def ~[B, C, D, E, F, G](other: Rule6[B, C, D, E, F, G]) = new ReductionRule2_7[Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, RA](f: (X, A) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack2(Pop), f))))
  def ~~>[RA](f: A => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack1(Pop), f))))
  def ~~?[X](f: (X, A) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack2(Pop), f)))
  def ~~?(f: A => Boolean) = new PopRule2[Y, Z](append(exec(stack1(Pop), f)))
  def ~~%[X](f: (X, A) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: A => Unit) = new PopRule2[Y, Z](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: A => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: A => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[YY <: Y, ZZ <: Z, AA >: A](other: ReductionRule2[YY, ZZ, AA]) = new ReductionRule2[YY, ZZ, AA](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule2[Y, Z, A](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking two values off the value stack and replacing them with two values.
  */
class ReductionRule2_2[-Y, -Z, +A, +B](val matcher: Matcher) extends ReductionRule {
  def ~[X, AA >: A, BB >: B](other: PopRule3[X, AA, BB]) = new PopRule3[X, Y, Z](append(other))
  def ~[AA >: A, BB >: B](other: PopRule2[AA, BB]) = new PopRule2[Y, Z](append(other))
  def ~[BB >: B](other: PopRule1[BB]) = new ReductionRule2[Y, Z, A](append(other))
  def ~[X, AA >: A, BB >: B, RA](other: ReductionRule3[X, AA, BB, RA]) = new ReductionRule3[X, Y, Z, RA](append(other))
  def ~[X, AA >: A, BB >: B, RA, RB](other: ReductionRule3_2[X, AA, BB, RA, RB]) = new ReductionRule3_2[X, Y, Z, RA, RB](append(other))
  def ~[X, AA >: A, BB >: B, RA, RB, C](other: ReductionRule3_3[X, AA, BB, RA, RB, C]) = new ReductionRule3_3[X, Y, Z, RA, RB, C](append(other))
  def ~[X, AA >: A, BB >: B, RA, RB, C, D](other: ReductionRule3_4[X, AA, BB, RA, RB, C, D]) = new ReductionRule3_4[X, Y, Z, RA, RB, C, D](append(other))
  def ~[X, AA >: A, BB >: B, RA, RB, C, D, E](other: ReductionRule3_5[X, AA, BB, RA, RB, C, D, E]) = new ReductionRule3_5[X, Y, Z, RA, RB, C, D, E](append(other))
  def ~[X, AA >: A, BB >: B, RA, RB, C, D, E, F](other: ReductionRule3_6[X, AA, BB, RA, RB, C, D, E, F]) = new ReductionRule3_6[X, Y, Z, RA, RB, C, D, E, F](append(other))
  def ~[X, AA >: A, BB >: B, RA, RB, C, D, E, F, G](other: ReductionRule3_7[X, AA, BB, RA, RB, C, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, RA, RB, C, D, E, F, G](append(other))
  def ~[AA >: A, BB >: B, RA](other: ReductionRule2[AA, BB, RA]) = new ReductionRule2[Y, Z, RA](append(other))
  def ~[AA >: A, BB >: B, RA, RB](other: ReductionRule2_2[AA, BB, RA, RB]) = new ReductionRule2_2[Y, Z, RA, RB](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C](other: ReductionRule2_3[AA, BB, RA, RB, C]) = new ReductionRule2_3[Y, Z, RA, RB, C](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D](other: ReductionRule2_4[AA, BB, RA, RB, C, D]) = new ReductionRule2_4[Y, Z, RA, RB, C, D](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D, E](other: ReductionRule2_5[AA, BB, RA, RB, C, D, E]) = new ReductionRule2_5[Y, Z, RA, RB, C, D, E](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D, E, F](other: ReductionRule2_6[AA, BB, RA, RB, C, D, E, F]) = new ReductionRule2_6[Y, Z, RA, RB, C, D, E, F](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D, E, F, G](other: ReductionRule2_7[AA, BB, RA, RB, C, D, E, F, G]) = new ReductionRule2_7[Y, Z, RA, RB, C, D, E, F, G](append(other))
  def ~[BB >: B, RB](other: ReductionRule1[BB, RB]) = new ReductionRule2_2[Y, Z, A, RB](append(other))
  def ~[BB >: B, RB, C](other: ReductionRule1_2[BB, RB, C]) = new ReductionRule2_3[Y, Z, A, RB, C](append(other))
  def ~[BB >: B, RB, C, D](other: ReductionRule1_3[BB, RB, C, D]) = new ReductionRule2_4[Y, Z, A, RB, C, D](append(other))
  def ~[BB >: B, RB, C, D, E](other: ReductionRule1_4[BB, RB, C, D, E]) = new ReductionRule2_5[Y, Z, A, RB, C, D, E](append(other))
  def ~[BB >: B, RB, C, D, E, F](other: ReductionRule1_5[BB, RB, C, D, E, F]) = new ReductionRule2_6[Y, Z, A, RB, C, D, E, F](append(other))
  def ~[BB >: B, RB, C, D, E, F, G](other: ReductionRule1_6[BB, RB, C, D, E, F, G]) = new ReductionRule2_7[Y, Z, A, RB, C, D, E, F, G](append(other))
  def ~[C](other: Rule1[C]) = new ReductionRule2_3[Y, Z, A, B, C](append(other))
  def ~[C, D](other: Rule2[C, D]) = new ReductionRule2_4[Y, Z, A, B, C, D](append(other))
  def ~[C, D, E](other: Rule3[C, D, E]) = new ReductionRule2_5[Y, Z, A, B, C, D, E](append(other))
  def ~[C, D, E, F](other: Rule4[C, D, E, F]) = new ReductionRule2_6[Y, Z, A, B, C, D, E, F](append(other))
  def ~[C, D, E, F, G](other: Rule5[C, D, E, F, G]) = new ReductionRule2_7[Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, RA](f: (X, A, B) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack3(Pop), f))))
  def ~~>[RA](f: (A, B) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack2(Pop), f))))
  def ~~>[RB](f: B => RB) = new ReductionRule2_2[Y, Z, A, RB](append(push(exec(stack1(Pop), f))))
  def ~~?[X](f: (X, A, B) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack3(Pop), f)))
  def ~~?(f: (A, B) => Boolean) = new PopRule2[Y, Z](append(exec(stack2(Pop), f)))
  def ~~?(f: B => Boolean) = new ReductionRule2[Y, Z, A](append(exec(stack1(Pop), f)))
  def ~~%[X](f: (X, A, B) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (A, B) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: B => Unit) = new ReductionRule2[Y, Z, A](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: B => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: B => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[YY <: Y, ZZ <: Z, AA >: A, BB >: B](other: ReductionRule2_2[YY, ZZ, AA, BB]) = new ReductionRule2_2[YY, ZZ, AA, BB](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule2_2[Y, Z, A, B](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking two values off the value stack and replacing them with three values.
  */
class ReductionRule2_3[-Y, -Z, +A, +B, +C](val matcher: Matcher) extends ReductionRule {
  def ~[AA >: A, BB >: B, CC >: C](other: PopRule3[AA, BB, CC]) = new PopRule2[Y, Z](append(other))
  def ~[BB >: B, CC >: C](other: PopRule2[BB, CC]) = new ReductionRule2[Y, Z, A](append(other))
  def ~[CC >: C](other: PopRule1[CC]) = new ReductionRule2_2[Y, Z, A, B](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA](other: ReductionRule3[AA, BB, CC, RA]) = new ReductionRule2[Y, Z, RA](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB](other: ReductionRule3_2[AA, BB, CC, RA, RB]) = new ReductionRule2_2[Y, Z, RA, RB](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC](other: ReductionRule3_3[AA, BB, CC, RA, RB, RC]) = new ReductionRule2_3[Y, Z, RA, RB, RC](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D](other: ReductionRule3_4[AA, BB, CC, RA, RB, RC, D]) = new ReductionRule2_4[Y, Z, RA, RB, RC, D](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D, E](other: ReductionRule3_5[AA, BB, CC, RA, RB, RC, D, E]) = new ReductionRule2_5[Y, Z, RA, RB, RC, D, E](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D, E, F](other: ReductionRule3_6[AA, BB, CC, RA, RB, RC, D, E, F]) = new ReductionRule2_6[Y, Z, RA, RB, RC, D, E, F](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D, E, F, G](other: ReductionRule3_7[AA, BB, CC, RA, RB, RC, D, E, F, G]) = new ReductionRule2_7[Y, Z, RA, RB, RC, D, E, F, G](append(other))
  def ~[BB >: B, CC >: C, RB](other: ReductionRule2[BB, CC, RB]) = new ReductionRule2_2[Y, Z, A, RB](append(other))
  def ~[BB >: B, CC >: C, RB, RC](other: ReductionRule2_2[BB, CC, RB, RC]) = new ReductionRule2_3[Y, Z, A, RB, RC](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D](other: ReductionRule2_3[BB, CC, RB, RC, D]) = new ReductionRule2_4[Y, Z, A, RB, RC, D](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D, E](other: ReductionRule2_4[BB, CC, RB, RC, D, E]) = new ReductionRule2_5[Y, Z, A, RB, RC, D, E](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D, E, F](other: ReductionRule2_5[BB, CC, RB, RC, D, E, F]) = new ReductionRule2_6[Y, Z, A, RB, RC, D, E, F](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D, E, F, G](other: ReductionRule2_6[BB, CC, RB, RC, D, E, F, G]) = new ReductionRule2_7[Y, Z, A, RB, RC, D, E, F, G](append(other))
  def ~[CC >: C, RC](other: ReductionRule1[CC, RC]) = new ReductionRule2_3[Y, Z, A, B, RC](append(other))
  def ~[CC >: C, RC, D](other: ReductionRule1_2[CC, RC, D]) = new ReductionRule2_4[Y, Z, A, B, RC, D](append(other))
  def ~[CC >: C, RC, D, E](other: ReductionRule1_3[CC, RC, D, E]) = new ReductionRule2_5[Y, Z, A, B, RC, D, E](append(other))
  def ~[CC >: C, RC, D, E, F](other: ReductionRule1_4[CC, RC, D, E, F]) = new ReductionRule2_6[Y, Z, A, B, RC, D, E, F](append(other))
  def ~[CC >: C, RC, D, E, F, G](other: ReductionRule1_5[CC, RC, D, E, F, G]) = new ReductionRule2_7[Y, Z, A, B, RC, D, E, F, G](append(other))
  def ~[D](other: Rule1[D]) = new ReductionRule2_4[Y, Z, A, B, C, D](append(other))
  def ~[D, E](other: Rule2[D, E]) = new ReductionRule2_5[Y, Z, A, B, C, D, E](append(other))
  def ~[D, E, F](other: Rule3[D, E, F]) = new ReductionRule2_6[Y, Z, A, B, C, D, E, F](append(other))
  def ~[D, E, F, G](other: Rule4[D, E, F, G]) = new ReductionRule2_7[Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, RA](f: (X, A, B, C) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack4(Pop), f))))
  def ~~>[RA](f: (A, B, C) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack3(Pop), f))))
  def ~~>[RB](f: (B, C) => RB) = new ReductionRule2_2[Y, Z, A, RB](append(push(exec(stack2(Pop), f))))
  def ~~>[RC](f: C => RC) = new ReductionRule2_3[Y, Z, A, B, RC](append(push(exec(stack1(Pop), f))))
  def ~~?[X](f: (X, A, B, C) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack4(Pop), f)))
  def ~~?(f: (A, B, C) => Boolean) = new PopRule2[Y, Z](append(exec(stack3(Pop), f)))
  def ~~?(f: (B, C) => Boolean) = new ReductionRule2[Y, Z, A](append(exec(stack2(Pop), f)))
  def ~~?(f: C => Boolean) = new ReductionRule2_2[Y, Z, A, B](append(exec(stack1(Pop), f)))
  def ~~%[X](f: (X, A, B, C) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (A, B, C) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (B, C) => Unit) = new ReductionRule2[Y, Z, A](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: C => Unit) = new ReductionRule2_2[Y, Z, A, B](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (B, C) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: C => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (B, C) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: C => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[YY <: Y, ZZ <: Z, AA >: A, BB >: B, CC >: C](other: ReductionRule2_3[YY, ZZ, AA, BB, CC]) = new ReductionRule2_3[YY, ZZ, AA, BB, CC](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule2_3[Y, Z, A, B, C](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking two values off the value stack and replacing them with four values.
  */
class ReductionRule2_4[-Y, -Z, +A, +B, +C, +D](val matcher: Matcher) extends ReductionRule {
  def ~[BB >: B, CC >: C, DD >: D](other: PopRule3[BB, CC, DD]) = new ReductionRule2[Y, Z, A](append(other))
  def ~[CC >: C, DD >: D](other: PopRule2[CC, DD]) = new ReductionRule2_2[Y, Z, A, B](append(other))
  def ~[DD >: D](other: PopRule1[DD]) = new ReductionRule2_3[Y, Z, A, B, C](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB](other: ReductionRule3[BB, CC, DD, RB]) = new ReductionRule2_2[Y, Z, A, RB](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC](other: ReductionRule3_2[BB, CC, DD, RB, RC]) = new ReductionRule2_3[Y, Z, A, RB, RC](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD](other: ReductionRule3_3[BB, CC, DD, RB, RC, RD]) = new ReductionRule2_4[Y, Z, A, RB, RC, RD](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD, E](other: ReductionRule3_4[BB, CC, DD, RB, RC, RD, E]) = new ReductionRule2_5[Y, Z, A, RB, RC, RD, E](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD, E, F](other: ReductionRule3_5[BB, CC, DD, RB, RC, RD, E, F]) = new ReductionRule2_6[Y, Z, A, RB, RC, RD, E, F](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD, E, F, G](other: ReductionRule3_6[BB, CC, DD, RB, RC, RD, E, F, G]) = new ReductionRule2_7[Y, Z, A, RB, RC, RD, E, F, G](append(other))
  def ~[CC >: C, DD >: D, RC](other: ReductionRule2[CC, DD, RC]) = new ReductionRule2_3[Y, Z, A, B, RC](append(other))
  def ~[CC >: C, DD >: D, RC, RD](other: ReductionRule2_2[CC, DD, RC, RD]) = new ReductionRule2_4[Y, Z, A, B, RC, RD](append(other))
  def ~[CC >: C, DD >: D, RC, RD, E](other: ReductionRule2_3[CC, DD, RC, RD, E]) = new ReductionRule2_5[Y, Z, A, B, RC, RD, E](append(other))
  def ~[CC >: C, DD >: D, RC, RD, E, F](other: ReductionRule2_4[CC, DD, RC, RD, E, F]) = new ReductionRule2_6[Y, Z, A, B, RC, RD, E, F](append(other))
  def ~[CC >: C, DD >: D, RC, RD, E, F, G](other: ReductionRule2_5[CC, DD, RC, RD, E, F, G]) = new ReductionRule2_7[Y, Z, A, B, RC, RD, E, F, G](append(other))
  def ~[DD >: D, RD](other: ReductionRule1[DD, RD]) = new ReductionRule2_4[Y, Z, A, B, C, RD](append(other))
  def ~[DD >: D, RD, E](other: ReductionRule1_2[DD, RD, E]) = new ReductionRule2_5[Y, Z, A, B, C, RD, E](append(other))
  def ~[DD >: D, RD, E, F](other: ReductionRule1_3[DD, RD, E, F]) = new ReductionRule2_6[Y, Z, A, B, C, RD, E, F](append(other))
  def ~[DD >: D, RD, E, F, G](other: ReductionRule1_4[DD, RD, E, F, G]) = new ReductionRule2_7[Y, Z, A, B, C, RD, E, F, G](append(other))
  def ~[E](other: Rule1[E]) = new ReductionRule2_5[Y, Z, A, B, C, D, E](append(other))
  def ~[E, F](other: Rule2[E, F]) = new ReductionRule2_6[Y, Z, A, B, C, D, E, F](append(other))
  def ~[E, F, G](other: Rule3[E, F, G]) = new ReductionRule2_7[Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, RA](f: (X, A, B, C, D) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack5(Pop), f))))
  def ~~>[RA](f: (A, B, C, D) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack4(Pop), f))))
  def ~~>[RB](f: (B, C, D) => RB) = new ReductionRule2_2[Y, Z, A, RB](append(push(exec(stack3(Pop), f))))
  def ~~>[RC](f: (C, D) => RC) = new ReductionRule2_3[Y, Z, A, B, RC](append(push(exec(stack2(Pop), f))))
  def ~~>[RD](f: D => RD) = new ReductionRule2_4[Y, Z, A, B, C, RD](append(push(exec(stack1(Pop), f))))
  def ~~?[X](f: (X, A, B, C, D) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack5(Pop), f)))
  def ~~?(f: (A, B, C, D) => Boolean) = new PopRule2[Y, Z](append(exec(stack4(Pop), f)))
  def ~~?(f: (B, C, D) => Boolean) = new ReductionRule2[Y, Z, A](append(exec(stack3(Pop), f)))
  def ~~?(f: (C, D) => Boolean) = new ReductionRule2_2[Y, Z, A, B](append(exec(stack2(Pop), f)))
  def ~~?(f: D => Boolean) = new ReductionRule2_3[Y, Z, A, B, C](append(exec(stack1(Pop), f)))
  def ~~%[X](f: (X, A, B, C, D) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (A, B, C, D) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (B, C, D) => Unit) = new ReductionRule2[Y, Z, A](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (C, D) => Unit) = new ReductionRule2_2[Y, Z, A, B](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: D => Unit) = new ReductionRule2_3[Y, Z, A, B, C](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (B, C, D) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (C, D) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: D => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (B, C, D) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (C, D) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: D => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[YY <: Y, ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D](other: ReductionRule2_4[YY, ZZ, AA, BB, CC, DD]) = new ReductionRule2_4[YY, ZZ, AA, BB, CC, DD](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule2_4[Y, Z, A, B, C, D](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking two values off the value stack and replacing them with five values.
  */
class ReductionRule2_5[-Y, -Z, +A, +B, +C, +D, +E](val matcher: Matcher) extends ReductionRule {
  def ~[CC >: C, DD >: D, EE >: E](other: PopRule3[CC, DD, EE]) = new ReductionRule2_2[Y, Z, A, B](append(other))
  def ~[DD >: D, EE >: E](other: PopRule2[DD, EE]) = new ReductionRule2_3[Y, Z, A, B, C](append(other))
  def ~[EE >: E](other: PopRule1[EE]) = new ReductionRule2_4[Y, Z, A, B, C, D](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC](other: ReductionRule3[CC, DD, EE, RC]) = new ReductionRule2_3[Y, Z, A, B, RC](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD](other: ReductionRule3_2[CC, DD, EE, RC, RD]) = new ReductionRule2_4[Y, Z, A, B, RC, RD](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD, RE](other: ReductionRule3_3[CC, DD, EE, RC, RD, RE]) = new ReductionRule2_5[Y, Z, A, B, RC, RD, RE](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD, RE, F](other: ReductionRule3_4[CC, DD, EE, RC, RD, RE, F]) = new ReductionRule2_6[Y, Z, A, B, RC, RD, RE, F](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD, RE, F, G](other: ReductionRule3_5[CC, DD, EE, RC, RD, RE, F, G]) = new ReductionRule2_7[Y, Z, A, B, RC, RD, RE, F, G](append(other))
  def ~[DD >: D, EE >: E, RD](other: ReductionRule2[DD, EE, RD]) = new ReductionRule2_4[Y, Z, A, B, C, RD](append(other))
  def ~[DD >: D, EE >: E, RD, RE](other: ReductionRule2_2[DD, EE, RD, RE]) = new ReductionRule2_5[Y, Z, A, B, C, RD, RE](append(other))
  def ~[DD >: D, EE >: E, RD, RE, F](other: ReductionRule2_3[DD, EE, RD, RE, F]) = new ReductionRule2_6[Y, Z, A, B, C, RD, RE, F](append(other))
  def ~[DD >: D, EE >: E, RD, RE, F, G](other: ReductionRule2_4[DD, EE, RD, RE, F, G]) = new ReductionRule2_7[Y, Z, A, B, C, RD, RE, F, G](append(other))
  def ~[EE >: E, RE](other: ReductionRule1[EE, RE]) = new ReductionRule2_5[Y, Z, A, B, C, D, RE](append(other))
  def ~[EE >: E, RE, F](other: ReductionRule1_2[EE, RE, F]) = new ReductionRule2_6[Y, Z, A, B, C, D, RE, F](append(other))
  def ~[EE >: E, RE, F, G](other: ReductionRule1_3[EE, RE, F, G]) = new ReductionRule2_7[Y, Z, A, B, C, D, RE, F, G](append(other))
  def ~[F](other: Rule1[F]) = new ReductionRule2_6[Y, Z, A, B, C, D, E, F](append(other))
  def ~[F, G](other: Rule2[F, G]) = new ReductionRule2_7[Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, RA](f: (X, A, B, C, D, E) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack6(Pop), f))))
  def ~~>[RA](f: (A, B, C, D, E) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack5(Pop), f))))
  def ~~>[RB](f: (B, C, D, E) => RB) = new ReductionRule2_2[Y, Z, A, RB](append(push(exec(stack4(Pop), f))))
  def ~~>[RC](f: (C, D, E) => RC) = new ReductionRule2_3[Y, Z, A, B, RC](append(push(exec(stack3(Pop), f))))
  def ~~>[RD](f: (D, E) => RD) = new ReductionRule2_4[Y, Z, A, B, C, RD](append(push(exec(stack2(Pop), f))))
  def ~~>[RE](f: E => RE) = new ReductionRule2_5[Y, Z, A, B, C, D, RE](append(push(exec(stack1(Pop), f))))
  def ~~?[X](f: (X, A, B, C, D, E) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack6(Pop), f)))
  def ~~?(f: (A, B, C, D, E) => Boolean) = new PopRule2[Y, Z](append(exec(stack5(Pop), f)))
  def ~~?(f: (B, C, D, E) => Boolean) = new ReductionRule2[Y, Z, A](append(exec(stack4(Pop), f)))
  def ~~?(f: (C, D, E) => Boolean) = new ReductionRule2_2[Y, Z, A, B](append(exec(stack3(Pop), f)))
  def ~~?(f: (D, E) => Boolean) = new ReductionRule2_3[Y, Z, A, B, C](append(exec(stack2(Pop), f)))
  def ~~?(f: E => Boolean) = new ReductionRule2_4[Y, Z, A, B, C, D](append(exec(stack1(Pop), f)))
  def ~~%[X](f: (X, A, B, C, D, E) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack6(Pop), f))))
  def ~~%(f: (A, B, C, D, E) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (B, C, D, E) => Unit) = new ReductionRule2[Y, Z, A](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (C, D, E) => Unit) = new ReductionRule2_2[Y, Z, A, B](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (D, E) => Unit) = new ReductionRule2_3[Y, Z, A, B, C](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: E => Unit) = new ReductionRule2_4[Y, Z, A, B, C, D](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D, E) => Boolean) = withMatcher(append(exec(stack5(Peek), f)))
  def ~~~?(f: (B, C, D, E) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (C, D, E) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (D, E) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: E => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D, E) => Unit) = withMatcher(append(ok(exec(stack5(Peek), f))))
  def ~~~%(f: (B, C, D, E) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (C, D, E) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (D, E) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: E => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[YY <: Y, ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D, EE >: E](other: ReductionRule2_5[YY, ZZ, AA, BB, CC, DD, EE]) = new ReductionRule2_5[YY, ZZ, AA, BB, CC, DD, EE](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule2_5[Y, Z, A, B, C, D, E](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking two values off the value stack and replacing them with six values.
  */
class ReductionRule2_6[-Y, -Z, +A, +B, +C, +D, +E, +F](val matcher: Matcher) extends ReductionRule {
  def ~[DD >: D, EE >: E, FF >: F](other: PopRule3[DD, EE, FF]) = new ReductionRule2_3[Y, Z, A, B, C](append(other))
  def ~[EE >: E, FF >: F](other: PopRule2[EE, FF]) = new ReductionRule2_4[Y, Z, A, B, C, D](append(other))
  def ~[FF >: F](other: PopRule1[FF]) = new ReductionRule2_5[Y, Z, A, B, C, D, E](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD](other: ReductionRule3[DD, EE, FF, RD]) = new ReductionRule2_4[Y, Z, A, B, C, RD](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD, RE](other: ReductionRule3_2[DD, EE, FF, RD, RE]) = new ReductionRule2_5[Y, Z, A, B, C, RD, RE](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD, RE, RF](other: ReductionRule3_3[DD, EE, FF, RD, RE, RF]) = new ReductionRule2_6[Y, Z, A, B, C, RD, RE, RF](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD, RE, RF, G](other: ReductionRule3_4[DD, EE, FF, RD, RE, RF, G]) = new ReductionRule2_7[Y, Z, A, B, C, RD, RE, RF, G](append(other))
  def ~[EE >: E, FF >: F, RE](other: ReductionRule2[EE, FF, RE]) = new ReductionRule2_5[Y, Z, A, B, C, D, RE](append(other))
  def ~[EE >: E, FF >: F, RE, RF](other: ReductionRule2_2[EE, FF, RE, RF]) = new ReductionRule2_6[Y, Z, A, B, C, D, RE, RF](append(other))
  def ~[EE >: E, FF >: F, RE, RF, G](other: ReductionRule2_3[EE, FF, RE, RF, G]) = new ReductionRule2_7[Y, Z, A, B, C, D, RE, RF, G](append(other))
  def ~[FF >: F, RF](other: ReductionRule1[FF, RF]) = new ReductionRule2_6[Y, Z, A, B, C, D, E, RF](append(other))
  def ~[FF >: F, RF, G](other: ReductionRule1_2[FF, RF, G]) = new ReductionRule2_7[Y, Z, A, B, C, D, E, RF, G](append(other))
  def ~[G](other: Rule1[G]) = new ReductionRule2_7[Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[X, RA](f: (X, A, B, C, D, E, F) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack7(Pop), f))))
  def ~~>[RA](f: (A, B, C, D, E, F) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack6(Pop), f))))
  def ~~>[RB](f: (B, C, D, E, F) => RB) = new ReductionRule2_2[Y, Z, A, RB](append(push(exec(stack5(Pop), f))))
  def ~~>[RC](f: (C, D, E, F) => RC) = new ReductionRule2_3[Y, Z, A, B, RC](append(push(exec(stack4(Pop), f))))
  def ~~>[RD](f: (D, E, F) => RD) = new ReductionRule2_4[Y, Z, A, B, C, RD](append(push(exec(stack3(Pop), f))))
  def ~~>[RE](f: (E, F) => RE) = new ReductionRule2_5[Y, Z, A, B, C, D, RE](append(push(exec(stack2(Pop), f))))
  def ~~>[RF](f: F => RF) = new ReductionRule2_6[Y, Z, A, B, C, D, E, RF](append(push(exec(stack1(Pop), f))))
  def ~~?[X](f: (X, A, B, C, D, E, F) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack7(Pop), f)))
  def ~~?(f: (A, B, C, D, E, F) => Boolean) = new PopRule2[Y, Z](append(exec(stack6(Pop), f)))
  def ~~?(f: (B, C, D, E, F) => Boolean) = new ReductionRule2[Y, Z, A](append(exec(stack5(Pop), f)))
  def ~~?(f: (C, D, E, F) => Boolean) = new ReductionRule2_2[Y, Z, A, B](append(exec(stack4(Pop), f)))
  def ~~?(f: (D, E, F) => Boolean) = new ReductionRule2_3[Y, Z, A, B, C](append(exec(stack3(Pop), f)))
  def ~~?(f: (E, F) => Boolean) = new ReductionRule2_4[Y, Z, A, B, C, D](append(exec(stack2(Pop), f)))
  def ~~?(f: F => Boolean) = new ReductionRule2_5[Y, Z, A, B, C, D, E](append(exec(stack1(Pop), f)))
  def ~~%[X](f: (X, A, B, C, D, E, F) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack7(Pop), f))))
  def ~~%(f: (A, B, C, D, E, F) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack6(Pop), f))))
  def ~~%(f: (B, C, D, E, F) => Unit) = new ReductionRule2[Y, Z, A](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (C, D, E, F) => Unit) = new ReductionRule2_2[Y, Z, A, B](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (D, E, F) => Unit) = new ReductionRule2_3[Y, Z, A, B, C](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (E, F) => Unit) = new ReductionRule2_4[Y, Z, A, B, C, D](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: F => Unit) = new ReductionRule2_5[Y, Z, A, B, C, D, E](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D, E, F) => Boolean) = withMatcher(append(exec(stack6(Peek), f)))
  def ~~~?(f: (B, C, D, E, F) => Boolean) = withMatcher(append(exec(stack5(Peek), f)))
  def ~~~?(f: (C, D, E, F) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (D, E, F) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (E, F) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: F => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D, E, F) => Unit) = withMatcher(append(ok(exec(stack6(Peek), f))))
  def ~~~%(f: (B, C, D, E, F) => Unit) = withMatcher(append(ok(exec(stack5(Peek), f))))
  def ~~~%(f: (C, D, E, F) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (D, E, F) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (E, F) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: F => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[YY <: Y, ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D, EE >: E, FF >: F](other: ReductionRule2_6[YY, ZZ, AA, BB, CC, DD, EE, FF]) = new ReductionRule2_6[YY, ZZ, AA, BB, CC, DD, EE, FF](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule2_6[Y, Z, A, B, C, D, E, F](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking two values off the value stack and replacing them with seven values.
  */
class ReductionRule2_7[-Y, -Z, +A, +B, +C, +D, +E, +F, +G](val matcher: Matcher) extends ReductionRule {
  def ~[EE >: E, FF >: F, GG >: G](other: PopRule3[EE, FF, GG]) = new ReductionRule2_4[Y, Z, A, B, C, D](append(other))
  def ~[FF >: F, GG >: G](other: PopRule2[FF, GG]) = new ReductionRule2_5[Y, Z, A, B, C, D, E](append(other))
  def ~[GG >: G](other: PopRule1[GG]) = new ReductionRule2_6[Y, Z, A, B, C, D, E, F](append(other))
  def ~[EE >: E, FF >: F, GG >: G, RE](other: ReductionRule3[EE, FF, GG, RE]) = new ReductionRule2_5[Y, Z, A, B, C, D, RE](append(other))
  def ~[EE >: E, FF >: F, GG >: G, RE, RF](other: ReductionRule3_2[EE, FF, GG, RE, RF]) = new ReductionRule2_6[Y, Z, A, B, C, D, RE, RF](append(other))
  def ~[EE >: E, FF >: F, GG >: G, RE, RF, RG](other: ReductionRule3_3[EE, FF, GG, RE, RF, RG]) = new ReductionRule2_7[Y, Z, A, B, C, D, RE, RF, RG](append(other))
  def ~[FF >: F, GG >: G, RF](other: ReductionRule2[FF, GG, RF]) = new ReductionRule2_6[Y, Z, A, B, C, D, E, F](append(other))
  def ~[FF >: F, GG >: G, RF, RG](other: ReductionRule2_2[FF, GG, RF, RG]) = new ReductionRule2_7[Y, Z, A, B, C, D, E, F, G](append(other))
  def ~[GG >: G, RG](other: ReductionRule1[GG, RG]) = new ReductionRule2_7[Y, Z, A, B, C, D, E, F, RG](append(other))
  def ~~>[RA](f: (A, B, C, D, E, F, G) => RA) = new ReductionRule2[Y, Z, RA](append(push(exec(stack7(Pop), f))))
  def ~~>[RB](f: (B, C, D, E, F, G) => RB) = new ReductionRule2_2[Y, Z, A, RB](append(push(exec(stack6(Pop), f))))
  def ~~>[RC](f: (C, D, E, F, G) => RC) = new ReductionRule2_3[Y, Z, A, B, RC](append(push(exec(stack5(Pop), f))))
  def ~~>[RD](f: (D, E, F, G) => RD) = new ReductionRule2_4[Y, Z, A, B, C, RD](append(push(exec(stack4(Pop), f))))
  def ~~>[RE](f: (E, F, G) => RE) = new ReductionRule2_5[Y, Z, A, B, C, D, RE](append(push(exec(stack3(Pop), f))))
  def ~~>[RF](f: (F, G) => RF) = new ReductionRule2_6[Y, Z, A, B, C, D, E, RF](append(push(exec(stack2(Pop), f))))
  def ~~>[RG](f: G => RG) = new ReductionRule2_7[Y, Z, A, B, C, D, E, F, RG](append(push(exec(stack1(Pop), f))))
  def ~~?(f: (A, B, C, D, E, F, G) => Boolean) = new PopRule2[Y, Z](append(exec(stack7(Pop), f)))
  def ~~?(f: (B, C, D, E, F, G) => Boolean) = new ReductionRule2[Y, Z, A](append(exec(stack6(Pop), f)))
  def ~~?(f: (C, D, E, F, G) => Boolean) = new ReductionRule2_2[Y, Z, A, B](append(exec(stack5(Pop), f)))
  def ~~?(f: (D, E, F, G) => Boolean) = new ReductionRule2_3[Y, Z, A, B, C](append(exec(stack4(Pop), f)))
  def ~~?(f: (E, F, G) => Boolean) = new ReductionRule2_4[Y, Z, A, B, C, D](append(exec(stack3(Pop), f)))
  def ~~?(f: (F, G) => Boolean) = new ReductionRule2_5[Y, Z, A, B, C, D, E](append(exec(stack2(Pop), f)))
  def ~~?(f: G => Boolean) = new ReductionRule2_6[Y, Z, A, B, C, D, E, F](append(exec(stack1(Pop), f)))
  def ~~%(f: (A, B, C, D, E, F, G) => Unit) = new PopRule2[Y, Z](append(ok(exec(stack7(Pop), f))))
  def ~~%(f: (B, C, D, E, F, G) => Unit) = new ReductionRule2[Y, Z, A](append(ok(exec(stack6(Pop), f))))
  def ~~%(f: (C, D, E, F, G) => Unit) = new ReductionRule2_2[Y, Z, A, B](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (D, E, F, G) => Unit) = new ReductionRule2_3[Y, Z, A, B, C](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (E, F, G) => Unit) = new ReductionRule2_4[Y, Z, A, B, C, D](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (F, G) => Unit) = new ReductionRule2_5[Y, Z, A, B, C, D, E](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: G => Unit) = new ReductionRule2_6[Y, Z, A, B, C, D, E, F](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D, E, F, G) => Boolean) = withMatcher(append(exec(stack7(Peek), f)))
  def ~~~?(f: (B, C, D, E, F, G) => Boolean) = withMatcher(append(exec(stack6(Peek), f)))
  def ~~~?(f: (C, D, E, F, G) => Boolean) = withMatcher(append(exec(stack5(Peek), f)))
  def ~~~?(f: (D, E, F, G) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (E, F, G) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (F, G) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: G => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack7(Peek), f))))
  def ~~~%(f: (B, C, D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack6(Peek), f))))
  def ~~~%(f: (C, D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack5(Peek), f))))
  def ~~~%(f: (D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (E, F, G) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (F, G) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: G => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[YY <: Y, ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D, EE >: E, FF >: F, GG >: G](other: ReductionRule2_7[YY, ZZ, AA, BB, CC, DD, EE, FF, GG]) = new ReductionRule2_7[YY, ZZ, AA, BB, CC, DD, EE, FF, GG](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule2_7[Y, Z, A, B, C, D, E, F, G](matcher).asInstanceOf[this.type]
}

/**
 * A rule taking three values off the value stack and replacing them with one other value.
 */
class ReductionRule3[-X, -Y, -Z, +A](val matcher: Matcher) extends ReductionRule {
  def ~[AA >: A](other: PopRule1[AA]) = new PopRule3[X, Y, Z](append(other))
  def ~[AA >: A, RA](other: ReductionRule1[AA, RA]) = new ReductionRule3[X, Y, Z, RA](append(other))
  def ~[AA >: A, RA, B](other: ReductionRule1_2[AA, RA, B]) = new ReductionRule3_2[X, Y, Z, RA, B](append(other))
  def ~[AA >: A, RA, B, C](other: ReductionRule1_3[AA, RA, B, C]) = new ReductionRule3_3[X, Y, Z, RA, B, C](append(other))
  def ~[AA >: A, RA, B, C, D](other: ReductionRule1_4[AA, RA, B, C, D]) = new ReductionRule3_4[X, Y, Z, RA, B, C, D](append(other))
  def ~[AA >: A, RA, B, C, D, E](other: ReductionRule1_5[AA, RA, B, C, D, E]) = new ReductionRule3_5[X, Y, Z, RA, B, C, D, E](append(other))
  def ~[AA >: A, RA, B, C, D, E, F](other: ReductionRule1_6[AA, RA, B, C, D, E, F]) = new ReductionRule3_6[X, Y, Z, RA, B, C, D, E, F](append(other))
  def ~[AA >: A, RA, B, C, D, E, F, G](other: ReductionRule1_7[AA, RA, B, C, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, RA, B, C, D, E, F, G](append(other))
  def ~[B](other: Rule1[B]) = new ReductionRule3_2[X, Y, Z, A, B](append(other))
  def ~[B, C](other: Rule2[B, C]) = new ReductionRule3_3[X, Y, Z, A, B, C](append(other))
  def ~[B, C, D](other: Rule3[B, C, D]) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(other))
  def ~[B, C, D, E](other: Rule4[B, C, D, E]) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](append(other))
  def ~[B, C, D, E, F](other: Rule5[B, C, D, E, F]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, F](append(other))
  def ~[B, C, D, E, F, G](other: Rule6[B, C, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[RA](f: A => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack1(Pop), f))))
  def ~~?(f: A => Boolean) = new PopRule3[X, Y, Z](append(exec(stack1(Pop), f)))
  def ~~%(f: A => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: A => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: A => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[XX <: X, YY <: Y, ZZ <: Z, AA >: A](other: ReductionRule3[XX, YY, ZZ, AA]) = new ReductionRule3[XX, YY, ZZ, AA](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule3[X, Y, Z, A](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking three values off the value stack and replacing them with two values.
  */
class ReductionRule3_2[-X, -Y, -Z, +A, +B](val matcher: Matcher) extends ReductionRule {
  def ~[AA >: A, BB >: B](other: PopRule2[AA, BB]) = new PopRule3[X, Y, Z](append(other))
  def ~[BB >: B](other: PopRule1[BB]) = new ReductionRule3[X, Y, Z, A](append(other))
  def ~[AA >: A, BB >: B, RA](other: ReductionRule2[AA, BB, RA]) = new ReductionRule3[X, Y, Z, RA](append(other))
  def ~[AA >: A, BB >: B, RA, RB](other: ReductionRule2_2[AA, BB, RA, RB]) = new ReductionRule3_2[X, Y, Z, RA, RB](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C](other: ReductionRule2_3[AA, BB, RA, RB, C]) = new ReductionRule3_3[X, Y, Z, RA, RB, C](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D](other: ReductionRule2_4[AA, BB, RA, RB, C, D]) = new ReductionRule3_4[X, Y, Z, RA, RB, C, D](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D, E](other: ReductionRule2_5[AA, BB, RA, RB, C, D, E]) = new ReductionRule3_5[X, Y, Z, RA, RB, C, D, E](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D, E, F](other: ReductionRule2_6[AA, BB, RA, RB, C, D, E, F]) = new ReductionRule3_6[X, Y, Z, RA, RB, C, D, E, F](append(other))
  def ~[AA >: A, BB >: B, RA, RB, C, D, E, F, G](other: ReductionRule2_7[AA, BB, RA, RB, C, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, RA, RB, C, D, E, F, G](append(other))
  def ~[BB >: B, RB](other: ReductionRule1[BB, RB]) = new ReductionRule3_2[X, Y, Z, A, RB](append(other))
  def ~[BB >: B, RB, C](other: ReductionRule1_2[BB, RB, C]) = new ReductionRule3_3[X, Y, Z, A, RB, C](append(other))
  def ~[BB >: B, RB, C, D](other: ReductionRule1_3[BB, RB, C, D]) = new ReductionRule3_4[X, Y, Z, A, RB, C, D](append(other))
  def ~[BB >: B, RB, C, D, E](other: ReductionRule1_4[BB, RB, C, D, E]) = new ReductionRule3_5[X, Y, Z, A, RB, C, D, E](append(other))
  def ~[BB >: B, RB, C, D, E, F](other: ReductionRule1_5[BB, RB, C, D, E, F]) = new ReductionRule3_6[X, Y, Z, A, RB, C, D, E, F](append(other))
  def ~[BB >: B, RB, C, D, E, F, G](other: ReductionRule1_6[BB, RB, C, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, A, RB, C, D, E, F, G](append(other))
  def ~[C](other: Rule1[C]) = new ReductionRule3_3[X, Y, Z, A, B, C](append(other))
  def ~[C, D](other: Rule2[C, D]) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(other))
  def ~[C, D, E](other: Rule3[C, D, E]) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](append(other))
  def ~[C, D, E, F](other: Rule4[C, D, E, F]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, F](append(other))
  def ~[C, D, E, F, G](other: Rule5[C, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[RA](f: (A, B) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack2(Pop), f))))
  def ~~>[RB](f: B => RB) = new ReductionRule3_2[X, Y, Z, A, RB](append(push(exec(stack1(Pop), f))))
  def ~~?(f: (A, B) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack2(Pop), f)))
  def ~~?(f: B => Boolean) = new ReductionRule3[X, Y, Z, A](append(exec(stack1(Pop), f)))
  def ~~%(f: (A, B) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: B => Unit) = new ReductionRule3[X, Y, Z, A](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: B => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: B => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[XX <: X, YY <: Y, ZZ <: Z, AA >: A, BB >: B](other: ReductionRule3_2[XX, YY, ZZ, AA, BB]) = new ReductionRule3_2[XX, YY, ZZ, AA, BB](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule3_2[X, Y, Z, A, B](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking three values off the value stack and replacing them with three values.
  */
class ReductionRule3_3[-X, -Y, -Z, +A, +B, +C](val matcher: Matcher) extends ReductionRule {
  def ~[AA >: A, BB >: B, CC >: C](other: PopRule3[AA, BB, CC]) = new PopRule3[X, Y, Z](append(other))
  def ~[BB >: B, CC >: C](other: PopRule2[BB, CC]) = new ReductionRule3[X, Y, Z, A](append(other))
  def ~[CC >: C](other: PopRule1[CC]) = new ReductionRule3_2[X, Y, Z, A, B](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA](other: ReductionRule3[AA, BB, CC, RA]) = new ReductionRule3[X, Y, Z, RA](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB](other: ReductionRule3_2[AA, BB, CC, RA, RB]) = new ReductionRule3_2[X, Y, Z, RA, RB](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC](other: ReductionRule3_3[AA, BB, CC, RA, RB, RC]) = new ReductionRule3_3[X, Y, Z, RA, RB, RC](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D](other: ReductionRule3_4[AA, BB, CC, RA, RB, RC, D]) = new ReductionRule3_4[X, Y, Z, RA, RB, RC, D](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D, E](other: ReductionRule3_5[AA, BB, CC, RA, RB, RC, D, E]) = new ReductionRule3_5[X, Y, Z, RA, RB, RC, D, E](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D, E, F](other: ReductionRule3_6[AA, BB, CC, RA, RB, RC, D, E, F]) = new ReductionRule3_6[X, Y, Z, RA, RB, RC, D, E, F](append(other))
  def ~[AA >: A, BB >: B, CC >: C, RA, RB, RC, D, E, F, G](other: ReductionRule3_7[AA, BB, CC, RA, RB, RC, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, RA, RB, RC, D, E, F, G](append(other))
  def ~[BB >: B, CC >: C, RB](other: ReductionRule2[BB, CC, RB]) = new ReductionRule3_2[X, Y, Z, A, RB](append(other))
  def ~[BB >: B, CC >: C, RB, RC](other: ReductionRule2_2[BB, CC, RB, RC]) = new ReductionRule3_3[X, Y, Z, A, RB, RC](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D](other: ReductionRule2_3[BB, CC, RB, RC, D]) = new ReductionRule3_4[X, Y, Z, A, RB, RC, D](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D, E](other: ReductionRule2_4[BB, CC, RB, RC, D, E]) = new ReductionRule3_5[X, Y, Z, A, RB, RC, D, E](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D, E, F](other: ReductionRule2_5[BB, CC, RB, RC, D, E, F]) = new ReductionRule3_6[X, Y, Z, A, RB, RC, D, E, F](append(other))
  def ~[BB >: B, CC >: C, RB, RC, D, E, F, G](other: ReductionRule2_6[BB, CC, RB, RC, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, A, RB, RC, D, E, F, G](append(other))
  def ~[CC >: C, RC](other: ReductionRule1[CC, RC]) = new ReductionRule3_3[X, Y, Z, A, B, RC](append(other))
  def ~[CC >: C, RC, D](other: ReductionRule1_2[CC, RC, D]) = new ReductionRule3_4[X, Y, Z, A, B, RC, D](append(other))
  def ~[CC >: C, RC, D, E](other: ReductionRule1_3[CC, RC, D, E]) = new ReductionRule3_5[X, Y, Z, A, B, RC, D, E](append(other))
  def ~[CC >: C, RC, D, E, F](other: ReductionRule1_4[CC, RC, D, E, F]) = new ReductionRule3_6[X, Y, Z, A, B, RC, D, E, F](append(other))
  def ~[CC >: C, RC, D, E, F, G](other: ReductionRule1_5[CC, RC, D, E, F, G]) = new ReductionRule3_7[X, Y, Z, A, B, RC, D, E, F, G](append(other))
  def ~[D](other: Rule1[D]) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(other))
  def ~[D, E](other: Rule2[D, E]) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](append(other))
  def ~[D, E, F](other: Rule3[D, E, F]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, F](append(other))
  def ~[D, E, F, G](other: Rule4[D, E, F, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[RA](f: (A, B, C) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack3(Pop), f))))
  def ~~>[RB](f: (B, C) => RB) = new ReductionRule3_2[X, Y, Z, A, RB](append(push(exec(stack2(Pop), f))))
  def ~~>[RC](f: C => RC) = new ReductionRule3_3[X, Y, Z, A, B, RC](append(push(exec(stack1(Pop), f))))
  def ~~?(f: (A, B, C) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack3(Pop), f)))
  def ~~?(f: (B, C) => Boolean) = new ReductionRule3[X, Y, Z, A](append(exec(stack2(Pop), f)))
  def ~~?(f: C => Boolean) = new ReductionRule3_2[X, Y, Z, A, B](append(exec(stack1(Pop), f)))
  def ~~%(f: (A, B, C) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (B, C) => Unit) = new ReductionRule3[X, Y, Z, A](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: C => Unit) = new ReductionRule3_2[X, Y, Z, A, B](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (B, C) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: C => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (B, C) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: C => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[XX <: X, YY <: Y, ZZ <: Z, AA >: A, BB >: B, CC >: C](other: ReductionRule3_3[XX, YY, ZZ, AA, BB, CC]) = new ReductionRule3_3[XX, YY, ZZ, AA, BB, CC](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule3_3[X, Y, Z, A, B, C](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking three values off the value stack and replacing them with four values.
  */
class ReductionRule3_4[-X, -Y, -Z, +A, +B, +C, +D](val matcher: Matcher) extends ReductionRule {
  def ~[BB >: B, CC >: C, DD >: D](other: PopRule3[BB, CC, DD]) = new ReductionRule3[X, Y, Z, A](append(other))
  def ~[CC >: C, DD >: D](other: PopRule2[CC, DD]) = new ReductionRule3_2[X, Y, Z, A, B](append(other))
  def ~[DD >: D](other: PopRule1[DD]) = new ReductionRule3_3[X, Y, Z, A, B, C](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB](other: ReductionRule3[BB, CC, DD, RB]) = new ReductionRule3_2[X, Y, Z, A, RB](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC](other: ReductionRule3_2[BB, CC, DD, RB, RC]) = new ReductionRule3_3[X, Y, Z, A, RB, RC](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD](other: ReductionRule3_3[BB, CC, DD, RB, RC, RD]) = new ReductionRule3_4[X, Y, Z, A, RB, RC, RD](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD, E](other: ReductionRule3_4[BB, CC, DD, RB, RC, RD, E]) = new ReductionRule3_5[X, Y, Z, A, RB, RC, RD, E](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD, E, F](other: ReductionRule3_5[BB, CC, DD, RB, RC, RD, E, F]) = new ReductionRule3_6[X, Y, Z, A, RB, RC, RD, E, F](append(other))
  def ~[BB >: B, CC >: C, DD >: D, RB, RC, RD, E, F, G](other: ReductionRule3_6[BB, CC, DD, RB, RC, RD, E, F, G]) = new ReductionRule3_7[X, Y, Z, A, RB, RC, RD, E, F, G](append(other))
  def ~[CC >: C, DD >: D, RC](other: ReductionRule2[CC, DD, RC]) = new ReductionRule3_3[X, Y, Z, A, B, RC](append(other))
  def ~[CC >: C, DD >: D, RC, RD](other: ReductionRule2_2[CC, DD, RC, RD]) = new ReductionRule3_4[X, Y, Z, A, B, RC, RD](append(other))
  def ~[CC >: C, DD >: D, RC, RD, E](other: ReductionRule2_3[CC, DD, RC, RD, E]) = new ReductionRule3_5[X, Y, Z, A, B, RC, RD, E](append(other))
  def ~[CC >: C, DD >: D, RC, RD, E, F](other: ReductionRule2_4[CC, DD, RC, RD, E, F]) = new ReductionRule3_6[X, Y, Z, A, B, RC, RD, E, F](append(other))
  def ~[CC >: C, DD >: D, RC, RD, E, F, G](other: ReductionRule2_5[CC, DD, RC, RD, E, F, G]) = new ReductionRule3_7[X, Y, Z, A, B, RC, RD, E, F, G](append(other))
  def ~[DD >: D, RD](other: ReductionRule1[DD, RD]) = new ReductionRule3_4[X, Y, Z, A, B, C, RD](append(other))
  def ~[DD >: D, RD, E](other: ReductionRule1_2[DD, RD, E]) = new ReductionRule3_5[X, Y, Z, A, B, C, RD, E](append(other))
  def ~[DD >: D, RD, E, F](other: ReductionRule1_3[DD, RD, E, F]) = new ReductionRule3_6[X, Y, Z, A, B, C, RD, E, F](append(other))
  def ~[DD >: D, RD, E, F, G](other: ReductionRule1_4[DD, RD, E, F, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, RD, E, F, G](append(other))
  def ~[E](other: Rule1[E]) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](append(other))
  def ~[E, F](other: Rule2[E, F]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, F](append(other))
  def ~[E, F, G](other: Rule3[E, F, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[RA](f: (A, B, C, D) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack4(Pop), f))))
  def ~~>[RB](f: (B, C, D) => RB) = new ReductionRule3_2[X, Y, Z, A, RB](append(push(exec(stack3(Pop), f))))
  def ~~>[RC](f: (C, D) => RC) = new ReductionRule3_3[X, Y, Z, A, B, RC](append(push(exec(stack2(Pop), f))))
  def ~~>[RD](f: D => RD) = new ReductionRule3_4[X, Y, Z, A, B, C, RD](append(push(exec(stack1(Pop), f))))
  def ~~?(f: (A, B, C, D) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack4(Pop), f)))
  def ~~?(f: (B, C, D) => Boolean) = new ReductionRule3[X, Y, Z, A](append(exec(stack3(Pop), f)))
  def ~~?(f: (C, D) => Boolean) = new ReductionRule3_2[X, Y, Z, A, B](append(exec(stack2(Pop), f)))
  def ~~?(f: D => Boolean) = new ReductionRule3_3[X, Y, Z, A, B, C](append(exec(stack1(Pop), f)))
  def ~~%(f: (A, B, C, D) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (B, C, D) => Unit) = new ReductionRule3[X, Y, Z, A](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (C, D) => Unit) = new ReductionRule3_2[X, Y, Z, A, B](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: D => Unit) = new ReductionRule3_3[X, Y, Z, A, B, C](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (B, C, D) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (C, D) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: D => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (B, C, D) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (C, D) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: D => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[XX <: X, YY <: Y, ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D](other: ReductionRule3_4[XX, YY, ZZ, AA, BB, CC, DD]) = new ReductionRule3_4[XX, YY, ZZ, AA, BB, CC, DD](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule3_4[X, Y, Z, A, B, C, D](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking three values off the value stack and replacing them with five values.
  */
class ReductionRule3_5[-X, -Y, -Z, +A, +B, +C, +D, +E](val matcher: Matcher) extends ReductionRule {
  def ~[CC >: C, DD >: D, EE >: E](other: PopRule3[CC, DD, EE]) = new ReductionRule3_2[X, Y, Z, A, B](append(other))
  def ~[DD >: D, EE >: E](other: PopRule2[DD, EE]) = new ReductionRule3_3[X, Y, Z, A, B, C](append(other))
  def ~[EE >: E](other: PopRule1[EE]) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC](other: ReductionRule3[CC, DD, EE, RC]) = new ReductionRule3_3[X, Y, Z, A, B, RC](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD](other: ReductionRule3_2[CC, DD, EE, RC, RD]) = new ReductionRule3_4[X, Y, Z, A, B, C, RD](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD, RE](other: ReductionRule3_3[CC, DD, EE, RC, RD, RE]) = new ReductionRule3_5[X, Y, Z, A, B, C, RD, RE](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD, RE, F](other: ReductionRule3_4[CC, DD, EE, RC, RD, RE, F]) = new ReductionRule3_6[X, Y, Z, A, B, C, RD, RE, F](append(other))
  def ~[CC >: C, DD >: D, EE >: E, RC, RD, RE, F, G](other: ReductionRule3_5[CC, DD, EE, RC, RD, RE, F, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, RD, RE, F, G](append(other))
  def ~[DD >: D, EE >: E, RD](other: ReductionRule2[DD, EE, RD]) = new ReductionRule3_4[X, Y, Z, A, B, C, RD](append(other))
  def ~[DD >: D, EE >: E, RD, RE](other: ReductionRule2_2[DD, EE, RD, RE]) = new ReductionRule3_5[X, Y, Z, A, B, C, RD, RE](append(other))
  def ~[DD >: D, EE >: E, RD, RE, F](other: ReductionRule2_3[DD, EE, RD, RE, F]) = new ReductionRule3_6[X, Y, Z, A, B, C, RD, RE, F](append(other))
  def ~[DD >: D, EE >: E, RD, RE, F, G](other: ReductionRule2_4[DD, EE, RD, RE, F, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, RD, RE, F, G](append(other))
  def ~[EE >: E, RE](other: ReductionRule1[EE, RE]) = new ReductionRule3_5[X, Y, Z, A, B, C, D, RE](append(other))
  def ~[EE >: E, RE, F](other: ReductionRule1_2[EE, RE, F]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, RE, F](append(other))
  def ~[EE >: E, RE, F, G](other: ReductionRule1_3[EE, RE, F, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, RE, F, G](append(other))
  def ~[F](other: Rule1[F]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, F](append(other))
  def ~[F, G](other: Rule2[F, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[RA](f: (A, B, C, D, E) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack5(Pop), f))))
  def ~~>[RB](f: (B, C, D, E) => RB) = new ReductionRule3_2[X, Y, Z, A, RB](append(push(exec(stack4(Pop), f))))
  def ~~>[RC](f: (C, D, E) => RC) = new ReductionRule3_3[X, Y, Z, A, B, RC](append(push(exec(stack3(Pop), f))))
  def ~~>[RD](f: (D, E) => RD) = new ReductionRule3_4[X, Y, Z, A, B, C, RD](append(push(exec(stack2(Pop), f))))
  def ~~>[RE](f: E => RE) = new ReductionRule3_5[X, Y, Z, A, B, C, D, RE](append(push(exec(stack1(Pop), f))))
  def ~~?(f: (A, B, C, D, E) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack5(Pop), f)))
  def ~~?(f: (B, C, D, E) => Boolean) = new ReductionRule3[X, Y, Z, A](append(exec(stack4(Pop), f)))
  def ~~?(f: (C, D, E) => Boolean) = new ReductionRule3_2[X, Y, Z, A, B](append(exec(stack3(Pop), f)))
  def ~~?(f: (D, E) => Boolean) = new ReductionRule3_3[X, Y, Z, A, B, C](append(exec(stack2(Pop), f)))
  def ~~?(f: E => Boolean) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(exec(stack1(Pop), f)))
  def ~~%(f: (A, B, C, D, E) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (B, C, D, E) => Unit) = new ReductionRule3[X, Y, Z, A](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (C, D, E) => Unit) = new ReductionRule3_2[X, Y, Z, A, B](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (D, E) => Unit) = new ReductionRule3_3[X, Y, Z, A, B, C](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: E => Unit) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D, E) => Boolean) = withMatcher(append(exec(stack5(Peek), f)))
  def ~~~?(f: (B, C, D, E) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (C, D, E) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (D, E) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: E => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D, E) => Unit) = withMatcher(append(ok(exec(stack5(Peek), f))))
  def ~~~%(f: (B, C, D, E) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (C, D, E) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (D, E) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: E => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[XX <: X, YY <: Y, ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D, EE >: E](other: ReductionRule3_5[XX, YY, ZZ, AA, BB, CC, DD, EE]) = new ReductionRule3_5[XX, YY, ZZ, AA, BB, CC, DD, EE](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking three values off the value stack and replacing them with six values.
  */
class ReductionRule3_6[-X, -Y, -Z, +A, +B, +C, +D, +E, +F](val matcher: Matcher) extends ReductionRule {
  def ~[DD >: D, EE >: E, FF >: F](other: PopRule3[DD, EE, FF]) = new ReductionRule3_3[X, Y, Z, A, B, C](append(other))
  def ~[EE >: E, FF >: F](other: PopRule2[EE, FF]) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(other))
  def ~[FF >: F](other: PopRule1[FF]) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD](other: ReductionRule3[DD, EE, FF, RD]) = new ReductionRule3_4[X, Y, Z, A, B, C, RD](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD, RE](other: ReductionRule3_2[DD, EE, FF, RD, RE]) = new ReductionRule3_5[X, Y, Z, A, B, C, RD, RE](append(other))
  def ~[DD >: D, EE >: E, FF >: F, RD, RE, RF](other: ReductionRule3_3[DD, EE, FF, RD, RE, RF]) = new ReductionRule3_6[X, Y, Z, A, B, C, RD, RE, RF](append(other))
  def ~[EE >: E, FF >: F, RE](other: ReductionRule2[EE, FF, RE]) = new ReductionRule3_5[X, Y, Z, A, B, C, D, RE](append(other))
  def ~[EE >: E, FF >: F, RE, RF](other: ReductionRule2_2[EE, FF, RE, RF]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, RE, RF](append(other))
  def ~[EE >: E, FF >: F, RE, RF, G](other: ReductionRule2_3[EE, FF, RE, RF, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, RE, RF, G](append(other))
  def ~[FF >: F, RF](other: ReductionRule1[FF, RF]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, RF](append(other))
  def ~[FF >: F, RF, G](other: ReductionRule1_2[FF, RF, G]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, RF, G](append(other))
  def ~[G](other: Rule1[G]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, F, G](append(other))
  def ~~>[RA](f: (A, B, C, D, E, F) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack6(Pop), f))))
  def ~~>[RB](f: (B, C, D, E, F) => RB) = new ReductionRule3_2[X, Y, Z, A, RB](append(push(exec(stack5(Pop), f))))
  def ~~>[RC](f: (C, D, E, F) => RC) = new ReductionRule3_3[X, Y, Z, A, B, RC](append(push(exec(stack4(Pop), f))))
  def ~~>[RD](f: (D, E, F) => RD) = new ReductionRule3_4[X, Y, Z, A, B, C, RD](append(push(exec(stack3(Pop), f))))
  def ~~>[RE](f: (E, F) => RE) = new ReductionRule3_5[X, Y, Z, A, B, C, D, RE](append(push(exec(stack2(Pop), f))))
  def ~~>[RF](f: F => RF) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, RF](append(push(exec(stack1(Pop), f))))
  def ~~?(f: (A, B, C, D, E, F) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack6(Pop), f)))
  def ~~?(f: (B, C, D, E, F) => Boolean) = new ReductionRule3[X, Y, Z, A](append(exec(stack5(Pop), f)))
  def ~~?(f: (C, D, E, F) => Boolean) = new ReductionRule3_2[X, Y, Z, A, B](append(exec(stack4(Pop), f)))
  def ~~?(f: (D, E, F) => Boolean) = new ReductionRule3_3[X, Y, Z, A, B, C](append(exec(stack3(Pop), f)))
  def ~~?(f: (E, F) => Boolean) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(exec(stack2(Pop), f)))
  def ~~?(f: F => Boolean) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](append(exec(stack1(Pop), f)))
  def ~~%(f: (A, B, C, D, E, F) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack6(Pop), f))))
  def ~~%(f: (B, C, D, E, F) => Unit) = new ReductionRule3[X, Y, Z, A](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (C, D, E, F) => Unit) = new ReductionRule3_2[X, Y, Z, A, B](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (D, E, F) => Unit) = new ReductionRule3_3[X, Y, Z, A, B, C](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (E, F) => Unit) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: F => Unit) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D, E, F) => Boolean) = withMatcher(append(exec(stack6(Peek), f)))
  def ~~~?(f: (B, C, D, E, F) => Boolean) = withMatcher(append(exec(stack5(Peek), f)))
  def ~~~?(f: (C, D, E, F) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (D, E, F) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (E, F) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: F => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D, E, F) => Unit) = withMatcher(append(ok(exec(stack6(Peek), f))))
  def ~~~%(f: (B, C, D, E, F) => Unit) = withMatcher(append(ok(exec(stack5(Peek), f))))
  def ~~~%(f: (C, D, E, F) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (D, E, F) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (E, F) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: F => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[XX <: X, YY <: Y, ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D, EE >: E, FF >: F](other: ReductionRule3_6[XX, YY, ZZ, AA, BB, CC, DD, EE, FF]) = new ReductionRule3_6[XX, YY, ZZ, AA, BB, CC, DD, EE, FF](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, F](matcher).asInstanceOf[this.type]
}

/**
  * A rule taking three values off the value stack and replacing them with seven values.
  */
class ReductionRule3_7[-X, -Y, -Z, +A, +B, +C, +D, +E, +F, +G](val matcher: Matcher) extends ReductionRule {
  def ~[EE >: E, FF >: F, GG >: G](other: PopRule3[EE, FF, GG]) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(other))
  def ~[FF >: F, GG >: G](other: PopRule2[FF, GG]) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](append(other))
  def ~[GG >: G](other: PopRule1[GG]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, F](append(other))
  def ~[EE >: E, FF >: F, GG >: G, RE](other: ReductionRule3[EE, FF, GG, RE]) = new ReductionRule3_5[X, Y, Z, A, B, C, D, RE](append(other))
  def ~[EE >: E, FF >: F, GG >: G, RE, RF](other: ReductionRule3_2[EE, FF, GG, RE, RF]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, RE, RF](append(other))
  def ~[EE >: E, FF >: F, GG >: G, RE, RF, RG](other: ReductionRule3_3[EE, FF, GG, RE, RF, RG]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, RE, RF, RG](append(other))
  def ~[FF >: F, GG >: G, RF](other: ReductionRule2[FF, GG, RF]) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, RF](append(other))
  def ~[FF >: F, GG >: G, RF, RG](other: ReductionRule2_2[FF, GG, RF, RG]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, RF, RG](append(other))
  def ~[GG >: G, RG](other: ReductionRule1[GG, RG]) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, F, RG](append(other))
  def ~~>[RA](f: (A, B, C, D, E, F, G) => RA) = new ReductionRule3[X, Y, Z, RA](append(push(exec(stack7(Pop), f))))
  def ~~>[RB](f: (B, C, D, E, F, G) => RB) = new ReductionRule3_2[X, Y, Z, A, RB](append(push(exec(stack6(Pop), f))))
  def ~~>[RC](f: (C, D, E, F, G) => RC) = new ReductionRule3_3[X, Y, Z, A, B, RC](append(push(exec(stack5(Pop), f))))
  def ~~>[RD](f: (D, E, F, G) => RD) = new ReductionRule3_4[X, Y, Z, A, B, C, RD](append(push(exec(stack4(Pop), f))))
  def ~~>[RE](f: (E, F, G) => RE) = new ReductionRule3_5[X, Y, Z, A, B, C, D, RE](append(push(exec(stack3(Pop), f))))
  def ~~>[RF](f: (F, G) => RF) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, RF](append(push(exec(stack2(Pop), f))))
  def ~~>[RG](f: G => RG) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, F, RG](append(push(exec(stack1(Pop), f))))
  def ~~?(f: (A, B, C, D, E, F, G) => Boolean) = new PopRule3[X, Y, Z](append(exec(stack7(Pop), f)))
  def ~~?(f: (B, C, D, E, F, G) => Boolean) = new ReductionRule3[X, Y, Z, A](append(exec(stack6(Pop), f)))
  def ~~?(f: (C, D, E, F, G) => Boolean) = new ReductionRule3_2[X, Y, Z, A, B](append(exec(stack5(Pop), f)))
  def ~~?(f: (D, E, F, G) => Boolean) = new ReductionRule3_3[X, Y, Z, A, B, C](append(exec(stack4(Pop), f)))
  def ~~?(f: (E, F, G) => Boolean) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(exec(stack3(Pop), f)))
  def ~~?(f: (F, G) => Boolean) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](append(exec(stack2(Pop), f)))
  def ~~?(f: G => Boolean) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, F](append(exec(stack1(Pop), f)))
  def ~~%(f: (A, B, C, D, E, F, G) => Unit) = new PopRule3[X, Y, Z](append(ok(exec(stack7(Pop), f))))
  def ~~%(f: (B, C, D, E, F, G) => Unit) = new ReductionRule3[X, Y, Z, A](append(ok(exec(stack6(Pop), f))))
  def ~~%(f: (C, D, E, F, G) => Unit) = new ReductionRule3_2[X, Y, Z, A, B](append(ok(exec(stack5(Pop), f))))
  def ~~%(f: (D, E, F, G) => Unit) = new ReductionRule3_3[X, Y, Z, A, B, C](append(ok(exec(stack4(Pop), f))))
  def ~~%(f: (E, F, G) => Unit) = new ReductionRule3_4[X, Y, Z, A, B, C, D](append(ok(exec(stack3(Pop), f))))
  def ~~%(f: (F, G) => Unit) = new ReductionRule3_5[X, Y, Z, A, B, C, D, E](append(ok(exec(stack2(Pop), f))))
  def ~~%(f: G => Unit) = new ReductionRule3_6[X, Y, Z, A, B, C, D, E, F](append(ok(exec(stack1(Pop), f))))
  def ~~~?(f: (A, B, C, D, E, F, G) => Boolean) = withMatcher(append(exec(stack7(Peek), f)))
  def ~~~?(f: (B, C, D, E, F, G) => Boolean) = withMatcher(append(exec(stack6(Peek), f)))
  def ~~~?(f: (C, D, E, F, G) => Boolean) = withMatcher(append(exec(stack5(Peek), f)))
  def ~~~?(f: (D, E, F, G) => Boolean) = withMatcher(append(exec(stack4(Peek), f)))
  def ~~~?(f: (E, F, G) => Boolean) = withMatcher(append(exec(stack3(Peek), f)))
  def ~~~?(f: (F, G) => Boolean) = withMatcher(append(exec(stack2(Peek), f)))
  def ~~~?(f: G => Boolean) = withMatcher(append(exec(stack1(Peek), f)))
  def ~~~%(f: (A, B, C, D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack7(Peek), f))))
  def ~~~%(f: (B, C, D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack6(Peek), f))))
  def ~~~%(f: (C, D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack5(Peek), f))))
  def ~~~%(f: (D, E, F, G) => Unit) = withMatcher(append(ok(exec(stack4(Peek), f))))
  def ~~~%(f: (E, F, G) => Unit) = withMatcher(append(ok(exec(stack3(Peek), f))))
  def ~~~%(f: (F, G) => Unit) = withMatcher(append(ok(exec(stack2(Peek), f))))
  def ~~~%(f: G => Unit) = withMatcher(append(ok(exec(stack1(Peek), f))))
  def |[XX <: X, YY <: Y, ZZ <: Z, AA >: A, BB >: B, CC >: C, DD >: D, EE >: E, FF >: F, GG >: G](other: ReductionRule3_7[XX, YY, ZZ, AA, BB, CC, DD, EE, FF, GG]) = new ReductionRule3_7[XX, YY, ZZ, AA, BB, CC, DD, EE, FF, GG](appendChoice(other))
  protected def withMatcher(matcher: Matcher) = new ReductionRule3_7[X, Y, Z, A, B, C, D, E, F, G](matcher).asInstanceOf[this.type]
}
