=begin
=module NumRu::Met::Themo
Thermodynamic constants and functions

by T Horinouchi

==Module constants
---R
    Gas constant of dry air [J.kg-1.K-1] (287.04)
---Rv
    Gas constant of water vapor [J.kg-1.K-1] (461.50)
---Cp
    heat capacity at constant pressure for dry air [J.kg-1.K-1] (1004.6)
---Cpv
    heat capacity at constant pressure for water vapor [J.kg-1.K-1] (1870.0)
---Kappa
    R/Cp
---T0 
    K - Celcius (273.15)
---Lat0
    Latent heat of vaporization at 0 degC [J.kg-1] (2.500780e6)

==Module functions
---theta_prs2tempC(theta,prs)
    derive temprature[Celcius] from potential temperature[K] and pressure[hPa]
---tempC_prs2theta(temp,prs)
    derive potential temperature[K] from temprature[Celcius] and pressure[hPa]
---q2r(q)
    specific humidity -> mixing ratio

    ARGEUMENTS
    * q: specific humidty
    RETURN VALUE
    * q/(1.0-q)

---r2q(r)
    mixing ratio -> specific humidity

    ARGEUMENTS
    * r [Numeric Scalar or Array]  mixing ratio [g/g]
    RETURN VALUE
    * r/(1.0+r)


---r_p2e(r,prs)
    derive water vapor partial pressure from mixing ratio and pressure

    ARGUMENTS
    * r:  water vapor mixing ratio[g/g]
    * prs:  pressure[hPa]
    * rratio = R / Rv
    RETURN VALUE
    *  e = prs*r/(rratio+r)  ---  water vapor pertial pressure [hPa]

---q_p2e(q,prs)
    derive water vapor partial pressure from specific humidity and pressure

    ARGUMENTS
    * q:  specific humidity[g/g]
    * prs:  pressure[hPa]
    * rratio = R / Rv
    RETURN VALUE
    * e = prs*q/(rratio+(1-rraio)*q)  ---  water vapor pertial pressure [hPa]

---lat(tempC)
    Returns the latent heat as a function of temperature

    ARGUMENTS
    * tempC (Numeric, Array of Numeric, or NArray-like classes): temperature [Celcius]
---t_dewpoint(r,prs)
    calculate dew point temperature using enhanced Teten''s formula
    from mixing ratioa and pressure.
    Based on thermolib3.f in APRS 4.5.1
   
    ARGUMENTS:
    * r:  water vapor mixing ratio [g/g]
    * prs:  pressure[hPa]

    CAUTION:
    * shapes of the arguments must agree with each other(whether scalar or array)
    RETURN VALUE:
    * dew point temperature[degC]
---theta_es(tempC,prs)
    Saturation potential temperature (no condensate -- conserv along pseudo-adiabatic processes).
    In an approximate formulation theta_es = theta * exp(L r_sat / CpT).

    ARGUMENTS
    * tempC (Numeric, Array of Numeric, or NArray-like classes): temperature [Celcius]
    * prs (Numeric, Array of Numeric, or NArray-like clasees): pressure [hPa]

    RETURN VALUE
    * theta_es (class dependent on arguments): saturation potential temperature

---theta_es_prs2tempC(th_es,prs)
    derive tempareture from saturation potential temperature.
    See ((<theta_es>)) for argument specification.
    This method reverses ((<theta_es>)) iteratively by the secont method.

---e_sat(tempC,prs)
    calculates saturation water vapor pressure using enhanced Teten''s formula.
    Based on thermolib3.f in APRS 4.5.1

    ARGUMENTS:
    * tempC [Numeric, Array, or NArray]  temperature[degC]
    * prs [Numeric, Array, or NArray]  pressure[hPa]

    CAUTION:

    shapes of the arguments must agree with each other(whether scalar or array)

    RETURN VALUE:
    * esat   saturation water vapor pressure [hPa]

---r_sat(tempC,prs)
    Same as e_sat but for saturation water vapor mixing ratio [g/g]

---q_sat(tempC,prs)
    Same as e_sat but for saturation water vapor specific humidity [g/g].

---rel_hum(tempC,e,prs)
    calculates relative humidity (from tempC,e,prs)
   
    ARGUMENTS:
    * tempC:  temperature[degC]
    * e:  water vapor pressure[hPa]
    * prs:  pressure[hPa]
    CAUTION:
    *  shapes of the arguments must agree with each other(whether scalar or array)
    RETURN VALUE:
    * relative humidity [hPa/hPa]
=end

require 'narray'
require 'numru/misc'

module NumRu
  module Met
    module Thermo

      include EMath     #  in NumRu::Misc ('numru/misc')
      extend EMath      #  in NumRu::Misc ('numru/misc')

      R = 287.04  # Gas constant of dry air [J.kg-1.K-1]
      Rv = 461.50 # Gas constant of water vapor [J.kg-1.K-1]
      Cp = 1004.6 # heat capacity at constant pressure for dry air [J.kg-1.K-1]
      Cpv = 1870.0 # heat capacity at constant pressure for water vapor [J.kg-1.K-1]
      Kappa = R/Cp  # = 2.0/7.0
      T0 = 273.15  # K - Celcius
      Lat0 =  2.500780e6  # Latent heat of vaporization at 0 degC [J.kg-1]

      # for Teten's formula -->
      @@sfwa = 1.0007
      @@sfwb = 3.46e-6             #  for p in hPa
      @@sewa = 6.1121              # es in hPa
      @@sewb = 17.502
      @@sewc = 32.18
      @@sfia = 1.0003
      @@sfib = 4.18e-6             # for p in hPa
      @@seia = 6.1115              # es in hPa
      @@seib = 22.452
      @@seic = 0.6
      # <--- for Teten's formula

      module_function

      def theta_prs2tempC(theta,prs)
	tempC = theta * (prs/1000.0)**Kappa - T0
      end

      def tempC_prs2theta(tempC,prs)
	theta = (tempC+T0) * (prs/1000.0)**(-Kappa)
      end

      def __ary_to_real_na(a)
	case a
	when Array
	  a = NArray.to_na(a)
	  a = a.to_f if a.typecode <= NArray::INT
	when NArray
	  a = a.to_f if a.typecode <= NArray::INT
	end
	a
      end
      private :__ary_to_real_na
#      def __ary_to_real_na(a)
#	a = NArray.to_na(a) if a.is_a?(Array)
#	a = a.to_f if (a.respond_to?(:typecode) && a.typecode <= NArray::INT)
#	a
#      end
#      private :__ary_to_real_na

      def q2r(q)
	# specific humidity -> mixing ratio
	#   q:  specific humidty [g/g]
	q = __ary_to_real_na(q)
	q/(1.0-q)
      end

      def r2q(r)
	# mixing ratio -> specific humidity
	#   r:  mixing ratio [g/g]
	r = __ary_to_real_na(r)
	r/(1.0+r)
      end

      def r_p2e(r,prs)
	#   r:  water vapor mixing ratio[g/g]
	#   prs:  pressure[hPa]
	r = __ary_to_real_na(r)
	prs = __ary_to_real_na(prs)
	rratio = R / Rv
	e = prs*r/(rratio+r)     # water vapor pertial pressure
      end
      def q_p2e(q,prs)
	#   q:  specific humidity[g/g]
	#   prs:  pressure[hPa]
	q = __ary_to_real_na(q)
	prs = __ary_to_real_na(prs)
	rratio = R / Rv
	e = prs*q/(rratio+(1-rraio)*q)     # water vapor pertial pressure
      end

      def lat(tempC)
	# latent heat [J.kg-1]
	# good for -100<T<50
	tempC = __ary_to_real_na(tempC)
	tempK = tempC+T0
	lat = Lat0*(T0/tempK)**(0.167+3.67E-4*tempK) 
      end

      def t_dewpoint(r,prs)
	# calculate dew point temperature using enhanced Teten''s formula
	# from mixing ratioa and pressure
	# Based on thermolib3.f in APRS 4.5.1

	r = __ary_to_real_na(r)
	prs = __ary_to_real_na(prs)

	rratio = R / Rv

	rvs = r
	if r.is_a?(Numeric)
	  rvs = 1.0e-8 if r < 1.0e-8
	else
	  rvs=r.dup
	  rvs[r < 1.0e-8] = 1.0e-8   # set dummy if extremely small to avoid
                                     # undeflow (as in arps)
        end
	es = prs*rvs/(rratio+rvs)

	# Firstly assume liquid water (correct later if not)

	f = @@sfwa + @@sfwb * prs
	ln = log( es/(f*@@sewa) )
	tdew = ( ln*@@sewc - T0*@@sewb ) / ( ln - @@sewb ) - T0

	ice_mask = tdew < 0
	if tdew.is_a?(Numeric) && ice_mask
1	  # ice
	  f = @@sfia + @@sfib * prs
	  ln = log( es/(f*@@seia) )
	  tdew = (ln*@@seic - T0*@@seib) / 
	         (ln-@@seib) - T0
	elsif !tdew.is_a?(Numeric) && ice_mask.any?
	  # ice
	  if prs.is_a?(Numeric)
	    f = @@sfia + @@sfib * prs
	  else
	    f = @@sfia + @@sfib * prs[ice_mask]
	  end
	  ln[ice_mask] = log( es[ice_mask]/(f*@@seia) )
	  tdew[ice_mask] = (ln[ice_mask]*@@seic - T0*@@seib) / 
	                   (ln[ice_mask]-@@seib) - T0
	end
	tdew
      end

      def theta_es(tempC,prs)
	tempC = __ary_to_real_na(tempC)
	prs = __ary_to_real_na(prs)
	tempK = tempC+T0
	theta_es = tempK * (1000.0/prs)**Kappa *
	             exp( lat(tempC) * r_sat(tempC,prs)/Cp/tempK )
      end

      def theta_es_prs2tempC(th_es,prs)
	# root finding is by the secont method
	eps = 0.1    # error allowed in K
	x2 = th_es * (prs/1000.0)**Kappa - T0   # first guess: when r_sat=0
	if x2.is_a?(Numeric)
	  x2 = 40 if x2 > 40
	else
	  x2[x2 > 40] = 40
	end

	y2 = theta_es(x2,prs)
        x1 = x2 - 3.0             # set a value a bit smaller than x2
	y1 = theta_es(x1,prs)
        x=mask=nil  # dummy
        c=0
        loop do
	  if y1.is_a?(Numeric)
	    x = x1 + (x2-x1)/(y2-y1)*(th_es-y1)
	    y = theta_es(x,prs)
	    break if (y-th_es).abs < eps 
	  else
	    #p '!#!#',c,x2,x1,y2.ne(y1)
	    if c==0
	      x = x1 + (x2-x1)/(y2-y1)*(th_es-y1)
	    else
	      x = x.dup
	      x[mask] = ( x1 + (x2-x1)/(y2-y1)*(th_es-y1) )[mask]
	    end
	    y = theta_es(x,prs)
	    mask = ( (y-th_es).abs > eps )
	    if mask.none?
	      break
	    end
	  end
	  x2 = x1
	  y2 = y1
	  x1 = x
	  y1 = y
	  c += 1
	end
	x
      end

      def e_sat(tempC,prs)
	# calculates saturation water vapor pressure using enhanced 
        # Teten''s formula.  Based on thermolib3.f in APRS 4.5.1
	tempC = __ary_to_real_na(tempC)
	prs = __ary_to_real_na(prs)
	ice_mask = tempC < 0
	if tempC.is_a?(Numeric)
	  water_mask = !ice_mask
	else
	  water_mask = ice_mask.not
	  es = tempC.dup.fill!(0)  # initialization
	end
	if tempC.is_a?(Numeric) 
	  if water_mask
	    t = tempC
	    f = @@sfwa + @@sfwb * prs
	    es = f * @@sewa * exp( @@sewb*t/(t+T0-@@sewc) )
	  end
	elsif water_mask.any?
	  t = tempC[water_mask]
	  if prs.is_a?(Numeric)
	    f = @@sfwa + @@sfwb * prs
	  else
	    f = @@sfwa + @@sfwb * prs[water_mask]
	  end
	  es[water_mask] = f * @@sewa * exp( @@sewb*t/(t+T0-@@sewc) )
	end
	if tempC.is_a?(Numeric) 
	  if ice_mask
	    t = tempC
	    f = @@sfia + @@sfib * prs
	    es = f * @@seia * exp( @@seib*t/(t+T0-@@seic) )
	  end
	elsif ice_mask.any?
	  t = tempC[ice_mask]
	  if prs.is_a?(Numeric)
	    f = @@sfia + @@sfib * prs
	  else
	    f = @@sfia + @@sfib * prs[ice_mask]
	  end
	  es[ice_mask] = f * @@seia * exp( @@seib*t/(t+T0-@@seic) )
	end
	es
      end

      def r_sat(tempC,prs)
	es=e_sat(tempC,prs)
	prs = __ary_to_real_na(prs)
	rratio = R / Rv
	rsat = rratio * es / (prs-es)
      end

      def q_sat(tempC,prs)
	es=e_sat(tempC,prs)
	prs = __ary_to_real_na(prs)
	rratio = R / Rv
	rsat = rratio * es / (prs-(1-rratio)*es)
      end

      def rel_hum(tempC,e,prs)
	e = __ary_to_real_na(e)
	e/e_sat(tempC,prs)
      end

    end
  end
end


if __FILE__ == $0
  begin
    #require 'irb/xmp'   # very slow but can handle exceptions, comments, etc
    require 'xmp'       # fast but not good for debug
  rescue LoadError
    #require '../../../xmp/irb-xmp'  # copy of irb/xmp
    require '../../../xmp/xmp'      # copy of xmp
  end

  include NumRu::Met::Thermo

  xmp <<-EOS
  NumRu::Met::Thermo.constants.sort
  r2q(0.01)
  tempC = [20,10,0,-10,-20]
  prs = [1000,800,700,600,500]
  lat(tempC)
  r_sat(tempC,prs)
  rsat = r_sat(tempC,1000.0)
  t_dewpoint(rsat,1000.0)
  t_dewpoint(rsat/2,1000.0)
  t_dewpoint([0.003,0.005,0.01,0.02],1000.0)
  t_dewpoint(0.02,1000.0)
  th_es = theta_es([-10.0,10.0,30.0],1000.0)
  theta_es_prs2tempC(th_es,1000.0)
  EOS
end
