
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libdap, A C++ implementation of the OPeNDAP Data
// Access Protocol.

// Copyright (c) 2002,2003,2005 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//         Reza Nekovei <reza@intcomm.net>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.

// (c) COPYRIGHT URI/MIT 1994-1996
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      reza            Reza Nekovei (reza@intcomm.net)

// Implmentation for NCConnect: a subclass of Connect specialized for netcdf.
//
// jhrg 9/29/94

// See Trac tickets #50-53
// #define SCOPED_GRID_DIMENSIONS 1

#include "config_nc.h"

static char rcsid[] not_used = {"$Id: NCConnect.cc 11983 2005-08-18 21:16:31Z jimg $"};

#include "netcdf.h"

#include "DDS.h"
#include "AttrTable.h"
#include "InternalErr.h"
#include "debug.h"

#include "ClientParams.h"
#include "NCArray.h"
#include "NCStructure.h"
#include "NCSequence.h"
#include "NCGrid.h"
#include "NCConnect.h"
#include "nc_util.h"

using namespace std;

const string spr = "."; // structure rename

static inline bool
is_string(BaseType *btp)
{
    return (btp->type() == dods_str_c || btp->type() == dods_url_c);
}

static inline bool
is_array(BaseType *btp)
{
    return (btp->type() == dods_array_c);
}

static inline bool
is_grid(BaseType *btp)
{
    return (btp->type() == dods_grid_c);
}

static nc_type
get_attr_nc_type(AttrType attrT) throw(Error)
{
    switch (attrT) {
      case Attr_byte:
        return NC_BYTE;

      case Attr_int32:
      case Attr_uint32:
        return NC_LONG;

      case Attr_int16:
      case Attr_uint16:
        return NC_SHORT;

      case Attr_float32:
        return NC_FLOAT;

      case Attr_float64:
        return NC_DOUBLE;

      case Attr_string:
        return NC_CHAR;

     default:
        throw Error(NC_ENOTATT, "No such attribute");
    }
}

static int
compute_string_attr_length(AttrTable &attr, AttrTable::Attr_iter &p)
{
    int len = 0;

    // string lenght is the total number of bytes plus
    // terminating character. There is a '\n' at the end of each
    // string, but a '\0' at the end of the attibutes.
    unsigned int attr_siz = attr.get_attr_num(p);

    for (unsigned int i = 0; i < attr_siz; i++)
        len +=  attr.get_attr(p, i).length() - 1; // EOS included, quotes removed  
     
    // single string (coming from a netCDF server) remove EOS
    // for compatibility
    if (attr.get_attr_num(p) == 1)
       len -= 1;

    return len;
}

/** Initialize an instance. If \e name is a URL, the method 
    init_remote_source() must also be called.
    
    @param name A filename or URL. */
NCConnect::NCConnect(const string &name, const ClientParams &cp) 
    : AISConnect(name), d_client_params(cp), d_global_attributes(0),
      d_factory(0),
      d_ncid(-1), d_nvars(-1), d_recdim(-1), d_ndims(0)
{
    d_factory = new NCTypeFactory;
    d_constrained_dds.set_factory(d_factory);
    d_translated_dds.set_factory(d_factory);
}	

NCConnect::~NCConnect()
{
    delete d_global_attributes;
    delete d_factory;
}

/** Split a constraint expression into the projection and selection
    parts and store them in this object.
    @param ce The constraint expression. */
void
NCConnect::store_ce(const string &ce)
{
    if (ce.empty()) {
        d_proj_ce = d_sel_ce = "";
        return;
    }
    
    string::size_type amp = ce.find_first_of('&');
    if (amp == string::npos) {
        d_proj_ce = ce;
        d_sel_ce = "";
        return;
    }
    
    d_proj_ce = ce.substr(0, amp);
    d_sel_ce = ce.substr(amp);
}

/** Initialize a NCConnect instance when talking to a remote data source.

    @param ce Limit the DDS using this constraint expression.
    @exception Error Thrown by some of the NCConnect methods called here. 
 */
void
NCConnect::init_remote_source(const string &ce) throw(Error)
{
    // Get the DDS. If a CE was appended to the URL, use it.
    request_dds(d_constrained_dds, ce);

    // Record the ce for later use.
    store_ce(ce);
    
    // Get the DAS.
    DAS das;
    request_das(das);

    // Combine the Attributes with the variables in the constrained DDS.
    d_constrained_dds.transfer_attributes(&das);
    
    // Search for global attributes. For netCDF, the global attributes must
    // all be atomic; no containers within the global attr container. These
    // two lines of code transfer the global attributes from the constrained
    // DDS to the AttrTable held by this object and then flatten those
    // attributes. 
    set_global_attributes();
    AttrTable *at = flatten_attributes(d_global_attributes);
    delete d_global_attributes;
    d_global_attributes = at;
    
    // Mutate data types; from here on use the translated DDS! Note that
    // the variables' attributes are stored with the variables, so as they 
    // are translated the variable attributes will move to the 'new'
    // variables. Note that the global attributes have already been copied
    // from the constrained DDS to this (NCConnect) object.
    translate_dds();
    d_nvars = d_translated_dds.var_end() - d_translated_dds.var_begin();

    // Build info about dimensions.    
    d_ndims=0;
    parse_grid_dims(d_translated_dds);
    parse_array_dims(d_translated_dds);
#ifdef STRING_AS_ARRAY
    parse_string_dims(d_translated_dds);
#endif
    set_recdim(das); 
}


int
NCConnect::get_ncid()
{
    DBG(cerr << "Returning ncid: " << d_ncid << endl);
    return d_ncid;
}

void
NCConnect::set_ncid(int id)
{
    DBG(cerr << "Setting ncid: " << id << endl);
    d_ncid = id;
}

int
NCConnect::get_ndims()
{
    return d_ndims;
}

int
NCConnect::get_nvars()
{
    return d_nvars;
}

void
NCConnect::set_nvars(int n)
{
    d_nvars = n;
}  

// Returns the dimension ID for the unlimited dimension
int
NCConnect::recdim()
{
    return d_recdim;
}


// Returns the dimension size for the given dimension ID
int
NCConnect::dim_size(int dimid)
{
    return d_dim_size[dimid];
}


// Returns the dimension name for the given dimension ID
const string & 
NCConnect::dim_name(int dimid)
{
    return d_dim_name[dimid];
}

DDS &
NCConnect::get_constrained_dds()
{
    return d_constrained_dds;
}

DDS &
NCConnect::get_translated_dds()
{
    return d_translated_dds;
}

/** Scan a DAS object looking for information about a NetCDF unlimited
    dimension. The OPeNDAP netCDF server adds this information by creating a
    new global attribute container at the top level named "DODS_EXTRA." The
    unlimited dimension is the value of the String attribute
    "Unlimited_Dimension." 

    @param das Search this DAS object. */    
void
NCConnect::set_recdim(DAS &das)
{
    // d_ndims and d_dim_name must be defined
    AttrTable *attr = das.find_container("DODS_EXTRA");
    if (attr) {
    	string dim = attr->get_attr("Unlimited_Dimension");
    	DBG2(cerr << "Found an ulimitied dimension attribute: " << dim << endl);
    	for (int i = 0; i < d_ndims; i++) {
    	    // A String attribute may be quoted
    	    DBG2(cerr << "Examining: " << d_dim_name[i] << endl);
    	    if (d_dim_name[i] == dim 
    		|| dim.substr(1, d_dim_name[i].size()) == d_dim_name[i]) {
                DBG2(cerr << "Set record dim to: " << i << endl);
        	d_recdim = i;
        	return;
    	    }
    	}
    }
}

/** Get the variable from this instance's DDS which corresponds to the
    given variable id.

    @note This method and others should check for overflow and throw Error.

    @param varid The netCDF variable id.
    @return The BaseType pointer to the variable */
BaseType *
NCConnect::get_variable(int varid) throw(Error)
{
    if ((varid < 0) || (varid > get_nvars()))
        throw Error(NC_ENOTVAR, "Invalid variable id number.");

    return *(d_translated_dds.var_begin() + varid);
}

AttrTable &
NCConnect::get_attribute_table(int varid) throw(Error)
{
    // varid = -1 is used for global atrributes
    if ((varid+1 < 0) || (varid+1 > get_nvars())) 
        throw Error (NC_ENOTVAR, "No such variable.");;

    return (varid == -1) ?  get_global_attributes()
            : get_variable(varid)->get_attr_table();
}

/** return the number of attributes for variable \e varid. if \e varid is 
    the special value \c NC_GLOBAL, then return the number of attributes in
    the global attribute container.
    
    @param varid The netCDF file id of the variable
    @return The number of attribtues */
int
NCConnect::get_num_attr(int varid)
{
    return get_attribute_table(varid).get_size();
}

/** Read attribute values and load them in a char *. The caller is 
    responsible for freeing values unless an exception is thrown. Free the 
    values using delete[]. Be sure to read the note!
    
    @note Even though the attributes are held in the DAS object as character
    data, this function converts the values to the attribute's declared
    datatype. Don't be fooled by the char * return type; if the attributes
    are Int32s, that's the type of the return data.

    @param cdfid NetCDF file id
    @param varid NetCDF variable id
    @param name Name of the attribute.
    @param count Value-result parameter that holds the number of values. 
    @param datatype The datatype of the attribute; a value-result param.
    @return A pointer to the raw values. Delete using delete[].
    @exception Error Thrown if the values cannot be read or don't exist. 
*/
char *
NCConnect::get_raw_values(int varid, const char *name, size_t *count,
    nc_type *datatype) throw(Error)
{
    char *values = 0;
    try {
        AttrTable &attr = get_attribute_table(varid);
        AttrTable::Attr_iter p;
        AttrTable *dummy;
        attr.find(name, &dummy, &p);

        if (p == attr.attr_end())
            throw Error(NC_ENOTATT, "No such attribute");
            //return NC_ENOTATT;

        *datatype = get_attr_nc_type(attr.get_attr_type(p));
 
        if (*datatype == NC_CHAR)
            *count = compute_string_attr_length(attr, p);
        else
            *count = static_cast<size_t>(attr.get_attr_num(p));

        DBG2(cerr << "name: " << name << ", count: " << *count << ", dt: " 
            << *datatype << endl);

        if (*count == 0)
            throw Error(NC_NOERR, "Zero count.");

        values = new char [(*count + 1) * nctypelen(*datatype)];
        copy_attributes(attr, p, values);
    }
    catch (Error &e) {
        delete [] values; values = 0;
        throw e;
    }

    return values;
}
    
/** For the BaseType \e bt in \e cdfid, determine its type and dimension
    info.

    @note Currently only handles BaseTypes, Grid and Array. 
    @note If any of the value-result parameters are zero, no value will be
    returned. Thus the function will return only the type if \e ndimsp and \e
    dims are both zero. Also, in this latter case, cdfid will be ignored.
    
    @param cdfid netCDF library id.
    @param bt Pointer to the BaseType within a data source referenced by \e
    cdfid. 
    @param typep Value-result paramter; return the type of \e bt.
    @param ndimsp Value-result paramter; return the number of dimensions. For
    a scalar variable, *ndimsp is zero.
    @param dims Value-result paramter; return the size of each dimension. For
    a scalar variable, dims[0] is one. 
    @exception InternalErr Thrown if \e bt is a Structure, Sequence, or
    unknown type. */
void
NCConnect:: var_info(BaseType *bt, nc_type *typep, int *ndimsp, int dims[]) 
    throw(InternalErr) 
{
    if (typep)
        *typep = dynamic_cast<NCAccess&>(*bt).get_nc_type();

    // Is this an Array? If not, is it a Grid? If so, grab the array
    // part of the grid. If not set ar to null and let the rest of the method
    // handle things. This could be turned into an NCAccess method that's
    // specialized by NCArray and NCGrid. 4/6/05 jhrg
    Array *ar = dynamic_cast<Array*>(bt);
    if (!ar) {
        Grid *g = dynamic_cast<Grid*>(bt);
        ar = (g) ? dynamic_cast<Array*>(g->array_var()) : 0;
    }
    
    // Special case arrays; note that there are special tests for arrays 
    // of strings.
    if (ar) {
        // set the number of dimentions and the shape for this array
        if (dims) {
            int ii = 0;
            for (Array::Dim_iter d = ar->dim_begin(); d != ar->dim_end(); ++d) {
                // get dim. IDs for shape construction
                // match using dimension's name and size, 
                // if not available use dim. size only
                DBG2(cerr << "var_info: Looking at dimension: " << ii << endl);
                for (int jj=0; jj < d_ndims; jj++) {
                    if (ar->dimension_name(d).empty()) {
                        if (d_dim_size[jj] == ar->dimension_size(d))
                            dims[ii++] = jj;
                            break;
                    } 
                    else if ((d_dim_size[jj] == ar->dimension_size(d))
                             && (d_dim_name[jj] == ar->dimension_name(d))) {
                        dims[ii++] = jj;
                        break;       
                    }
                }
            }
#ifdef STRING_AS_ARRAY
            if (dynamic_cast<NCAccess&>(*ar).get_translated() && is_string(ar->var())) {
                for (int jj=0; jj < d_ndims; jj++) {
                    if (d_dim_name[jj] == ar->name() + "-chars") {
                        dims[ii++] = jj;
                        DBG2(cerr << "Scalar Dims: " << ar->name() << " " << jj << endl);
                        break;
                    }
                }
            }
#endif          
        }
     
        if (ndimsp) {
#ifdef STRING_AS_ARRAY
            // Why isn't ndimsp == ii??? jhrg 2/24/05
            if (dynamic_cast<NCAccess&>(*ar).get_translated()
                && is_string(ar->var())) {
               *ndimsp = ar->dimensions() + 1;
            }
            else {
                *ndimsp = ar->dimensions();
            }
#else
            *ndimsp = ar->dimensions();
#endif
        }
    }           // if (ar)
#ifdef STRING_AS_ARRAY
    else if (dynamic_cast<NCAccess&>(*bt).get_translated() 
             && is_string(bt)) {
        if (dims) {
            // It's a string, but not an array. Find the dims index, et c.
            for (int jj=0; jj < d_ndims; jj++) {
                if (d_dim_name[jj] == bt->name() + "-chars") {
                    dims[0] = jj;
                    DBG2(cerr << "Scalar Dims: " << bt->name() << " " << jj << endl);
                    break;
                }
            }
        }
        if (ndimsp)
            *ndimsp = 1;
    }
#endif
    else {
        //set default values for single value base types (non-array)
        if (dims)
            dims[0] = 1;
        if (ndimsp)
            *ndimsp = 0;
    }
}

const char *likely_global_attrs[]={"NC_GLOBAL", "HDF_GLOBAL", "FF_GLOBAL",
      "DSP_GLOBAL", "MAT_GLOBAL", "CoreMetadata", NULL};

/** Set the global attributes. This method assumes that 
    DDS::transfer_attributes() has already been called and that 
    DDS::get_attr_table() when envoked on d_constrained_dds will return
    a valid AttrTable. It scans that AttrTable for likely netCDF global
    attributes, looking for attribute containers with certain special names.
    If no 'special' containers are found, it punts and sets the NCConnect
    global attribute object to an empty table.
    
    @note This method could do more, such as scan for more than one likely
    container and merge them. It could also read the names of likely 
    containers from the .dodsrc file.

    @note This method uses the 'donstrained DDS' not the 'translated DDS'
    because it assumes that a previous call has transferred the attributes
    from a DAS into the constrained DDS (which is the DDS straing from the 
    server, with no translation applied). If we tried to merge the DAS info
    from the server with the translated DDS there's little chance it would
    work since the variable names are likely to be different in a translated
    DDS.
    
    @see likely_global_attrs */
void
NCConnect::set_global_attributes()
{    
    int i = 0;
    while (likely_global_attrs[i] && !d_global_attributes) {
        AttrTable *a = d_constrained_dds.get_attr_table().find_container(likely_global_attrs[i++]);
        // Choose this container only if it has elements. For some data sources
        // there is a container like HDF_GLOBAL, but it's empty and global
        // attributes are tucked away somewhere else (like CoreMetadata...).
        // Force a copy of the table's values into a newly allocated table.
        if (a && a->get_size() != 0)
            d_global_attributes = new AttrTable(*a);
    }
    
    // If there's nothing that matches, create an empty table.
    if (!d_global_attributes)
        d_global_attributes = new AttrTable;
}

/** Return the global attribute table for this data source.
    @return A reference to a container that holds the global attributes.
    @see set_global_attributes() */
AttrTable &
NCConnect::get_global_attributes()
{
    if (!d_global_attributes)
        set_global_attributes();
        
    return *d_global_attributes;
}

/** Remove attribute containers from the attribute table by 'flattening' 
    the table. An attribute inside a container is renamed 
    <container name>:<attribtue name>.
        
    @param src The attribute table to flatten. 
    @return A pointer to a new AttrTable. It is the caller's responsibility
    to delete this object when done using it. */
AttrTable *
NCConnect::flatten_attributes(AttrTable *src)
{
    EntryList *el = transfer_attributes_to_list(src);
    
    AttrTable *at = attributes_list_to_table(el);
    
    delete el; el = 0;
    
    return at;
}

static inline void
delete_basetype(BaseType *btp)
{
    delete btp;
    btp = 0;
}

/** This method translates the DDS held in the field d_constrained_dds and
    stores the result in d_translated_dds. It performs a deep copy of all
    elements in the process, even if no variables need to be translated.
    
    I think this could be improved upon, maybe by passing the d_translated_dds
    object to NCAccess::flatten() as a parameter. */
void
NCConnect::translate_dds()
{
    
    DDS::Vars_iter field = d_constrained_dds.var_begin();
    DDS::Vars_iter field_end = d_constrained_dds.var_end();
    VarList new_vars;       // Store new vars here

    while (field != field_end) {
        DBG2((*field)->print_decl(cerr, ""));
        
        VarList embedded_vars = dynamic_cast<NCAccess*>(*field)->flatten(d_client_params, string(""));
        new_vars.splice(new_vars.end(), embedded_vars);

        // Some of the flatten() methods bind 'translation' attributes to
        // 'field' BaseTypes as a way of sending information about the
        // translation process 'up the chain.'
        //
        // If one of the 'field' variables has a translation attribute, 
        // move that attribute to the global table. The code does this because
        // all of the field BaseTypes will be lost when the d_translated_dds
        // is populated using variables stored on the new_vars list.
        string trans = (*field)->get_attr_table().get_attr("translation");
        if (!trans.empty())
            d_global_attributes->append_attr("translation", "String", trans);
            
        ++field;
    }

    // Load new_vars into d_translated_dds
    VarListIter i = new_vars.begin();
    VarListIter end = new_vars.end();
    while (i != end) {
        d_translated_dds.add_var(*i);
        delete *i; *i++ = 0;
    }

    d_translated_dds.set_dataset_name(d_constrained_dds.get_dataset_name());
}

// Adds dimensions to the look up table for Grid class 
void
NCConnect::parse_grid_dims(DDS &dds)
{
    DDS::Vars_iter s = find_if(dds.var_begin(), dds.var_end(), is_grid);
    while (s != dds.var_end()) {
    	Grid *gr = dynamic_cast<Grid*>(*s);
    
    	for (Grid::Map_iter q = gr->map_begin(); q != gr->map_end(); ++q) {
    	    Array *ar = dynamic_cast<Array*>(*q);

#if defined(SCOPED_GRID_DIMENSIONS)    
            // Get dimension name; all maps are vectors in DAP 2
            string mapdimname = gr->name() + "." + ar->dimension_name(ar->dim_begin());
            int mapdimsize = ar->dimension_size(ar->dim_begin());
    
            if (mapdimname == "") // if dim. name is missing use var. name 
                    mapdimname = gr->name() + "." + ar->name();
#else
    	    // Get dimension name; all maps are vectors in DAP 2
    	    string mapdimname = ar->dimension_name(ar->dim_begin());
    	    int mapdimsize = ar->dimension_size(ar->dim_begin());
    
    	    if (mapdimname == "") // if dim. name is missing use var. name 
    		    mapdimname = ar->name();
#endif    
    	    // Find a match to the dimension name and size ?
    	    bool match_found = false;
    	    for (int j=0; j <d_ndims; j++) {
		if(d_dim_name[j] ==  mapdimname && d_dim_size[j] == mapdimsize) {
		    match_found = true;
		    break;
		}
    	    }
    
    	    // No match found, add it to the dimension tables
    	    if (!match_found) {
		d_dim_name[d_ndims] =  mapdimname;
		d_dim_size[d_ndims++] = mapdimsize;
    	    }
    	}

    	// Find the next grid
    	s = find_if(++s, dds.var_end(), is_grid);
    }
}

// Note: the members _ndims, d_dim_name and d_dim_size are first modified by
// NCConnect::parse_grid_dims(). The initial value of _ndims is zero when
// parse_grid_dims() Is called (see parse_dims() below).

// Adds dimensions to the look up table for Array class 
void
NCConnect::parse_array_dims(DDS &dds)
{
    DDS::Vars_iter s = find_if(dds.var_begin(), dds.var_end(), is_array);
    while (s != dds.var_end()) {
    	Array *ar = dynamic_cast<Array*>(*s);
        int dim_cnt = 0;
    
#ifdef STRING_AS_ARRAY
        // If this is an array of strings, add an extra dimension because
        // netCDF will store the string as an array of NC_CHARs.
        DBG2(cerr << "Name: " << ar->name() << " (" << ar << ") "
             << "Translated: " << dynamic_cast<NCAccess&>(*ar).get_translated()
             << endl);
        if (dynamic_cast<NCAccess&>(*ar).get_translated()
            && is_string(ar->var())) {
            d_dim_name[d_ndims + dim_cnt] = ar->name() + "-chars";
            d_dim_size[d_ndims + dim_cnt] = STRING_ARRAY_SIZE;
            dim_cnt++;
        }
#endif

    	// search all dimensions
    	for (Array::Dim_iter d = ar->dim_begin(); d != ar->dim_end(); ++d) {
    	    // get dimension name and size
    	    string dimname = ar->dimension_name(d);
    	    int dim_size = ar->dimension_size(d);
    
    	    // Look for a match to the dimension size (and name if it exists) ?
    	    bool match_found = false;
    	    for(int j = 0; j <d_ndims; j++) {
		if(d_dim_size[j] == dim_size
		   && (dimname == "" || d_dim_name[j] == dimname)) {
		    match_found = true;
		    break;
		}
    	    }

    	    // No match found, add it to the dimension tables and incr count
    	    if (!match_found) {
		if (dimname == "")
		    dimname = ar->name() + string("_") + long_to_string(dim_cnt);

		d_dim_name[d_ndims + dim_cnt] = dimname;
		d_dim_size[d_ndims + dim_cnt] = dim_size;

		dim_cnt++;
    	    }
    	}

    	d_ndims += dim_cnt; // All the dimensions in a variable must be diff.
    
    	// Find the next array
    	s = find_if(++s, dds.var_end(), is_array);
    }
}

// Since string dimensions are name <string-name>-chars, each unique. 
// So, there's no search for an existing dimension; we just append a new
// dimension for each string variable.
void
NCConnect::parse_string_dims(DDS &dds)
{
    DDS::Vars_iter s = find_if(dds.var_begin(), dds.var_end(), is_string);
    while (s != dds.var_end()) {
        if (dynamic_cast<NCAccess&>(**s).get_translated()) {
            string dim_name = (*s)->name() + "-chars";
            d_dim_name[d_ndims] = dim_name;
            d_dim_size[d_ndims] = STRING_ARRAY_SIZE;
            ++d_ndims;
        }
        
        s = find_if(++s, dds.var_end(), is_string);
    }
}

// $Log: NCConnect.cc,v $
// Revision 1.40  2005/06/03 16:30:14  jimg
// Added instrumentation using to find bugs found while working on bug 773.
//
// Revision 1.39  2005/04/11 18:38:20  jimg
// Fixed a problem with NCSequence where nested sequences were not flagged
// but instead were translated. The extract_values software cannot process a
// nested sequence yet. Now the code inserts an attribute that notes that a
// nested sequence has been elided.
//
// Revision 1.38  2005/04/08 17:08:47  jimg
// Removed old 'virtual ctor' functions which have now been replaced by the
// factory class code in libdap++.
//
// Revision 1.37  2005/04/07 23:32:38  jimg
// Modified the limits in netcdf.h (lnetcdf/lnetcdf.h and lnetcdf/netcdf.h)
// so that we can accommodate huge files.
//
// Revision 1.36  2005/04/06 20:50:07  jimg
// Fixed a problem reported by Kevin O'Brien where Grids were not processed
// correctly. The fix is in NCConnect::var_info(). The case where a netCDF array
// is an Opendap Grid was not handled.
//
// Revision 1.35  2005/03/31 00:04:51  jimg
// Modified to use the factory class in libdap++ 3.5.
//
// Revision 1.34  2005/03/23 19:50:11  jimg
// Removed the old implementation of flatten_attributes() and added a new one
// that uses the new functions in nc_util.cc.
//
// Revision 1.33  2005/03/19 00:33:03  jimg
// Checkpoint: All tests pass and only one memory leak remains (in
// NCConnect::flatten_attributes()).
//
// Revision 1.32  2005/03/04 18:10:49  jimg
// At this point valgrind runs the Unidata tests for both local and remote access
// and shows no errors or leaks. There are 8 bytes still reachable from an
// exception, but that's it.
//
// Revision 1.31  2005/03/02 17:51:50  jimg
// Considerable reduction in memory leaks and fixed all errant memory
// accesses found with nc_test. OPeNDAP error codes and Error object
// message strings are now reported using the nc_strerrror() function!
//
// Revision 1.30  2005/02/26 00:43:20  jimg
// Check point: This version of the CL can now translate strings from the
// server into char arrays. This is controlled by two things: First a
// compile-time directive STRING_AS_ARRAY can be used to remove/include
// this feature. When included in the code, only Strings associated with
// variables created by the translation process will be turned into char
// arrays. Other String variables are assumed to be single character strings
// (although there may be a bug with the way these are handled, see
// NCAccess::extract_values()).
//
// Revision 1.29  2005/02/17 23:44:13  jimg
// Modifications for processing of command line projections combined
// with the limit stuff and projection info passed in from the API. I also
// consolodated some of the code by moving d_source from various
// classes to NCAccess. This may it so that DODvario() could be simplified
// as could build_constraint() and store_projection() in NCArray.
//
// Revision 1.28  2005/01/29 00:20:29  jimg
// Checkpoint: CEs ont he command line/ncopen() almost work.
//
// Revision 1.27  2004/12/06 17:20:36  jimg
// Added store_ce() and fields to store the projection and selection parts of a CE.
//
// Revision 1.26  2004/11/30 22:11:35  jimg
// I replaced the flatten_*() functions with a flatten() method in
// NCAccess. The default version of this method is in NCAccess and works
// for the atomic types; constructors must provide a specialization.
// Then I removed the code that copied the variables from vectors to
// lists. The translation code in NCConnect was modified to use the
// new method.
//
// Revision 1.25  2004/11/05 17:13:57  jimg
// Added code to copy the BaseType pointers from the vector container into
// a list. This will enable more efficient translation software to be
// written.
//
// Revision 1.24  2004/10/28 16:38:19  jimg
// Added support for error handling to ClientParams. Added use of
// ClientParams to NCConnect, although that's not complete yet. NCConnect
// now has an instance of ClientParams. The instance is first built and
// then passed into NCConnect's ctor which stores a const reference to the CP
// object.
//
// Revision 1.23  2004/10/22 21:51:34  jimg
// More massive changes: Translation of Sequences now works so long as the
// Sequence contains only atomic types.
//
// Revision 1.22  2004/09/08 22:08:21  jimg
// More Massive changes: Code moved from the files that clone the netCDF
// function calls into NCConnect, NCAccess or nc_util.cc. Much of the
// translation functions are now methods. The netCDF type classes now
// inherit from NCAccess in addition to the DAP type classes.
//
// Revision 1.21  2004/08/03 23:23:13  jimg
// Vast changes to get attribute support working for variables from
// data sources with Structures. The Structure variables are 'translated'
// to sets of variables with 'dots in their names.' These new changes
// correctly find the attribtues for the old variables and add them to
// the built in attribute table objects for the new variables. Global
// attributes are now handled as well. This software relies heavily on the
// heuristic code in DDS::transfer_attributes(). During the course of the
// work, I moved all of the software that interacts with the OPeNDAP servers
// to NCConnect or nc_util. The files Dvar.cc, Dattr.cc, ..., are now
// just the clones of the netCDF api functions.
//
// Revision 1.20  2004/07/28 18:10:05  jimg
// Fixed handling of global attributes. I used the new code in
// DDS::transfer_attributes().
//
// Revision 1.19  2004/07/26 19:10:44  jimg
// Moved netCDF CL <--> OPeNDAP server interface code to nc_util and
// NCConnect.
//
// Revision 1.18  2004/03/09 22:56:32  jimg
// Refactored so that Pix is no longer used. Some common code (in the
// glue routines) was also factored out to functions. The netCDF 2
// interface is now supplied by the lnetcdf/lv2i.c file (which is a mostly
// unaltered copy of the original v2i.c source file). See lnetcdf/README.
//
// Revision 1.17  2004/03/08 19:08:33  jimg
// This version of the code uses the Unidata netCDF 3.5.1 version of the
// netCDF 2 API emulation. This functions call our netCDF 3 API functions
// which may either interact with a DAP server r call the local netCDF 3
// functions.
//
// Revision 1.16  2004/03/01 22:30:58  jimg
// Update; more fixes for translation of Structure types. This code can
// be built with ncdump and that client can read from a host of HDF4
// datasets. The changes that enable this are filtering of Sequences.
//
// Revision 1.15  2004/02/25 00:47:52  jimg
// This code will translate Structures, including ones that are nested.
// Not tested much; needs work.
//
// Revision 1.14  2004/02/19 19:42:55  jimg
// Merged with release-3-4-2FCS and resolved conflicts.
//
// Revision 1.13  2003/12/08 18:06:37  edavis
// Merge release-3-4 into trunk
//
// Revision 1.12  2003/09/30 22:32:26  jimg
// I removed old code and removed some methods from the compile. I also changed
// two of the casts fo BaseType* so that the Constructor::var(Pix&) methods are
// found (but this code should really be changed to use the new iterator
// interface).
//
// Revision 1.11  2003/09/29 12:11:13  reza
// Merged v3.2 translation code.
//
// Revision 1.10  2003/05/02 16:30:29  jimg
// Fixes for the builds
//
// Revision 1.9  2003/03/06 01:41:06  jimg
// Removed calls to get/set_accept_types().
//
// Revision 1.8  2003/01/28 07:08:24  jimg
// Merged with release-3-2-8.
//
// Revision 1.4.2.9  2002/12/27 00:34:51  jimg
// Added the set_ncid() method. I added this because I'm tracking down a bug and
// what to be sure that the code isn't modifying the internal value of _ncid
// somewhere (which it could do if it has access to a reference to the field).
//
// Revision 1.4.2.8  2002/12/27 00:05:36  jimg
// I changed a number of the accessors here to throw InternalErr instead of
// writing to stderr.
//
// Revision 1.4.2.7  2002/12/23 23:06:40  jimg
// Build Fixes.
//
// Revision 1.4.2.6  2002/09/08 23:58:35  rmorris
// Added a "return" statement in a function call.  VC++ requires.
//
// Revision 1.4.2.5  2002/06/27 21:00:26  jimg
// Fixed call to Connect::request_dds(...). The local method request_Cdds(...)
// uses Connect's new version of request_dds(...). This file was 'fixed' once to
// use my first try at a new interface for Connect. I then fixed Connect some
// more and broke this code again... I think the current interface supported by
// Connect is much better.
//
// Revision 1.4.2.4  2002/06/21 00:31:40  jimg
// I changed many files throughout the source so that the 'make World' build
// works with the new versions of Connect and libdap++ that use libcurl.
// Most of these changes are either to Makefiles, configure scripts or to
// the headers included by various C++ files. In a few places the oddities
// of libwww forced us to hack up code and I've undone those and some of the
// clients had code that supported libwww's generous tracing capabilities
// (that's one part of libwww I'll miss); I had to remove support for that.
// Once this code compiles and more work is done on Connect, I'll return to
// each of these changes and polish them.
//
// Revision 1.7  2002/05/03 00:01:52  jimg
// Merged with release-3-2-7.
//
// Revision 1.4.2.3  2001/12/26 03:44:04  rmorris
// Make it use cerr, endl from the std namespace under WIN32.
//
// Revision 1.6  2001/09/28 17:18:41  jimg
// Merged with 3.2.5.
// CVS  Committing in .
//
// Revision 1.4.2.2  2001/09/27 06:00:37  jimg
// Addd debug.h
//
// Revision 1.5  2001/08/30 23:08:24  jimg
// Merged with 3.2.4
//
// Revision 1.4.2.1  2001/08/30 06:23:03  reza
// Fixed bugs reported by bugzilla.
//
// Revision 1.4  2000/10/30 17:28:12  jimg
// Fixes for exception-based error reporting.
//
// Revision 1.3  2000/10/06 01:22:02  jimg
// Moved the CVS Log entries to the ends of files.
// Modified the read() methods to match the new definition in the dap library.
// Added exception handlers in various places to catch exceptions thrown
// by the dap library.
//
// Revision 1.2  1999/11/05 05:15:05  jimg
// Result of merge woth 3-1-0
//
// Revision 1.1.2.1  1999/10/29 05:05:21  jimg
// Reza's fixes plus the configure & Makefile update
//
// Revision 1.1  1999/07/28 00:22:42  jimg
// Added
//
// Revision 1.17  1999/05/07 23:45:31  jimg
// String --> string fixes
//
// Revision 1.16  1999/04/09 17:56:23  jimg
// Added call to set_accept_types telling servers that we don't like
// Sequences.
//
// Revision 1.15  1998/08/06 16:33:22  jimg
// Fixed misuse of the read(...) member function. Return true if more data
// is to be read, false is if not and error if an error is detected
//
// Revision 1.14  1998/02/05 20:14:46  jimg
// DODS now compiles with gcc 2.8.x
//
// Revision 1.13  1997/03/26 05:44:22  reza
// GLOBAL attribute can now be named anything with "_GLOBAL" in it (i.e.
// HDF_GLOBAL, MAT_GLOBAL, DSP_GLOBAL, etc.). This should improve
// interoperability with non-netCDF servers. Also, the DAS no longer requires
// attributes for all the variables. An internal empty attribute table
// (NC_BLANK) manages missing ones.
//
// Revision 1.12  1997/02/12 19:14:11  reza
// Upgraded for release 2.11.
//
// Revision 1.11  1996/11/13 22:23:07  jimg
// Added debuging to NCConnect dtor
//
// Revision 1.10  1996/09/17 17:06:22  jimg
// Merge the release-2-0 tagged files (which were off on a branch) back into
// the trunk revision.
//
// Revision 1.9.2.3  1996/09/17 00:26:21  jimg
// Merged changes from a side branch which contained various changes from
// Reza and Charles.
// Removed ncdump and netexec since ncdump is now in its own directory and
// netexec is no longer used.
//
// Revision 1.9.2.2  1996/07/10 21:44:00  jimg
// Changes for version 2.06. These fixed lingering problems from the migration
// from version 1.x to version 2.x.
// Removed some (but not all) warning generated with gcc's -Wall option.
//
// Revision 1.9.2.1  1996/06/25 22:04:23  jimg
// Version 2.0 from Reza.
//
// Revision 1.10  1995/11/18  11:33:22  reza
// Updated member function names for DAP-1.1.1.
//
// Revision 1.9  1995/11/10  06:31:22  reza
// Removed API name from connect.
//
// Revision 1.8  1995/07/09  21:33:42  jimg
// Added copyright notice.
//
// Revision 1.7  1994/12/22  04:51:42  reza
// Updated to use DODS new named dimension capability.
//
// Revision 1.6  1994/11/23  20:59:52  reza
// Added heuristic dimension matching for Grid and Array classes.
// Added Dimension handling routines (ndims(), dim_size(), dim_name(),
// parse_dim(),and etc.)
//
// Revision 1.5  1994/11/18  21:33:33  reza
// Added _nvars and nvars() for number of variables in the file.
// Added d_das_loc[] and parse d_das_loc() to make a look-up table for
// variables in dds vs. das.
//
// Revision 1.4  1994/11/03  05:45:07  reza
// Moved error, request_das, and request _dds to the surrogate library for more
// flexibility and simpler error handling.
//
// Revision 1.3  1994/10/28  19:09:03  jimg
// Added _error, error() and fixed ncid() so that it retruns a reference.
//
// Revision 1.2  1994/10/05  20:23:30  jimg
// Fixed errors in *.h files comments - CVS bites again.
// Changed request_{das,dds} so that they use the field `_api_name'
// instead of requiring callers to pass the api name.
//
// Revision 1.1  1994/10/05  18:02:14  jimg
// First version of the connection management classes.
// This commit also includes early versions of the test code.
