// (c) COPYRIGHT URI/MIT 1994-1996
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      reza            Reza Nekovei (reza@ieee.org)

// DODS-netCDF surrogate library functions for file handling. Note that with
// the preprocessor symbol LOCAL defined this library will access local files
// without using the DODS network services. Defining LOCAL will add support
// for local files through a slightly modifed version of the original netCDF
// library. This library has the same functionality as the original (it can
// access only files on a local hard disk) but has renamed entry points to
// avoid linker conflicts.
// 
// ReZa 10/13/94
//
// Rewrite to support netCDF version 3 interface; 5/12/99, Reza
//

#include "config_nc.h"

static char rcsid[] not_used ={"$Id: Dnc.cc,v 1.25 2005/06/03 16:34:27 jimg Exp $"};

#include <string>

#include "Connections.h"
#include "Connections.cc"

#include "ClientParams.h"
#include "NCStructure.h"
#include "NCConnect.h"

#include "Dnetcdf.h"
#include "nc_util.h"

// Make these symbols external so as to avoid link-time conflicts with their
// definitions in the localized version of the original
// implementation of netcdf.

#ifdef LOCAL
extern char *cdf_routine_name;
extern int ncopts;
extern int ncerr;
#else
char *cdf_routine_name;
int ncopts = NC_VERBOSE;
int ncerr;
#endif

/** Defined in lnetcdf/lerror. This is not included in lnetcdf.h because
    doing so would require all sources that use lnetcdf.h to also include
    Error.h 03/02/05 jhrg */
extern void set_opendap_error(const Error &);

int rcode;

Connections<NCConnect *> *conns = 0;

static void
delete_conns()
{
    delete conns; conns = 0;
}

int
nc__create_mp(const char * path, int ioflags, size_t initialsz, int basepe,
	size_t *chunksizehintp, int *ncid_ptr)
{
    ClientParams cp(path);
    NCConnect *c = new NCConnect(cp.url_without_params(), cp);

#ifdef LOCAL
    if(c->is_local()) {  
        rcode = lnc__create_mp(path, ioflags, initialsz, basepe,
			    chunksizehintp, ncid_ptr);
    	if (rcode == NC_NOERR) {
    	    c->set_ncid(*ncid_ptr);
    	    if (!conns) {
                DBG2(cerr << "Allocating conns" << endl);
    		conns = new Connections<NCConnect *>(MAX_NC_OPEN);
                atexit(delete_conns);
            }
	    DBG(cerr << "nc__create_mp: Adding object " << c 
		<< " to Connections." << endl);
	    DBG(cerr << "nc__create_mp: local ncid is " << *ncid_ptr << endl);
    	    *ncid_ptr = conns->add_connect(c);
	    DBG(cerr << "nc__create_mp: Conns id is " << *ncid_ptr << endl);
    	    return NC_NOERR;
    	}
    	else {
            delete c;
    	    return rcode;
    	}
    }
#endif

    delete c;
    return NC_EPERM;
}

int
nc__create(const char * path, int ioflags, size_t initialsz,
	size_t *chunksizehintp, int *ncid_ptr)
{
	return nc__create_mp(path, ioflags, initialsz, 0,
		chunksizehintp, ncid_ptr);
}


int
nc_create(const char * path, int ioflags, int *ncid_ptr)
{
        return nc__create(path, ioflags, 0, NULL, ncid_ptr);
}


int
nc__open(const char * path, int ioflags,
	size_t *chunksizehintp, int *ncid_ptr)
{
	return nc__open_mp(path, ioflags, 0,
		chunksizehintp, ncid_ptr);
}

int
nc__open_mp(const char * urlpath, int ioflags, int basepe,
	size_t *chunksizehintp, int *ncid_ptr)
{
    string path = urlpath;
    string var_name;

    size_t q_mark = path.find("?");
    string ce = "";
    if (q_mark != string::npos) {
	ce = path.substr(q_mark + 1);
	path = path.substr(0, q_mark);
    }

    ClientParams cp(path);
    NCConnect *c = new NCConnect(cp.url_without_params(), cp);

#ifdef LOCAL
    if (c->is_local()) {  
	try {
	    rcode = lnc__open_mp(path.c_str(), ioflags, basepe, 
				 chunksizehintp, ncid_ptr);
	}
	catch (Error &e) {
	    delete c; c = 0;
            set_opendap_error(e);
	    return rcode;
	}

	if (rcode == NC_NOERR) {
	    c->set_ncid(*ncid_ptr);
	    if (!conns) {
                DBG2(cerr << "Allocating conns" << endl);
		conns = new Connections<NCConnect *>(MAX_NC_OPEN);
                atexit(delete_conns);
	    }
	    DBG(cerr << "nc__open_mp (local): Adding object " << c 
		<< " to Connections." << endl);
	    DBG(cerr << "nc__open_mp (local): ncid is " << *ncid_ptr << endl);
    	    *ncid_ptr = conns->add_connect(c);
	    DBG(cerr << "nc__open_mp (local): Conns id is " << *ncid_ptr 
		<< endl);
	    return NC_NOERR;
	} 
	else {
	    delete c; c = 0;
	    return rcode;
	}
    }
#endif

    // !_CRAYMPP, only pe 0 is valid
    if (basepe != 0) {
	delete c; c = 0;
	return NC_EINVAL;
    }

    // Read only network access
    if (ioflags != NC_NOWRITE) {
	delete c; c = 0;
	return NC_EPERM;
    }

    try {
        c->init_remote_source(ce);
    }
    catch (Error &e) {
        DBG(cerr << "Caught error while initializing remote source:" << endl
            << e.get_error_message() << endl);
        set_opendap_error(e);
	delete c; c = 0;
	if(e.get_error_code() != NC_NOERR)
	    return e.get_error_code(); 
	else
	    return NC_EINVAL;
    }

    if (!conns) {
        DBG2(cerr << "Allocating conns" << endl);
	conns = new Connections<NCConnect *>(MAX_NC_OPEN);
        atexit(delete_conns);
    }
    
    DBG(cerr << "nc__open_mp (remote): Adding object " << c 
	<< " to Connections." << endl);
    *ncid_ptr = conns->add_connect(c);
    DBG(cerr << "nc__open_mp (remote): Conns id is " << *ncid_ptr << endl);

    return NC_NOERR;
}


int
nc_open(const char * path, int ioflags, int *ncid_ptr)
{
    return nc__open(path, ioflags, NULL, ncid_ptr);
}


int
nc_close(int cdfid)
{
    DBG(cerr << "Inside nc_close() with cdfid: " << cdfid << endl);

    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

    int rcode = NC_NOERR;
    
#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_close((*conns)[cdfid]->get_ncid());
    }
#endif

    DBG(cerr << "        number of conns on free list: " 
	<< conns->free_elements() << endl);

    delete (*conns)[cdfid];
    conns->del_connect(cdfid);
    // Let delete_conns(), which is registered using atexit(), 
    // take care of deleting conns itself.

    DBG(cerr << "        number of conns on free list after delete: " 
	<< conns->free_elements() << endl);

    return rcode;
}

int
nc_delete(const char * path)
{
	return nc_delete_mp(path, 0);
}

int
nc_delete_mp(const char * path, int basepe)
{
    ClientParams cp(path);
    NCConnect *c = new NCConnect(cp.url_without_params(), cp);

#ifdef LOCAL
    if(c->is_local()) {  
      rcode = lnc_delete_mp(path, basepe);
      return rcode;
    }
#endif

    return NC_EPERM;
}

int
nc_inq_base_pe(int cdfid, int *pe)
{
  if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
      || (*conns)[cdfid] == NULL) // does the file id exist?
    return NC_EBADID;
 
#ifdef LOCAL
  if((*conns)[cdfid]->is_local()) {
    rcode = lnc_inq_base_pe((*conns)[cdfid]->get_ncid(), pe);
    return rcode;
  }
#endif
  *pe = 0; /* !_CRAYMPP, only pe 0 is valid */
  return NC_NOERR;
}

int
nc_set_base_pe(int cdfid, int pe)
{
  if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
      || (*conns)[cdfid] == NULL) // does the file id exist?
    return NC_EBADID;
 
#ifdef LOCAL
  if((*conns)[cdfid]->is_local()) {
    rcode = lnc_set_base_pe((*conns)[cdfid]->get_ncid(), pe);
    return rcode;
  }
#endif
  return NC_EPERM;
}

int
nc_inq(int cdfid, int *ndimsp, int *nvarsp, int *nattsp, int *xtendimp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_inq((*conns)[cdfid]->get_ncid(), ndimsp, nvarsp, nattsp, xtendimp);
	return rcode;
    }
#endif

  if(nvarsp != NULL)
    *nvarsp = (*conns)[cdfid]->get_nvars();

  if(nattsp != NULL)
    *nattsp = (*conns)[cdfid]->get_num_attr(NC_GLOBAL);

  if(ndimsp != NULL)
    *ndimsp = (*conns)[cdfid]->get_ndims();
		
  if(xtendimp != NULL)
    *xtendimp = (*conns)[cdfid]->recdim(); 
	
  return NC_NOERR;
}


int 
nc_inq_ndims(int cdfid, int *ndimsp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_inq_ndims((*conns)[cdfid]->get_ncid(), ndimsp);
	return rcode;
    }
#endif

  if(ndimsp != NULL)
    *ndimsp = (*conns)[cdfid]->get_ndims();
		
  return NC_NOERR;
}


int 
nc_inq_nvars(int cdfid, int *nvarsp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_inq_nvars((*conns)[cdfid]->get_ncid(), nvarsp);
	return rcode;
    }
#endif

  if(nvarsp != NULL)
    *nvarsp = (*conns)[cdfid]->get_nvars();

  return NC_NOERR;
}


int 
nc_inq_natts(int cdfid, int *nattsp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_inq_natts((*conns)[cdfid]->get_ncid(), nattsp);
	return rcode;
    }
#endif

  if(nattsp != NULL)
    *nattsp = (*conns)[cdfid]->get_num_attr(NC_GLOBAL);

  return NC_NOERR;
}


int 
nc_inq_unlimdim(int cdfid, int *xtendimp)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
	rcode = lnc_inq_unlimdim((*conns)[cdfid]->get_ncid(), xtendimp);
	return rcode;
    }
#endif

  if(xtendimp != NULL)
    *xtendimp = (*conns)[cdfid]->recdim(); 
	
  return NC_NOERR;
}



int
nc_sync(int cdfid)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode = lnc_sync((*conns)[cdfid]->get_ncid());
      return rcode;
    }
#endif

    return NC_EPERM;
}

int
nc_abort(int cdfid)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode = lnc_abort((*conns)[cdfid]->get_ncid());
      return rcode;
    }
#endif
   rcode = nc_close(cdfid);
   return rcode;
}

/* 
 * Deprecated function ;
 */
int
ncnobuf(int cdfid)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

    cdf_routine_name = "ncnobuf" ;

    /* NOOP */

    return(0);
}


int
nc_redef(int cdfid)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode = lnc_redef((*conns)[cdfid]->get_ncid());
      return rcode;
    }
#endif

    return NC_EPERM;
}


int
nc__enddef(int cdfid, size_t h_minfree, size_t v_align,
        size_t v_minfree, size_t r_align)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode = lnc__enddef((*conns)[cdfid]->get_ncid(), h_minfree,
			  v_align, v_minfree, r_align);
      return rcode;
    }
#endif

    return NC_NOERR; // no effect on a remote file

}


int
nc_enddef(int cdfid)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode = lnc_enddef((*conns)[cdfid]->get_ncid());
      return rcode;
    }
#endif

    return NC_NOERR; // no effect on a remote file
}

int
nc_set_fill(int cdfid, int fillmode, int *old_mode_ptr)
{
    if (!conns || cdfid < 0 || cdfid > MAX_NC_OPEN 
	|| (*conns)[cdfid] == NULL) // does the file id exist?
	return NC_EBADID;

    if((*conns)[cdfid] == NULL) // does the file id exist?
      return NC_EBADID ;

#ifdef LOCAL
    if((*conns)[cdfid]->is_local()) {
      rcode = lnc_set_fill((*conns)[cdfid]->get_ncid(), fillmode, old_mode_ptr);
      return rcode;
    }
#endif

    return NC_EPERM;
}

const char *
nc_inq_libvers(void)
{
#ifdef LOCAL
  return lnc_inq_libvers(); // local netCDF library version (not DODS)
#else
  return ((char *)0);
#endif
}


const char *
nc_strerror(int err)
{
#ifdef LOCAL
  return lnc_strerror(err); 
#endif
  return((char *)0);
}

// $Log: Dnc.cc,v $
// Revision 1.25  2005/06/03 16:34:27  jimg
// Added instrumentation using to find bugs found while working on bug 773.
//
// Revision 1.24  2005/03/31 00:04:51  jimg
// Modified to use the factory class in libdap++ 3.5.
//
// Revision 1.23  2005/03/04 18:10:49  jimg
// At this point valgrind runs the Unidata tests for both local and remote access
// and shows no errors or leaks. There are 8 bytes still reachable from an
// exception, but that's it.
//
// Revision 1.22  2005/03/02 17:51:50  jimg
// Considerable reduction in memory leaks and fixed all errant memory
// accesses found with nc_test. OPeNDAP error codes and Error object
// message strings are now reported using the nc_strerrror() function!
//
// Revision 1.21  2004/10/28 16:38:19  jimg
// Added support for error handling to ClientParams. Added use of
// ClientParams to NCConnect, although that's not complete yet. NCConnect
// now has an instance of ClientParams. The instance is first built and
// then passed into NCConnect's ctor which stores a const reference to the CP
// object.
//
// Revision 1.20  2004/09/08 22:08:21  jimg
// More Massive changes: Code moved from the files that clone the netCDF
// function calls into NCConnect, NCAccess or nc_util.cc. Much of the
// translation functions are now methods. The netCDF type classes now
// inherit from NCAccess in addition to the DAP type classes.
//
// Revision 1.19  2004/08/03 23:23:13  jimg
// Vast changes to get attribute support working for variables from
// data sources with Structures. The Structure variables are 'translated'
// to sets of variables with 'dots in their names.' These new changes
// correctly find the attribtues for the old variables and add them to
// the built in attribute table objects for the new variables. Global
// attributes are now handled as well. This software relies heavily on the
// heuristic code in DDS::transfer_attributes(). During the course of the
// work, I moved all of the software that interacts with the OPeNDAP servers
// to NCConnect or nc_util. The files Dvar.cc, Dattr.cc, ..., are now
// just the clones of the netCDF api functions.
//
// Revision 1.18  2004/07/28 18:10:05  jimg
// Fixed handling of global attributes. I used the new code in
// DDS::transfer_attributes().
//
// Revision 1.17  2004/07/26 19:10:44  jimg
// Moved netCDF CL <--> OPeNDAP server interface code to nc_util and
// NCConnect.
//
// Revision 1.16  2004/03/09 22:56:32  jimg
// Refactored so that Pix is no longer used. Some common code (in the
// glue routines) was also factored out to functions. The netCDF 2
// interface is now supplied by the lnetcdf/lv2i.c file (which is a mostly
// unaltered copy of the original v2i.c source file). See lnetcdf/README.
//
// Revision 1.15  2004/03/08 19:08:33  jimg
// This version of the code uses the Unidata netCDF 3.5.1 version of the
// netCDF 2 API emulation. This functions call our netCDF 3 API functions
// which may either interact with a DAP server r call the local netCDF 3
// functions.
//
// Revision 1.14  2004/03/01 22:30:58  jimg
// Update; more fixes for translation of Structure types. This code can
// be built with ncdump and that client can read from a host of HDF4
// datasets. The changes that enable this are filtering of Sequences.
//
// Revision 1.13  2004/02/25 00:47:52  jimg
// This code will translate Structures, including ones that are nested.
// Not tested much; needs work.
//
// Revision 1.12  2004/02/19 19:42:55  jimg
// Merged with release-3-4-2FCS and resolved conflicts.
//
// Revision 1.11  2003/12/08 18:06:37  edavis
// Merge release-3-4 into trunk
//
// Revision 1.8.2.4  2003/10/09 05:59:03  reza
// Fixed for Fortran compiler.
//
// Revision 1.10  2003/09/29 12:11:13  reza
// Merged v3.2 translation code.
//
// Revision 1.9  2003/09/25 23:09:36  jimg
// Meerged from 3.4.1.
//
// Revision 1.8.2.3  2003/07/21 07:05:48  reza
// Added support for Cray multi-processor (CRAYMPP) interfaces, they use processor element 0 for all non-CrayMPP platforms. The Fortran interface needs them for a clean build.
//
// Revision 1.8.2.2  2003/06/06 08:23:41  reza
// Updated the servers to netCDF-3 and fixed error handling between client and server.
//
// Revision 1.8.2.1  2003/05/06 13:23:54  reza
// Fixed the netCDF V3.5 upgrade.
//
// Revision 1.8  2003/05/02 16:30:29  jimg
// Fixes for the builds
//
// Revision 1.7  2003/01/28 07:08:24  jimg
// Merged with release-3-2-8.
//
// Revision 1.4.4.9  2002/12/27 00:37:36  jimg
// Removed some unused variables.
// Switched to set_ncid() where appropriate.
//
// Revision 1.4.4.8  2002/12/23 23:06:40  jimg
// Build Fixes.
//
// Revision 1.4.4.7  2002/11/21 23:02:19  pwest
// fixed some memory leaks and memory read errors
//
// Revision 1.4.4.6  2002/11/06 22:53:23  pwest
// Cleaned up some uninitialized memory read warnings from purify
//
// Revision 1.6  2002/05/03 00:01:52  jimg
// Merged with release-3-2-7.
//
// Revision 1.4.4.5  2002/04/17 20:11:32  jimg
// Fixed the return value of ncopen() so that any failure results in a return
// value of -1. This fixes bug 393.
//
// Revision 1.4.4.4  2001/12/26 05:53:53  rmorris
// Force a return value in nc_strerror().  VC++ always requires a return;
//
// Revision 1.4.4.3  2001/12/26 03:39:05  rmorris
// nc_inq_libvers() must return _something_ under win32.  No return
// never allowed.
//
// Revision 1.5  2001/08/30 23:08:23  jimg
// Merged with 3.2.4
//
// Revision 1.4.4.2  2001/08/30 06:23:03  reza
// Fixed bugs reported by bugzilla.
//
// Revision 1.4.4.1  2001/05/17 16:33:57  reza
// Fixed exception handlers to return correct error codes.
//
// Revision 1.4  2000/10/06 01:22:02  jimg
// Moved the CVS Log entries to the ends of files.
// Modified the read() methods to match the new definition in the dap library.
// Added exception handlers in various places to catch exceptions thrown
// by the dap library.
//
// Revision 1.3  1999/11/05 05:15:04  jimg
// Result of merge woth 3-1-0
//
// Revision 1.1.2.2  1999/10/29 05:05:19  jimg
// Reza's fixes plus the configure & Makefile update
//
// Revision 1.2  1999/10/21 13:19:06  reza
// IMAP and other bug fixed for version3.
//
// Revision 1.1.2.1  1999/10/15 19:50:55  jimg
// Changed return values and conditions for NC API entry points
//
// Revision 1.1  1999/07/28 00:22:41  jimg
// Added
