=begin
=class NumRu::GDir

A directory service of GPhys.

A GDir object represents a directory, or a file (such as a NetCDF
file or GrADS control file), for which
GPhys objects can be defined for the variables in it.
This means that a NetCDF file, for example, is treated a
directory, rather than a plain file.

Suitable for remote service using dRuby.

==Class Methods

---GDir.top=(top)
    Sets the root directory. This should be done before making
    any GDir objects by ((<GDir.new>)). The default root directory
    is the root directory of the local file system ('/').

    ARGUMENTS
    * ((|top|)) (String): path of the top directory

    RETURN VALUE
    * absolute path of the top directory (String) (followed by a '/')

---GDir.top

    RETURN VALUE
    * absolute path of the top directory (String) (followed by a '/')

---GDir.new(path)
    Constructor.

    ARGUMENTS
    * ((|path|)) (String): path of the directory to open as a GDir. 
      The path is expressed in terms of the top directory.

    RETURN VALUE
    * a GDir

    ERRORS
    * ArgumentError if ((|path|)) is out of the directory 
      tree (originated at the top directory).

    EXAMPLES
    * If the top directory is "/hoge", you can open
      "/hoge/ho" directory by any of the following.

         gdir = GDir.new("/ho")
         gdir = GDir.new("./ho")

      If you want to open "/hoge" (the top directly), then

         gdir = GDir.new("/")
         gdir = GDir.new(".")

      To open a NetCDF file or GrADS control file,

         gdir = GDir.new("/ho/data.nc")
         gdir = GDir.new("/ho/data.ctl")

---GDir.set_text_pattern(*regexps)

    Sets regular expressions to match the file name of text files.
    The default is /\.txt$/ and /^\w*$/.

    ARGUMENTS
    * zero or more Regular expressions (zero means no file will be treated
      as a NetCDF file).

    RETURN VALUE
    * nil

    ERRORS
    * TypeError if any of the arguments is not a Regexp

---GFile.add_text_pattern(regexp [, regexp [, ...]])
    Similar to ((<GFile.set_text_pattern>)), but adds regular expressions
    instead of replacing existing ones.

    RETURN VALUE
    * nil

    ERRORS
    * TypeError if any of the arguments is not a Regexp

==Instance Methods

---path
    Returns the path (relative to the top directory)

    RETURN VALUE
    * a String

---name
    Name of the GDir

---inspect
    Returns the path

---list_dir(path=nil)
    Returns the names of the directories.

    ARGUMENTS
    * ((|path|)) (nil or String): if nil, the method is applied to the
      current directory. If specified, listing is made for the directory
      designated by it. Either relative or absolute.

    RETURN VALUE
    * an Array of String

---list_data(path=nil)
    Returns the names of the data (variables on which GPhys objects
    can be defined.) Returns a non-empty array if the GDir (current
    or at the path) is actually a file recognized by GPhys (i.e.,
    NetCDF or GrADS control file).

    ARGUMENTS
    * ((|path|)) (nil or String): if nil, the method is applied to the
      current directory. If specified, listing is made for the directory
      designated by it. Either relative or absolute.

    RETURN VALUE
    * an Array of String

---list_texts(path=nil)
    Returns the names of the text files.
    Whether a file is a text file or not is judged based on the
    name of the file: That matched the predefined patters is
    judged as a text file regardless whether it is really so.
    The default pattern is /\.txt$/ and /^\w*$/. The patterns
    can be customized by ((<set_text_patterns>)) or 
    ((<add_text_patterns>)).

    ARGUMENTS
    * ((|path|)) (nil or String): if nil, the method is applied to the
      current directory. If specified, listing is made for the directory
      designated by it. Either relative or absolute.

    RETURN VALUE
    * an Array of String

---dir(path)
    Returns a GDir specified the ((|path|)).

    ARGUMENTS
    * ((|path|)) (String): Either relative or absolute.

    RETURN VALUE
    * a GDir

    ERROR
    * an exception is raised if ((|path|)) is invalid as a GDir.

---data(path)
    Returns a GPhys specified the ((|path|)).

    ARGUMENTS
    * ((|path|)) (String): Either relative or absolute.

    RETURN VALUE
    * a GPhys

    ERROR
    * an exception is raised if ((|path|)) is invalid as a GPhys.

---text(path)
    Returns a text file object specified the ((|path|)).

    ARGUMENTS
    * ((|path|)) (String): Either relative or absolute.

    RETURN VALUE
    * a File

    ERROR
    * an exception is raised if ((|path|)) is invalid as a text file
      (based on the pattern: see above).

---[](path)
    Returns a GDir, GPhys, or File (text), by using
    ((<dir>)), ((<data>)), or ((<text>)).

    ARGUMENTS
    * ((|path|)) (String): Either relative or absolute.

    RETURN VALUE
    * a GDir, GPhys, or File (text assumed)

    ERROR
    * an exception is raised if ((|path|)) is not appropriate.

=end

require "numru/gphys"    # used in GFile

module NumRu

  class GDir

    @@top = '/'        # default
    @@top_pat =  Regexp.new('^'+@@top)

    @@text_pattern = [/\.txt$/,/^\w*$/]

    @@fileclass = {GPhys::IO::NETCDF => NetCDF, 
                   GPhys::IO::GRADS => GrADS_Gridded}

    def GDir.top=(top)
      top = File.expand_path( top )   # to absolute path
      @@top_pat =  Regexp.new('^'+top)
      @@top = top + '/'
    end

    def GDir.top; @@top; end

    def initialize(path)
      path = '/' + path if /^\// !~ path      # to always start with '/'
      path = File.expand_path( path )
      abspath = @@top.sub(/\/$/,'')+path.sub(/\/$/,'')
      @path = abspath.sub(@@top_pat,'') + '/'
      @abspath = abspath + '/'
      ftype = __ftype(abspath)
      case ftype
      when "directory"
	@dir = Dir.new(abspath)
	@file = nil
      when "file"
	if GPhys::IO::file2type( abspath )
	  # non-nil means GPhys knows it.
	  @dir = nil
	  @file = @abspath.sub(/\/$/,'')
	else
	  raise path+" are not dealt with GPhys." +
	        "Use GPhys::IO::add_(nc|gr)_pattern if it should be."
	end
      else
	raise ArgumentError, abspath+": Unsupported file type "+ftype
      end
    end

    def [](path)
      begin
	begin
	  dir(path)
	rescue
	  begin
	    data(path)
	  rescue
	    text(path)
	  end
	end
      rescue
	raise @path+path+': no such data, directory or text (or is outside the tree)'
      end
    end

    def dir(path)
      case path
      when /^\//
	# absolute path
	GDir.new(path)
      else
	# relative path
	GDir.new(@path+path)
      end
    end

    def data(path)
      dirname = File.dirname(path)
      if dirname != '.'
	self.dir(dirname).data( File.basename(path) )
      else
	# path is for the current directory.
	if @dir
	  raise "data #{path} is not found"   # cannot be directly under a Dir
	else
	  GPhys::IO::open(@file,path)
	end
      end
    end

    def text(path)
      dirname = File.dirname(path)
      if dirname != '.'
	self.dir(dirname).text( File.basename(path) )
      else
	# path is for the current directory.
	if @dir
	  if __ftype(@abspath+path) == "file"
	    case path
	    when *@@text_pattern
	      File.open(@abspath+path)
	    else
	      raise "#{path} does not match patterns to find text files"
	    end
	  else
	    raise "text #{path} is not a plain file"
	  end
	else
	  raise "text #{path} is not found"
	end
      end
    end

    def list_dirs(path = nil)
      if path
	self.dir(path).list_dirs
      else
	if @dir
	  gdir_names = []
	  @dir.rewind
	  @dir.each{|fnm|
	    if fnm != '.' && fnm != '..' && __can_be_a_gdir(@abspath+fnm)
	      gdir_names.push( fnm )
	    end
	  }
	  gdir_names
	else
	  []
	end
      end
    end

    def list_data(path = nil)
      if path
	self.dir(path).list_data
      else
	if @file
	  file = @@fileclass[GPhys::IO::file2type(@file)].open(@file)
	  var_names = file.var_names
	  file.close
	  var_names
	else
	  []
	end
      end
    end

    def list_texts(path = nil)
      if path
	self.dir(path).list_texts
      else
	if @dir
	  text_files = []
	  @dir.rewind
	  @dir.each{|fnm|
	    if __ftype(@abspath+fnm) == "file"
	      case fnm
	      when *@@text_pattern
		text_files.push( fnm )
	      end
	    end
	  }
	  text_files
	else
	  []
	end
      end
    end

    def path
      @path
    end
    def name
      @path=='/' ? @path : File.basename( @path.sub(/\/$/,'') )
    end

    def inspect
      path
    end

    def GDir.add_text_pattern(*regexps)
      regexps.each{ |regexp|
	raise TypeError,"Regexp expected" unless Regexp===regexp
	@@text_pattern.push(regexp)
      }
      nil
    end

    def GDir.set_text_pattern(*regexps)
      regexps.each{ |regexp|
	raise TypeError,"Regexp expected" unless Regexp===regexp
      }
      @@text_pattern = regexps
      nil
    end

    #################################
    private

    def __ftype(path, _depth=0)
      # Same as File.ftype, but traces symbolic links
      ftype = File.ftype(path)
      if ftype == "link"
	if _depth < 10   # == max_depth
          begin
	    lkpath = File.readlink(path)
	    unless /^\// === lkpath
	      lkpath = File.dirname(path)+'/'+lkpath
	    end
	    ftype = __ftype( lkpath, _depth+1 )
	  rescue
	    nil    # possibly a link to non-existent file
	  end
	else
	  nil    # Max depth exeeded: Link too long (possibly circular)
	end
      end
      ftype
    end

    def __can_be_a_gdir(path)
      ftype = __ftype(path)
      if ftype=="directory" || ftype=="file" && GPhys::IO::file2type(path)
	true
      else
	false
      end
    end

  end      # class GDir
end      # module NumRu


###############################################
# test program usage
# % ruby gdir.rb [topdir]
# EXAMPLE
# % ruby gdir.rb
# % ruby gdir.rb  ..

if __FILE__ == $0

  include NumRu

  def find_gphys(gdir)
    gphyses = []
    subdirs = gdir.list_dirs.collect{ |dnm| gdir[dnm] }
    subdirs.each{ |subdir|
      ary = find_gphys(subdir)
      gphyses +=  ary  if ary.length != 0
    }
    gdir.list_data.each{|var|
      gp = gdir.data(var)
      gphyses.push( gp ) 
    }
    gphyses
  end

  def find_txt(gdir)
    texts = []
    subdirs = gdir.list_dirs.collect{ |dnm| gdir[dnm] }
    subdirs.each{ |subdir|
      ary = find_txt(subdir)
      texts +=  ary  if ary.length != 0
    }
    gdir.list_texts.each{|var|
      gp = gdir.text(var)
      texts.push( gp ) 
    }
    texts
  end

  top = ARGV.shift || '.'
  GDir.top = top

  gdir = GDir.new('/')
  #gdir = GDir.new('.')   # same as above

  print "*(test)* Set the top directory to #{top}\n"

  p gdir.list_dirs
  p gdir.list_texts
  p gdir.list_data
  print gdir.path, ' ', gdir.name, "\n"
  print gdir[gdir.list_dirs[0]].path, ' ', gdir[gdir.list_dirs[0]].name, "\n"
  p gdir[gdir.list_dirs[0]]['.']['..']['..']

  print "*(test)* Searching all NetCDF&GrADS files under the top dir...\n"
  gphyses = find_gphys(gdir)
  gphyses.each{ |gp|
    print gp.name, gp.shape_current.inspect, '  ', gp.data.file.path,"\n"
  }

  print "*(test)* Searching all text files under the top dir...\n"
  find_txt(gdir).each{|txt| print txt.path,"\n"}

end
