!******************************************************************************
!  2003-10-01
!    const_set subroutine is added. 
!------------------------------------------------------------------------------
!  2003-09-07
!******************************************************************************

  module const_module

    use vtype_module

    implicit none

    private

    ! pi
    ! pix2     : 2 * pi
    !
    ! pi2      : same as pix2 (this is remained for the purpose of compativility)
    !
    ! d2r      : pi / 180
    ! r2d      : 180 / pi
    ! c        : speed of light (m/s)
    ! amu      : atomic mass unit
    ! planc    : planck constant
    ! boltz    : boltzmann constant
    ! ugasc    : universal gas constant
    ! gconst   : gravitational constant
    !
    real(dp), parameter :: &
         pi     = 3.1415926535897932385d0, &
         pix2   = 2.0d0 * pi             , &
         pi2    = pix2                   , &
         d2r    = pi / 180.0d0           , &
         r2d    = 180.0d0 / pi           , &
         c      = 2.99792458d8           , &
         amu    = 1.6605655d-27          , &
         planc  = 6.6260755d-34          , &
         boltz  = 1.380658d-23           , &
         ugasc  = 8.314510d0             , &
         gconst = 6.67260d-11


    !
    ! sboltz   : Stefan-Boltzmann constant
    !
    real(dp), parameter :: sboltz &
         = 2.0d0*pi*pi*pi*pi*pi*boltz*boltz*boltz*boltz &
         /(15.0d0*c*c*planc*planc*planc)

    !
    ! dayearth  : 1 day of the Earth (sec)
    !
    real(dp), parameter :: dayearth = 60.0d0 * 60.0d0 * 24.0d0


    !**************************************************************************

    !
    ! lhh2o0    : latent heat of condensation at 0 degree C (J/kg)
    !
    real(dp), parameter :: lhh2o0  = 2.5d6


    ! lhco2     : Latent heat of CO2 (J kg^{-1})
    !           : This value is obtained from Pollack et al. [1981] and 
    !           : Forget et al. [1998]. 
    ! co2lh     : = lhco2
    !
    real(dp), parameter :: lhco2 = 5.9d5
    real(dp), parameter :: co2lh = lhco2

    !**************************************************************************


    logical, save :: sw_const_set

    data sw_const_set / .false. /


    !
    ! pradi    : planetary radius (m)
    ! pradisq  : planetary radius squared (m^2)
    !
    ! pradi2   : same as pradisq (this is remained for the purpose of 
    !          : compativility)
    !
    ! grav     : gravitational acceleration (m/s^2)
    !
    real(dp), save :: pradi
    real(dp), save :: pradisq
    real(dp), save :: pradi2
    real(dp), save :: grav


    ! hour     : one solar hour (sec)
    ! day      : one solar day (sec)
    ! omega    : planetary angular velocity (rad/sec)
    !
    real(dp), save :: hour
    real(dp), save :: day
    real(dp), save :: omega

    ! cp       : specific heat at constant pressure
    !
    real(dp), save :: cp


    data pradi     / 1.0d100 /
    data pradisq   / 1.0d100 /
    data pradi2    / 1.0d100 /
    data grav      / 1.0d100 /
    data hour      / 1.0d100 /
    data day       / 1.0d100 /
    data omega     / 1.0d100 /

    data cp        / 1.0d100 /


    interface const_set
       module procedure const_set_id, const_set_name
    end interface


    !**************************************************************************

    public :: pi, pix2, pi2, d2r, r2d, c, amu, planc, boltz, ugasc, sboltz
    public :: pradi, pradisq, pradi2, grav, hour, day, omega
    public :: lhh2o0
    public :: lhco2, co2lh

!    public :: cp

    public :: const_set, const_set_check

    !**************************************************************************

  contains

    !**************************************************************************

    function const_set_check() result( sw )

      logical :: sw

      sw = sw_const_set

    end function const_set_check

    !**************************************************************************
    ! planetid  : Planet ID
    !           : 0  : the Earth
    !           : 1  : Mars
    !**************************************************************************

    subroutine const_set_id( planetid )

      integer(i4b),  intent(in) :: planetid


      select case( planetid )
      case( 0 )

         !
         ! for the Earth
         !
         pradi = 6371.0d3
         grav  = 9.81d0
!         omega = pix2 / dayearth
         omega = 7.292d-5
         day   = 1.0d0 * dayearth
         hour  = day / 24.0d0

      case( 1 )

         !
         ! for Mars
         !
         pradi = 3382.946d3
         omega = pix2 / ( 22.663d0 + 60.0d0 * ( 37.0d0 + 60.0d0 * 24.0d0 ) )
         day   = 1.0274912d0 * dayearth
         hour  = day / 24.0d0
         grav  = 3.72d0

      case default

         write( 6, * ) "Unexpected planet ID, ", planetid

      end select

      pradisq = pradi * pradi
      pradi2  = pradisq

!      omega=0.0d0


      sw_const_set = .true. 


    end subroutine const_set_id

    !**************************************************************************

    subroutine const_set_name( planetname )

      character(len=*), intent(in) :: planetname


      !
      ! local variables
      !
      integer(i4b) :: planetid


      if( ( planetname .eq. 'Earth' ) &
           .or. ( planetname .eq. 'EARTH' ) ) then
         planetid = 0
      else if( ( planetname .eq. 'Mars' ) &
           .or. ( planetname .eq. 'MARS' ) ) then
         planetid = 1
      else
         write( 6, * ) 'Unexpected planet name : ', trim( planetname ), '.'
         stop
      end if


      call const_set_id( planetid )


    end subroutine const_set_name

    !**************************************************************************

  end module const_module
