!= Sample program for gtool_history/gtool5
!
! * 2009/10/18 Y.Morikawa
! * 2003/08/21 M.Odaka
! * 2001/02/27 S.Takehiro
!
! Solving diffusion equation
! \[
!     du/dt = \kappa d^2 u/dx^2
! \]
! for giving values of $u$ at $x=[0,1]$.
!
program diffusion_cal1

  use gtool_history                                 ! Access module (⥸塼)
  use dc_calendar                               ! Access module (⥸塼)
  use dc_types                                      ! Access module (⥸塼)

  integer, parameter     :: nx=30                   ! Grid number (åɿ)
  real(DP), parameter    :: dx=1.0/(nx-1)           ! Grid interval (åɴֳ)
  real(DP), parameter    :: dt=1.                   ! Time step [s] (ֳִ [])
  type(DC_CAL_DATE)    :: start_date          ! Start Date (ǥ볫)
  type(DC_CAL_DATE)    :: end_date            ! End Date (ǥ뽪λ)
  real(DP)             :: curtime             ! Current time [sec] (߻ [])
  real(DP)             :: endtime             ! End time [sec] (λ [])
  real(DP)             :: dsptime             ! Output interval [min] (ϴֳ [ʬ])
  real(DP), dimension(nx):: x=(/(dx*(i-1),i=1,nx)/) ! X coordinate (ɸѿ)
  real(DP), dimension(nx):: temp                    ! Temperature ()
  real(DP), parameter    :: kappa=0.0001            ! Diffusion coefficient (ǮȻ)


  character(256):: date_str

  call DCCalCreate( cal_type = 'Gregorian' )        ! Set calendar
                                                    ! ()
  call DCCalDateCreate( year = 2009, month =  2, day = 28, &
    &                   hour =   23, min   = 59, sec = 0.0_DP, &
    &                   date = start_date )
                                                    ! Set date
                                                    ! ()
  curtime = DCCalConvertToSec(  0.0_DP, 'sec' )
                                                    ! Set initial time 
                                                    ! ()
  endtime = DCCalConvertToSec( 10.0_DP, 'min' )
                                                    ! Set end time 
                                                    ! (λ)
  dsptime = 1.0_DP
                                                    ! Set output interval
                                                    ! (ϴֳ)

  temp = exp(-((x-0.5)/0.1)**2)                     ! Set initial value 
                                                    ! ()

  call HistoryCreate( &                             ! Create output file 
    & file='diffusion_cal1.nc', &                   ! (ҥȥ꡼)
    & title='Diffusion equation',                        &
    & source='Sample program of gtool_history/gtool5',   &
    & institution='GFD_Dennou Club davis project',       &
    & dims=(/'x','t'/), dimsizes=(/nx,0/),               &
    & longnames=(/'X-coordinate','time        '/),       &
    & units=(/'m  ','min'/),                             &
    & origind=curtime, intervald=dsptime )

  call HistoryPut('x',x)                            ! Output 'x' (ѿ)

  call HistoryAddVariable( &                        ! Set output variable 
    & varname='temp', dims=(/'x','t'/), &           ! (ѿ) 
    & longname='temperature', units='K', xtype='double')

  call HistoryAddAttr('temp','gt_graph_tick_all',1)
  call HistoryAddAttr('temp','gt_graph_contour_spacing',(/0.0,1.0,0.01/))
  call HistoryAddAttr('temp','+gt_user_davis_kappa',kappa)

  call HistoryPut('temp',temp)                      ! Output 'temp' (ѿ)

  call DCCalDateInquire( date_str, date = start_date )
                                        ! Inquire date and set a character variable
                                        ! (䤤碌ʸѿ)

  call HistoryAddAttr( 't', 'units', 'min since ' // trim(date_str) )
                                                    ! Output start date to a file
                                                    ! (Υեؤν) 

  write(6,*) "Start: ", trim(date_str)              ! Display start date
                                                    ! (ɽ) 

  do while ( curtime < endtime )                    ! Check termination (λȽ)

    temp(2:nx-1) = temp(2:nx-1) &                   ! Time integration (ʬ)
      & + kappa*(temp(3:nx)-2*temp(2:nx-1)+temp(1:nx-2))/dx**2*dt

    if ( mod( curtime, dsptime ) == 0.0_DP ) then   ! Check output timing (ϥߥȽ)
      call HistoryPut('temp',temp)                  ! Output 'temp' (ѿ)
    endif

    curtime = curtime + dt                          ! Progress model time (ǥʹ)
  enddo


  call DCCalDateInquire( date_str, curtime, start_date )
                                        ! Inquire date and set a character variable
                                        ! (䤤碌ʸѿ)

  write(6,*) "End:   ", trim(date_str)              ! Display end date (λ) 

  call HistoryClose
  stop
end program diffusion_cal1
