module fftsub_mod_isp

contains

subroutine rearrange_rxy2ryx_isp( ix, jy, rvalxy, rvalyx )
!-- rearranging real(nx,ny) array to real(ny,nx)
  implicit none
  integer, intent(in) :: ix
  integer, intent(in) :: jy
  double precision, intent(in) :: rvalxy(ix,jy)
  double precision, intent(inout) :: rvalyx(jy,ix)
  integer :: j

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j)

  do j=1,jy
     rvalyx(j,1:ix)=rvalxy(1:ix,j)
  end do

!$omp end do
!$omp end parallel

end subroutine rearrange_rxy2ryx_isp


subroutine rearrange_ryx2rxy_isp( ix, jy, rvalyx, rvalxy )
!-- rearranging real(ny,nx) array to real(nx,ny)
  implicit none
  integer, intent(in) :: ix
  integer, intent(in) :: jy
  double precision, intent(in) :: rvalyx(jy,ix)
  double precision, intent(inout) :: rvalxy(ix,jy)
  integer :: j

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j)

  do j=1,jy
     rvalxy(1:ix,j)=rvalyx(j,1:ix)
  end do

!$omp end do
!$omp end parallel

end subroutine rearrange_ryx2rxy_isp


subroutine rearrange_ryx2cxy_isp( ix, jy, rvalyx, cvalxy )
!-- rearranging real(-ny:ny,-nx:nx) array 
!--          to complex(1:2*nx+1,1:2*ny+1)
  use Math_Const
  implicit none
  integer, intent(in) :: ix
  integer, intent(in) :: jy
  double precision, intent(in) :: rvalyx(-jy:jy,-ix:ix)
  complex(kind(0d0)), intent(inout) :: cvalxy(1:2*ix+1,1:2*jy+1)
  integer :: i, j

  cvalxy(1,1)=rvalyx(0,0)

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=1,ix
     cvalxy(i+1,1)=rvalyx(0,i)+img_cdp*rvalyx(0,-i)
     cvalxy(2*ix-i+2,1)=dconjg(cvalxy(i+1,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=1,jy
     cvalxy(1,j+1)=rvalyx(j,0)+img_cdp*rvalyx(-j,0)
     cvalxy(1,2*jy-j+2)=dconjg(cvalxy(1,j+1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=1,jy
     do i=1,ix
        cvalxy(i+1,j+1)=rvalyx(j,i)+img_cdp*rvalyx(-j,-i)
        cvalxy(2*ix-i+2,2*jy-j+2)=dconjg(cvalxy(i+1,j+1))
        cvalxy(i+1,2*jy-j+2)=rvalyx(-j,i)+img_cdp*rvalyx(j,-i)
        cvalxy(2*ix-i+2,j+1)=dconjg(cvalxy(i+1,2*jy-j+2))
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine rearrange_ryx2cxy_isp


subroutine rearrange_cxy2ryx_isp( ix, jy, cvalxy, rvalyx )
!-- rearranging complex(1:2*nx+1,1:2*ny+1) array 
!--          to real(-ny:ny,-nx:nx)
  implicit none
  integer, intent(in) :: ix
  integer, intent(in) :: jy
  complex(kind(0d0)), intent(in) :: cvalxy(1:2*ix+1,1:2*jy+1)
  double precision, intent(inout) :: rvalyx(-jy:jy,-ix:ix)
  integer :: i, j

  rvalyx(0,0)=cvalxy(1,1)

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

  do i=1,ix
     rvalyx(0,i)=dble(cvalxy(i+1,1))
     rvalyx(0,-i)=dimag(cvalxy(i+1,1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(j)

  do j=1,jy
     rvalyx(j,0)=dble(cvalxy(1,j+1))
     rvalyx(-j,0)=dimag(cvalxy(1,j+1))
  end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j)

  do j=1,jy
     do i=1,ix
        rvalyx(j,i)=dble(cvalxy(i+1,j+1))
        rvalyx(-j,-i)=dimag(cvalxy(i+1,j+1))
        rvalyx(-j,i)=dble(cvalxy(i+1,2*jy-j+2))
        rvalyx(j,-i)=dimag(cvalxy(i+1,2*jy-j+2))
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine rearrange_cxy2ryx_isp


subroutine make_restart_isp( itn, rtn, psik, zk, zopt )
!-- output restart file
  use fft_saveval_define
  use gtool_history
  implicit none
  integer, intent(in) :: itn
  real, intent(in) :: rtn
  complex(kind(0d0)), dimension(:,:), intent(in) :: psik
  complex(kind(0d0)), dimension(size(psik,1),size(psik,2)), intent(in) :: zk
  complex(kind(0d0)), dimension(size(psik,1),size(psik,2)), optional, intent(in) :: zopt
  double precision, dimension(size(psik,1),size(psik,2)) :: tmpd
  type(GT_HISTORY) :: res_hst
  integer :: ix, jy, ii, jj

  ix=size(psik,1)
  jy=size(psik,2)

  call HistoryCreate( file=trim(adjustl(resfname)),  &
  &    title='BAROTRO result data', &
  &    source='test', institution='test', dims=(/'x', 'y'/),  &
  &    dimsizes=(/ ix, jy /),  & 
  &    longnames=(/'X-coordinate','Y-coordinate'/),  &
  &    units=(/'m', 'm'/), history=res_hst )

  call HistoryAddVariable( varname=trim(adjustl(respsir)), dims=(/'x','y'/), &
  &                        longname='stream line function',  &
  &                        units='m2 s-1', xtype='double', history=res_hst )

  call HistoryAddVariable( varname=trim(adjustl(respsii)), dims=(/'x','y'/), &
  &                        longname='stream line function',  &
  &                        units='m2 s-1', xtype='double', history=res_hst )

  call HistoryAddVariable( varname=trim(adjustl(reszetar)), dims=(/'x','y'/), &
  &                        longname='vorticity', units='s-1',  &
  &                        xtype='double', history=res_hst )

  call HistoryAddVariable( varname=trim(adjustl(reszetai)), dims=(/'x','y'/), &
  &                        longname='vorticity', units='s-1',  &
  &                        xtype='double', history=res_hst )

  call HistoryAddVariable( varname='xd', dims=(/'x'/), &
  &                        longname='X-coord double',  &
  &                        units='m', xtype='double', history=res_hst )

  call HistoryAddVariable( varname='yd', dims=(/'y'/), &
  &                        longname='Y-coord double',  &
  &                        units='m', xtype='double', history=res_hst )

  if(present(zopt))then
     call HistoryAddVariable( varname=trim(adjustl(reszoptr)),  &
  &                           dims=(/'x','y'/), longname='vorticity',  &
  &                           units='s-1', xtype='double', history=res_hst )

     call HistoryAddVariable( varname=trim(adjustl(reszopti)),  &
  &                           dims=(/'x','y'/), longname='vorticity',  &
  &                           units='s-1', xtype='double', history=res_hst )
  end if

  !-- Trivial (not reuse)
  call HistoryPut( 'x', xd(1:ix), history=res_hst )
  call HistoryPut( 'y', yd(1:jy), history=res_hst )

  call HistoryAddAttr( trim(adjustl(respsir)), trim(adjustl(rest)),  &
  &                    rtn, history=res_hst )
  call HistoryAddAttr( trim(adjustl(respsir)), trim(adjustl(restn)),  &
  &                    itn, history=res_hst )

  do jj=1,jy
     do ii=1,ix
        tmpd(ii,jj)=dble(psik(ii,jj))
     end do
  end do

  call HistoryPut( trim(adjustl(respsir)), tmpd, history=res_hst )

  do jj=1,jy
     do ii=1,ix
        tmpd(ii,jj)=dimag(psik(ii,jj))
     end do
  end do

  call HistoryPut( trim(adjustl(respsii)), tmpd, history=res_hst )

  do jj=1,jy
     do ii=1,ix
        tmpd(ii,jj)=dble(zk(ii,jj))
     end do
  end do

  call HistoryPut( trim(adjustl(reszetar)), tmpd, history=res_hst )

  do jj=1,jy
     do ii=1,ix
        tmpd(ii,jj)=dimag(zk(ii,jj))
     end do
  end do

  call HistoryPut( trim(adjustl(reszetai)), tmpd, history=res_hst )

  !-- Trivial (not reuse)
  call HistoryPut( 'xd', x(1:ix), history=res_hst )
  call HistoryPut( 'yd', y(1:jy), history=res_hst )

  if(present(zopt))then

     do jj=1,jy
        do ii=1,ix
           tmpd(ii,jj)=dble(zopt(ii,jj))
        end do
     end do

     call HistoryPut( trim(adjustl(reszoptr)), tmpd, history=res_hst )

     do jj=1,jy
        do ii=1,ix
           tmpd(ii,jj)=dimag(zopt(ii,jj))
        end do
     end do

     call HistoryPut( trim(adjustl(reszopti)), tmpd, history=res_hst )

  end if

  call HistoryClose( history=res_hst )

end subroutine make_restart_isp


subroutine fft_all_clear_isp()
  use fft_saveval_define
  use fft_val_define_isp
  implicit none

  ITJR=0
  ITIR=0
  ITJJ=0
  ITIJ=0
  TJR=0.0d0
  TIR=0.0d0
  TJJ=0.0d0
  TIJ=0.0d0

  zi=0.0d0

  psid=0.0
  ud=0.0
  vd=0.0
  zd=0.0
  tmpr=0.0d0
  tmpi=0.0d0
  
  ur=0.0d0
  vr=0.0d0
  zor=0.0d0
  psior=0.0d0
  psiko=0.0d0
  zko=0.0d0
  psikn=0.0d0
  zopt1=0.0d0
  uk=0.0d0
  vk=0.0d0

  xi=0.0d0
  yi=0.0d0
  xd=0.0
  yd=0.0
  x=0.0d0
  y=0.0d0
  xj=0.0d0
  yj=0.0d0
!ORG  t=0.0d0

  zor_isp=0.0d0
  psior_isp=0.0d0
  ur_isp=0.0d0
  vr_isp=0.0d0
  zko_isp=0.0d0
  psiko_isp=0.0d0
  uk_isp=0.0d0
  vk_isp=0.0d0
  tmp_work=0.0d0

end subroutine fft_all_clear_isp

end module fftsub_mod_isp
